<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
class Yeekit_Dynamic_Discounts_Buy_X_Get_X
{
    private static $calculating = false;

    public static function handle($cart, $rule, $apply_mode = 'biggest_discount')
    {
        if (self::$calculating) return;

        $adjustments = $rule['buyx_getx_adjustments'] ?? [];
        $ranges = $adjustments['ranges'] ?? [];
        if (empty($ranges)) {
            return;
        }

        // Sort ranges by min_qty desc to match highest tier first
        // Sort ranges by min_qty desc to match highest tier first
        usort($ranges, function ($a, $b) {
            return ($b['min_qty'] ?? 0) <=> ($a['min_qty'] ?? 0);
        });

        // 1. Scan Cart
        [$paid_items, $free_items] = self::scan_cart($cart, $rule);

        // 2. Calculate & Sync
        self::$calculating = true;

        foreach ($paid_items as $product_hash => $qty) {
            $matched_range = self::find_matching_range($qty, $ranges);
            $target_free_qty = 0;
            $meta_data = [];

            if ($matched_range) {



                $meta_data = [
                    'rule_id'        => $rule['id'] ?? 0,
                    'rule_type'      => 'yeekit_buy_x_get_x',
                    'discount_type'  => $matched_range['free_type'] ?? 'free_product',
                    'discount_value' => floatval($matched_range['amount'] ?? 0),
                    'type'           => $matched_range['free_type'] ?? 'free_product',
                    'amount'         => floatval($matched_range['amount'] ?? 0),
                    'label'          => !empty($matched_range['label']) ? $matched_range['label'] : '',
                    // We need original price for Price Display, but here we only have hash/qty.
                    // We'll calculate it inside sync_free_item or rely on cart item data if available.
                    'original_price' => 0, // Placeholder, see sync_free_item
                ];

                // Calc target free qty
                $target_free_qty = self::calculate_free_qty($qty, $matched_range);
            }

            self::sync_free_item($cart, $product_hash, $target_free_qty, $free_items, $meta_data);
        }

        // Cleanup orphans: Remove free items if their matching paid item is gone
        foreach ($free_items as $hash => $item) {
            if (!isset($paid_items[$hash])) {
                $cart->remove_cart_item($item['key']);
            }
        }

        self::$calculating = false;

        // 3. Pricing
        self::apply_pricing($cart);
    }

    private static function scan_cart($cart, $rule)
    {
        $paid_items = [];
        $free_items = [];

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $product_id = $cart_item['product_id'];
            $variation_id = $cart_item['variation_id'];
            $product_hash = $variation_id ? $variation_id : $product_id;

            // Check if this is a "true" free item (locked qty) vs a simple discount (qty change allowed)
            $is_simple_discount = !empty($cart_item['_yeekit_free_item']['allow_qty_change']);

            if (!empty($cart_item['_yeekit_free_item']) && !$is_simple_discount) {
                // Check Rule ID match to avoid stealing/deleting other rules' items
                if (isset($cart_item['_yeekit_free_item']['rule_id']) && $cart_item['_yeekit_free_item']['rule_id'] != $rule['id']) {
                    continue;
                }

                // This is a BOGO free item, track it
                $free_items[$product_hash] = [
                    'key' => $cart_item_key,
                    'qty' => $cart_item['quantity'],
                    'product_id' => $product_id,
                    'variation_id' => $variation_id
                ];
                continue;
            }

            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                if (!isset($paid_items[$product_hash])) {
                    $paid_items[$product_hash] = 0;
                }
                $paid_items[$product_hash] += $cart_item['quantity'];
            }
        }
        return [$paid_items, $free_items];
    }

    private static function find_matching_range($qty, $ranges)
    {
        foreach ($ranges as $range) {
            $min_qty = intval($range['min_qty']);
            $max_qty = ($range['max_qty'] !== '') ? intval($range['max_qty']) : null;

            if ($qty >= $min_qty) {
                if ($max_qty !== null && $qty > $max_qty) {
                    continue;
                }
                return $range;
            }
        }
        return null;
    }

    private static function calculate_free_qty($qty, $range)
    {
        $min_qty = intval($range['min_qty']);
        $free_qty = intval($range['free_qty']);
        $is_recursive = !empty($range['is_recursive']);

        if ($is_recursive && $min_qty > 0) {
            $sets = floor($qty / $min_qty);
            return $sets * $free_qty;
        }
        return $free_qty;
    }

    private static function sync_free_item($cart, $product_hash, $target_qty, $free_items, $meta_data)
    {
        $existing = $free_items[$product_hash] ?? null;

        if ($existing) {
            if ($existing['qty'] != $target_qty) {
                if ($target_qty > 0) {
                    $cart->set_quantity($existing['key'], $target_qty);
                    // Update meta data for existing item to ensure Label is shown
                    $cart->cart_contents[$existing['key']]['_yeekit_free_item'] = $meta_data;
                } else {
                    $cart->remove_cart_item($existing['key']);
                }
            } else {
                // Even if qty hasn't changed, update meta data (e.g. if label changed)
                if ($target_qty > 0) {
                    $cart->cart_contents[$existing['key']]['_yeekit_free_item'] = $meta_data;
                }
            }
        } elseif ($target_qty > 0) {
            // Add new free item
            $v_id = 0;
            $p_id = $product_hash;

            // Check if hash is variation
            $type = get_post_type($product_hash);
            if ($type === 'product_variation') {
                $v_id = $product_hash;
                $p_id = wp_get_post_parent_id($v_id);
            }

            $key = $cart->add_to_cart($p_id, $target_qty, $v_id, [], [
                '_yeekit_free_item' => $meta_data
            ]);

            // Fix: Store Original Price logic
            if ($key && isset($cart->cart_contents[$key])) {
                $product = $cart->cart_contents[$key]['data'];
                $original_price = floatval($product->get_price());

                // Update meta with real price
                $cart->cart_contents[$key]['_yeekit_free_item']['original_price'] = $original_price;
            }
        }
    }

    private static function cleanup_orphans($cart, $free_items, $paid_items)
    {
        foreach ($free_items as $hash => $item) {
            if (!isset($paid_items[$hash])) {
                $cart->remove_cart_item($item['key']);
            }
        }
    }

    private static function apply_pricing($cart)
    {
        foreach ($cart->get_cart() as $cart_item) {
            if (!empty($cart_item['_yeekit_free_item'])) {
                $meta = $cart_item['_yeekit_free_item'];

                // Use standardized keys
                $type = $meta['discount_type'] ?? $meta['type'] ?? 'free_product';
                $amount = floatval($meta['discount_value'] ?? $meta['amount'] ?? 0);

                $product = $cart_item['data'];
                // Ensure we get base price before modification if possible, but get_price is standard
                $price = floatval($product->get_price());

                // DEBUG apply_pricing
                if ($type === 'free_product') {
                    $cart_item['data']->set_price(0);
                } elseif ($type === 'percentage') {
                    $new_price = max(0, $price * (1 - ($amount / 100)));
                    $cart_item['data']->set_price($new_price);
                } elseif ($type === 'flat' || $type === 'fixed') {
                    $new_price = max(0, $price - $amount);
                    $cart_item['data']->set_price($new_price);
                }
            }
        }
    }
}
