<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
class Yeekit_Dynamic_Discounts_Bulk_Table
{
    public function __construct()
    {
        add_action('wp', [$this, 'init']);
    }
    public function init()
    {
        // 1. Check global settings
        $show_table = get_option('yk_dd_show_bulk_discount_table', 'yes');
        if ($show_table !== 'yes') {
            return;
        }
        // 2. Get display position
        $position = get_option('yk_dd_position_to_show_bulk_table', 'woocommerce_before_add_to_cart_form');
        if ($position) {
            add_action($position, [$this, 'render_table']);
        }
    }
    public function render_table()
    {
        if (! is_product()) {
            return;
        }
        global $product;
        if (! $product instanceof WC_Product) {
            return;
        }
        // 3. Find matching Bulk Discount rule for this product
        $matched_rule = null;
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();
        foreach ($rules as $rule) {
            if (empty($rule['type']) || $rule['type'] !== 'yeekit_buy_x_get_x') {
                continue;
            }
            // Create fake cart item to validate
            $cart_item = [
                'product_id' => $product->get_id(),
                'variation_id' => 0,
                'data' => $product,
                'quantity' => 1
            ];
            if (Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                $matched_rule = $rule;
                break; // Get first matching rule
            }
        }
        if (! $matched_rule) {
            return;
        }
        $tiers = [];
        $is_bogo = ($matched_rule['type'] === 'yeekit_buy_x_get_x');
        if ($is_bogo) {
            $ranges = $matched_rule['buyx_getx_adjustments']['ranges'] ?? [];;
            // Handle Recursive: Generate purely synthetic rows for display if recursive
            // Assuming first valid range dictates the rule for simplicity?
            // Or merge all? BOGO usually processes sequential ranges. 
            // If recursive is set on a range, we expand it.
            foreach ($ranges as $r) {
                if (!empty($r['is_recursive'])) {
                    // Generate 5 steps
                    $step_min = intval($r['min_qty']);
                    $step_get = intval($r['free_qty']);
                    if ($step_min <= 0) $step_min = 1;
                    for ($i = 1; $i <= 5; $i++) {
                        $qty_buy = $step_min * $i;
                        $qty_get = $step_get * $i;
                        $tiers[] = [
                            'min_qty' => $qty_buy,
                            'max_qty' => '', // Point value
                            'type' => $r['free_type'] ?? 'free_product',
                            'value' => $r['amount'] ?? 0,
                            'get_qty' => $qty_get,
                            'label' => __('Tier', 'yeediscounts') . ' ' . $i,
                            'is_generated' => true
                        ];
                    }
                    // Recursive implies we stop processing other ranges usually?
                    // Let's break to avoid table flooding if multiple recursive ranges exist (rare)
                    break;
                } else {
                    // Normal Range
                    $tiers[] = [
                        'min_qty' => $r['min_qty'],
                        'max_qty' => $r['max_qty'],
                        'type' => $r['free_type'] ?? 'free_product',
                        'value' => $r['amount'] ?? 0,
                        'get_qty' => $r['free_qty'] ?? 0,
                        'label' => ''
                    ];
                }
            }
        } else {
            $tiers = $matched_rule['bulk_tiers'] ?? [];
        }
        if (empty($tiers)) {
            return;
        }
        // Sắp xếp tăng dần theo số lượng
        usort($tiers, function ($a, $b) {
            return ($a['min_qty'] ?? 0) <=> ($b['min_qty'] ?? 0);
        });
        $product_price = floatval($product->get_price());
?>
        <div class="yeekit-bulk-table-wrapper">
            <h3><?php echo esc_html($is_bogo) ? esc_html__('Buy X Get X', 'yeediscounts') : esc_html__('Bulk Discount', 'yeediscounts'); ?></h3>
            <table id="yeekit-bulk-table" class="yeekit-bulk-table-msg yeekit-table">
                <thead class="yeekit-bulk-table-thead">
                    <tr class="yeekit-bulk-table-tr yeekit-bulk-table-thead">
                        <th id="yeekit-bulk-table-title" class="yeekit-bulk-table-td">
                            <span><?php esc_html_e('Title', 'yeediscounts'); ?></span>
                        </th>
                        <th id="yeekit-bulk-table-range" class="yeekit-bulk-table-td">
                            <span><?php echo esc_html($is_bogo) ? esc_html__('Buy Quantity', 'yeediscounts') : esc_html__('Range', 'yeediscounts'); ?></span>
                        </th>
                        <th id="yeekit-bulk-table-discount" class="yeekit-bulk-table-td">
                            <span><?php echo esc_html($is_bogo) ? esc_html__('Get Benefit', 'yeediscounts') : esc_html__('Discount', 'yeediscounts'); ?></span>
                        </th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($tiers as $index => $tier) :
                        $min = $tier['min_qty'] ?? 1;
                        $max = $tier['max_qty'];
                        // Range display
                        if ($is_bogo && !empty($tier['is_generated'])) {
                            $range = $min; // Point value for recursive steps
                        } else {
                            $range = $min . ($max ? ' - ' . $max : '+');
                        }
                        $type = $tier['type'] ?? 'percentage';
                        $value = floatval($tier['value'] ?? 0);
                        $label = $tier['label'] ?? '';
                        $get_qty = $tier['get_qty'] ?? 0;
                        // Tự động tạo nhãn nếu k có
                        if (empty($label)) {
                            $label = __('Tier', 'yeediscounts') . ' ' . ($index + 1);
                        }
                        // Display Text Calculation
                        $discount_display_text = '';
                        $discounted_price_text = '';
                        if ($is_bogo) {
                            // BOGO Info
                            if ($type === 'free_product') {
                                /* translators: %s: Quantity */
                                $discount_display_text = sprintf(__('Get %s Free', 'yeediscounts'), $get_qty);
                            } elseif ($type === 'percentage') {
                                /* translators: 1: Quantity, 2: Percentage value */
                                $discount_display_text = sprintf(__('Get %1$s at %2$s%% Off', 'yeediscounts'), $get_qty, $value);
                            } elseif ($type === 'fixed_price') {
                                /* translators: 1: Quantity, 2: Price amount */
                                $discount_display_text = sprintf(__('Get %1$s at %2$s', 'yeediscounts'), $get_qty, wc_price($value));
                            } else {
                                /* translators: %s: Quantity */
                                $discount_display_text = sprintf(__('Get %s (Discount)', 'yeediscounts'), $get_qty);
                            }
                        } else {
                            // Bulk Discount Info
                            $discounted_price = $product_price;
                            if ($type === 'percentage') {
                                $discount_display_text = $value . '%';
                                $discounted_price = $product_price * (1 - ($value / 100));
                            } elseif ($type === 'fixed') {
                                $discount_display_text = wc_price($value);
                                $discounted_price = max(0, $product_price - $value);
                            } elseif ($type === 'fixed_price') {
                                $discount_display_text = wc_price($value) . ' ' . esc_html__('flat', 'yeediscounts');
                                $discounted_price = $value;
                            }
                            $discounted_price_text = wc_price($discounted_price);
                        }
                    ?>
                        <tr class="yeekit-bulk-table-tr yeekit-bulk-table-row">
                            <!-- Col 1: Title -->
                            <td class="yeekit-bulk-table-td yeekit-bulk-title col_index_1" data-colindex="1">
                                <?php echo esc_html($label); ?>
                            </td>
                            <!-- Col 2: Range -->
                            <td class="yeekit-bulk-table-td yeekit-bulk-range col_index_2" data-colindex="2">
                                <?php echo esc_html($range); ?>
                            </td>
                            <!-- Col 3: Discount -->
                            <td class="yeekit-bulk-table-td yeekit-bulk-table-discount col_index_3" data-colindex="3">
                                <span class="yeekit-table-discounted-value"><?php echo wp_kses_post($discount_display_text); ?></span>
                                <?php if (!$is_bogo && $discounted_price_text): ?>
                                    <span class="yeekit-table-discounted-price" style="display: none"><?php echo wp_kses_post($discounted_price_text); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
<?php
    }
}
new Yeekit_Dynamic_Discounts_Bulk_Table();
