<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly

class Yeekit_Dynamic_Discounts_Bulk_Discounts
{
    private static $calculating = false;
    private static $original_prices = [];

    /**
     * Main entry point for handling Bulk Discount
     */
    public static function handle($cart, $rule)
    {
        // Prevent recursive loop if needed (though set_price usually doesn't trigger calculate_totals immediately)
        if (self::$calculating) return;
        self::$calculating = true;

        $tiers = $rule['bulk_tiers'] ?? [];
        if (empty($tiers)) {
            self::$calculating = false;
            return;
        }

        // Sort tiers by descending min quantity to prioritize highest tier
        usort($tiers, function ($a, $b) {
            return ($b['min_qty'] ?? 0) <=> ($a['min_qty'] ?? 0);
        });

        // Loop through cart to find eligible products
        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            // Check product eligibility
            if (! Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                continue;
            }

            // Get quantity of this product
            $qty = $cart_item['quantity'];
            // Find suitable tier
            $matched_tier = null;
            foreach ($tiers as $tier) {
                $min = intval($tier['min_qty'] ?? 1);
                $max = ($tier['max_qty'] !== '') ? intval($tier['max_qty']) : PHP_INT_MAX;

                if ($qty >= $min && $qty <= $max) {
                    $matched_tier = $tier;
                    break; // Found highest matching tier (since sorted)
                }
            }

            if ($matched_tier) {
                $discount_type = $matched_tier['type'] ?? 'percentage';
                $discount_value = floatval($matched_tier['value'] ?? 0);

                // Lấy giá gốc
                $product = $cart_item['data'];

                // Fix: Lưu giá gốc vào biến static để tránh lỗi Deprecated và Double Discount
                if (! isset(self::$original_prices[$cart_item_key])) {
                    self::$original_prices[$cart_item_key] = floatval($product->get_price());
                }
                $original_price = self::$original_prices[$cart_item_key];

                // Tính giá mới
                $new_price = $original_price;

                if ($discount_type === 'percentage') {
                    if ($discount_value > 100) $discount_value = 100;
                    if ($discount_value < 0) $discount_value = 0;
                    $new_price = $original_price * (1 - ($discount_value / 100));
                } elseif ($discount_type === 'fixed') {
                    $new_price = max(0, $original_price - $discount_value);
                } elseif ($discount_type === 'fixed_price') {
                    // Nếu kiểu là 'fixed_price', tức là giá cố định mỗi sản phẩm
                    $new_price = max(0, $discount_value);
                }

                // Calculate actual discount amount
                $discount_amount = $original_price - $new_price;

                // Áp dụng giá mới
                $cart_item['data']->set_price($new_price);

                // Store metadata for order tracking and reports
                $cart->cart_contents[$cart_item_key]['_yeekit_free_item'] = [
                    'rule_id' => $rule['id'] ?? 0,
                    'rule_type' => 'yeekit_bulk_discount', // Standardized Key
                    'discount_type' => $discount_type, // Standardized Key
                    'discount_value' => $discount_value, // Standardized Key
                    'type' => $discount_type,
                    'value' => $discount_value, // Discount rate/value
                    'original_price' => $original_price,
                    'discount_amount' => $discount_amount, // Actual $ amount discounted
                    'final_price' => $new_price,
                    'allow_qty_change' => true
                ];
            }
        }

        self::$calculating = false;
    }
}
