<?php
if (! defined('ABSPATH')) exit;
class Yeekit_Dynamic_Discounts_Settings_Tab
{
    const TAB_ID = 'yeekit_dynamic_discounts';
    public function __construct()
    {
        add_filter('woocommerce_settings_tabs_array', [$this, 'add_tab'], 50);
        add_action('woocommerce_settings_tabs_' . self::TAB_ID, [$this, 'render_tab']);
        add_action('woocommerce_update_options_' . self::TAB_ID, [$this, 'save_settings']);
        add_filter('admin_body_class', [$this, 'add_body_class']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('wp_ajax_yeekit_dd_search_products', [$this, 'ajax_search_products']);
        add_action('wp_ajax_yeekit_dd_search_attributes', [$this, 'ajax_search_attributes']);
        add_action('wp_ajax_yeekit_dd_search_terms', [$this, 'ajax_search_terms']);
        add_action('wp_ajax_yeekit_dd_search_terms', [$this, 'ajax_search_terms']);
        add_action('wp_ajax_yeekit_dd_search_coupons', [$this, 'ajax_search_coupons']);
        add_action('wp_ajax_yeekit_dd_search_coupons', [$this, 'ajax_search_coupons']);
        // Form Blocks - Single Hook
        add_action('yeekit_dd_render_rule_form_blocks', [$this, 'render_form_blocks'], 10, 2);
        // Load Import/Export Logic Early (needs to hook into admin_init)
        require_once __DIR__ . '/import_export.php';
        // Load Order Tracker (needs to hook into checkout)
        require_once __DIR__ . '/order_tracker.php';
    }
    public function add_tab($tabs)
    {
        $tabs[self::TAB_ID] = __('Dynamic Discounts', 'yeediscounts');
        return $tabs;
    }
    public function enqueue_assets($hook)
    {
        if ($hook !== 'woocommerce_page_wc-settings') {
            return;
        }
        $tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin tab check, no state change.
        if ($tab !== self::TAB_ID) {
            return;
        }
        wp_enqueue_style(
            'yeekit_dd_admin',
            YEEKIT_DYNAMIC_DISCOUNTS_WOOCOMMERCE_URL . 'assets/admin-rules.css',
            [],
            YEEKIT_DYNAMIC_DISCOUNTS_VERSION
        );
        wp_enqueue_style('woocommerce_admin_styles');
        wp_enqueue_script('woocommerce_admin');
        wp_enqueue_script(
            'yeekit_dd_admin',
            YEEKIT_DYNAMIC_DISCOUNTS_WOOCOMMERCE_URL . 'assets/admin-rules.js',
            [
                'jquery',
                'wc-enhanced-select',
                'selectWoo',
            ],
            YEEKIT_DYNAMIC_DISCOUNTS_VERSION,
            true
        );
        wp_localize_script(
            'yeekit_dd_admin',
            'yeekit_dd_admin',
            [
                'i18n_all_products'       => __('All products will be affected', 'yeediscounts'),
                'i18n_select_products'    => __('Search products…', 'yeediscounts'),
                'i18n_attribute_placeholder' => __('Example: pa_color:red, pa_size:large', 'yeediscounts'),
                'i18n_sku_placeholder'    => __('Enter SKUs, separated by commas (e.g. 1001, 1002)', 'yeediscounts'),
                'i18n_on_sale'            => __('Only on-sale products will be affected', 'yeediscounts'),
                'search_products_nonce'   => wp_create_nonce('search-products'),
                'product_categories'      => wp_dropdown_categories([
                    'taxonomy'   => 'product_cat',
                    'hide_empty' => false,
                    'echo'       => false,
                ]),
                'product_tags'            => wp_dropdown_categories([
                    'taxonomy'   => 'product_tag',
                    'hide_empty' => false,
                    'echo'       => false,
                ]),
                'ajax_url'   => admin_url('admin-ajax.php'),
                'ajax_nonce' => wp_create_nonce('yeekit_dd_admin'),
                'default_condition' => 'cart_subtotal',
                'condition_groups' => $this->get_condition_groups(),
            ]
        );
        wp_localize_script('yeekit_dd_admin', 'yeekit_dd_rule_params', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('yeekit_dd_rule_nonce'),
            'condition_groups' => $this->get_condition_groups(),
            'i18n' => [
                'recursive_confirm' => __('If you make this recursive other row(s) will be removed!', 'yeediscounts'),
                'remove_condition' => __('Remove condition', 'yeediscounts'),
                'remove_filter' => __('Remove filter', 'yeediscounts'),
                'search_coupons' => __('Search coupons...', 'yeediscounts'),
                'min_qty' => __('Min Qty', 'yeediscounts'),
                'max_qty' => __('Max Qty', 'yeediscounts'),
                'type' => __('Discount Type', 'yeediscounts'),
                'value' => __('Value', 'yeediscounts'),
                'label' => __('Label', 'yeediscounts'),
                'remove_range' => __('Remove range', 'yeediscounts'),
                'fixed_price_set' => __('Fixed price for set / bundle', 'yeediscounts'),
                'percentage_item' => __('Percentage discount per item', 'yeediscounts'),
                'flat_item' => __('Fixed discount per item', 'yeediscounts'),
                'quantity' => __('Quantity', 'yeediscounts'),
                'minimum_quantity' => __('Minimum Quantity', 'yeediscounts'),
                'recursive' => __('Recursive', 'yeediscounts'),
                'change_type_confirm' => __('Changing type will reset selected items in rows. Continue?', 'yeediscounts'),
                'get_product' => __('Get Product', 'yeediscounts'),
                'get_category' => __('Get Category', 'yeediscounts'),
                'get_any_product' => __('Get Any Product', 'yeediscounts'),
                'customer_gets_any' => __('Customer gets any product from store', 'yeediscounts'),
                'buy_product' => __('Buy Product', 'yeediscounts'),
                'get_discount' => __('Get Discount', 'yeediscounts'),
                'percentage' => __('Percentage', 'yeediscounts'),
                'fixed' => __('Fixed Discount', 'yeediscounts'),
                'fixed_price' => __('Fixed price for item', 'yeediscounts'),
                'in_list' => __('In list', 'yeediscounts'),
                'not_in_list' => __('Not in list', 'yeediscounts'),
                'products' => __('Products', 'yeediscounts'),
                'category' => __('Category', 'yeediscounts'),
                'tags' => __('Tags', 'yeediscounts'),
                'attributes' => __('Attributes', 'yeediscounts'),
                'skus' => __('SKUs', 'yeediscounts'),
                'on_sale_products' => __('On sale products', 'yeediscounts'),
                'desc_products' => __('Choose products that get the discount using "In list". To exclude products, use "Not in list".', 'yeediscounts'),
                'desc_category' => __('Choose categories to include using "In list". To exclude categories, use "Not in list".', 'yeediscounts'),
                'desc_tags' => __('Choose tags to include using "In list". To exclude tags, use "Not in list".', 'yeediscounts'),
                'desc_attributes' => __('Choose attributes to include using "In list". To exclude attributes, use "Not in list".', 'yeediscounts'),
                'desc_skus' => __('Enter SKUs, separated by commas (e.g. 94651, 94652, 9463). Use "In list" to include, "Not in list" to exclude.', 'yeediscounts'),
                'desc_on_sale' => __('Matches items that are On Sale. Use "In list" to include them, "Not in list" to exclude them.', 'yeediscounts'),
                'monday' => __('Monday', 'yeediscounts'),
                'tuesday' => __('Tuesday', 'yeediscounts'),
                'wednesday' => __('Wednesday', 'yeediscounts'),
                'thursday' => __('Thursday', 'yeediscounts'),
                'friday' => __('Friday', 'yeediscounts'),
                'saturday' => __('Saturday', 'yeediscounts'),
                'sunday' => __('Sunday', 'yeediscounts'),
                'buy_quantity' => __('Buy Quantity', 'yeediscounts'),
                'get_quantity' => __('Get Quantity', 'yeediscounts'),
                'get_qty' => __('Get Qty', 'yeediscounts'),
                'free_qty' => __('Free Qty', 'yeediscounts'),
                'free' => __('Free', 'yeediscounts'),
            ],
        ]);
    }
    /**
     * Add body class for section-specific styling.
     */
    public function add_body_class($classes)
    {
        $screen = get_current_screen();
        if ($screen && $screen->id === 'woocommerce_page_wc-settings') {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for CSS class.
            $tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : '';
            if ($tab === self::TAB_ID) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for CSS class.
                $section = isset($_GET['section']) ? sanitize_text_field(wp_unslash($_GET['section'])) : 'rules';
                $classes .= ' yeekit-dd-tab-' . esc_attr($section);
            }
        }
        return $classes;
    }
    public function ajax_search_products()
    {
        check_ajax_referer('yeekit_dd_admin', 'nonce');
        if (! current_user_can('manage_woocommerce')) {
            wp_send_json([]);
        }
        $term = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';

        // wc_get_products() argument 'search' is not standard. Use get_posts() instead.
        $args = [
            'post_type'      => ['product', 'product_variation'],
            'post_status'    => ['publish', 'private'],
            'posts_per_page' => 20,
            's'              => $term,
            'fields'         => 'ids',
            'orderby'        => 'title',
            'order'          => 'ASC',
        ];

        $product_ids = get_posts($args);
        $results = [];

        foreach ($product_ids as $product_id) {
            $product = wc_get_product($product_id);
            if ($product) {
                $results[] = [
                    'id'   => $product->get_id(),
                    'text' => '#' . $product->get_id() . ' - ' . $product->get_name(),
                ];
            }
        }
        wp_send_json($results);
    }
    public function ajax_search_attributes()
    {
        check_ajax_referer('yeekit_dd_admin', 'nonce');
        if (! current_user_can('manage_woocommerce')) {
            wp_send_json([]);
        }
        $term = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $results = [];
        $attribute_taxonomies = wc_get_attribute_taxonomies();
        if (empty($attribute_taxonomies)) {
            wp_send_json([]);
        }
        foreach ($attribute_taxonomies as $tax) {
            $taxonomy = wc_attribute_taxonomy_name($tax->attribute_name);
            $terms = get_terms([
                'taxonomy'   => $taxonomy,
                'hide_empty' => false,
                'search'     => $term,
                'number'     => 20,
            ]);
            if (is_wp_error($terms)) {
                continue;
            }
            foreach ($terms as $term_obj) {
                $results[] = [
                    'id'   => $taxonomy . ':' . $term_obj->slug,
                    'text' => $tax->attribute_label . ' — ' . $term_obj->name,
                ];
            }
        }
        wp_send_json($results);
    }
    public function ajax_search_terms()
    {
        check_ajax_referer('yeekit_dd_admin', 'nonce');
        if (! current_user_can('manage_woocommerce')) {
            wp_send_json([]);
        }
        $taxonomy = isset($_GET['taxonomy']) ? sanitize_key($_GET['taxonomy']) : '';
        $term     = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        if (! taxonomy_exists($taxonomy)) {
            wp_send_json([]);
        }
        $terms = get_terms([
            'taxonomy'   => $taxonomy,
            'hide_empty' => false,
            'search'     => $term,
            'number'     => 20,
        ]);
        if (is_wp_error($terms)) {
            wp_send_json([]);
        }
        $results = [];
        foreach ($terms as $t) {
            $results[] = [
                'id'   => $t->term_id,
                'text' => $t->name,
            ];
        }
        wp_send_json($results);
    }
    public function ajax_search_coupons()
    {
        check_ajax_referer('yeekit_dd_admin', 'nonce');
        if (! current_user_can('manage_woocommerce')) {
            wp_send_json([]);
        }
        $term = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
        $args = [
            'posts_per_page' => 20,
            'post_type'      => 'shop_coupon',
            'post_status'    => 'publish',
            's'              => $term,
            'fields'         => 'ids',
        ];
        $coupons = get_posts($args);
        $results = [];
        foreach ($coupons as $id) {
            $code = get_the_title($id);
            $results[] = [
                'id'   => $code,
                'text' => $code,
            ];
        }
        wp_send_json($results);
    }
    /**
     * Get a rule by its ID.
     */
    private function get_rule_by_id($rule_id)
    {
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_rules();
        return isset($rules[$rule_id]) ? $rules[$rule_id] : null;
    }
    /**
     * Render the form blocks in order.
     */
    public function render_form_blocks($rule, $settings)
    {
        require __DIR__ . '/blocks/general.php';
        require __DIR__ . '/blocks/filter.php';
        require __DIR__ . '/blocks/discount.php';
        require __DIR__ . '/blocks/bulk.php';
        require __DIR__ . '/blocks/set.php';
        require __DIR__ . '/blocks/bogo.php';
        require __DIR__ . '/blocks/bxgy.php';
        require __DIR__ . '/blocks/rules.php';
        require __DIR__ . '/blocks/discount-bar.php';
    }
    /**
     * Render the Add/Edit Rule form.
     */
    private function render_add_rule_form($rule = null, $rule_id = null)
    {
        $is_edit = is_array($rule);
?>
        <div class="wrap yeekit-dd-rule-edit">
            <h1>
                <?php echo $is_edit ? esc_html__('Edit Discount Rule', 'yeediscounts') : esc_html__('Add Discount Rule', 'yeediscounts'); ?>
            </h1>
            <form method="post">
                <?php wp_nonce_field('yeekit_save_rule', 'yeekit_save_rule_nonce'); ?>
                <?php wp_nonce_field('yeekit_save_rule', 'yeekit_rule_nonce'); ?>
                <?php if ($is_edit): ?>
                    <input type="hidden" name="rule_id" value="<?php echo esc_attr($rule_id); ?>">
                <?php endif; ?>
                <?php
                do_action('yeekit_dd_render_rule_form_blocks', $rule, $this);
                ?>
                <p class="submit-box">
                    <button type="submit" class="button button-primary">
                        <?php esc_html_e('Save rule', 'yeediscounts'); ?>
                    </button>
                </p>
            </form>
        </div>
    <?php
    }
    /**
     * Handle saving Add/Edit Rule form.
     */
    public function handle_save_rule()
    {
        if (! isset($_POST['yeekit_rule_nonce']) || ! wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['yeekit_rule_nonce'])), 'yeekit_save_rule')) {
            return;
        }
        if (! current_user_can('manage_woocommerce')) {
            return;
        }
        $rule = [];
        // ===== GENERAL =====
        $rule['enabled'] = ! empty($_POST['rule']['enabled']) ? 1 : 0;
        $rule['ignore_other_rules'] = ! empty($_POST['rule']['ignore_other_rules']) ? 1 : 0;
        $rule['title']   = sanitize_text_field(wp_unslash($_POST['rule']['title']) ?? ''); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
        $rule['description'] = sanitize_textarea_field(wp_unslash($_POST['rule']['description']) ?? ''); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
        $rule['type']    = sanitize_text_field(wp_unslash($_POST['rule']['type']) ?? ''); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
        $rule['settings'] = [
            'apply_as_coupon' => ! empty($_POST['rule']['settings']['apply_as_coupon']) ? 1 : 0,
        ];
        // Discount Bar
        $rule['discount_bar_enabled'] = ! empty($_POST['rule']['discount_bar_enabled']) ? 1 : 0;
        $rule['discount_bar_message'] = sanitize_text_field(wp_unslash($_POST['rule']['discount_bar_message'] ?? '')); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated
        $rule['discount_bar_bg_color'] = isset($_POST['rule']['discount_bar_bg_color'])
            ? sanitize_hex_color(wp_unslash($_POST['rule']['discount_bar_bg_color']))
            : '';
        $rule['discount_bar_text_color'] = isset($_POST['rule']['discount_bar_text_color'])
            ? sanitize_hex_color(wp_unslash($_POST['rule']['discount_bar_text_color']))
            : '';
        // ===== FILTERS =====
        $rule['filters'] = [];
        if (isset($_POST['filters']) && is_array($_POST['filters'])) {
            foreach (wp_unslash($_POST['filters']) as $filter) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $rule['filters'][] = [
                    'type'     => sanitize_text_field($filter['type'] ?? ''),
                    'operator' => sanitize_text_field($filter['operator'] ?? ''),
                    'value'    => is_array($filter['value'] ?? null)
                        ? array_map('sanitize_textarea_field', (array) $filter['value'])
                        : sanitize_textarea_field($filter['value'] ?? ''),
                ];
            }
        }
        // ===== DISCOUNT =====
        $rule['discount'] = [
            'type'  => sanitize_text_field(wp_unslash($_POST['yeekit_discount_type'] ?? '')),
            'value' => floatval(wp_unslash($_POST['yeekit_discount_value'] ?? 0)),
            'title' => sanitize_text_field(wp_unslash($_POST['yeekit_discount_title'] ?? '')),
        ];
        // ===== BULK ADJUSTMENTS =====
        $rule['bulk_adjustments'] = [
            'operator' => sanitize_text_field(wp_unslash($_POST['rule']['bulk_adjustments']['operator'] ?? 'product_cumulative')),
        ];
        // ===== BULK TIERS =====
        $rule['bulk_tiers'] = [];
        if (isset($_POST['rule']['bulk_tiers']) && is_array($_POST['rule']['bulk_tiers'])) {
            foreach (wp_unslash($_POST['rule']['bulk_tiers']) as $tier) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $rule['bulk_tiers'][] = [
                    'min_qty' => absint($tier['min_qty'] ?? 1),
                    'max_qty' => (isset($tier['max_qty']) && $tier['max_qty'] !== '') ? absint($tier['max_qty']) : '',
                    'type'    => sanitize_text_field($tier['type'] ?? 'percentage'),
                    'value'   => floatval($tier['value'] ?? 0),
                    'label'   => sanitize_text_field($tier['label'] ?? ''),
                ];
            }
        }
        // ===== SET ADJUSTMENTS (Bundle) =====
        $rule['set_adjustments'] = [
            'operator' => isset($_POST['rule']['set_adjustments']['operator'])
                ? sanitize_text_field(wp_unslash($_POST['rule']['set_adjustments']['operator']))
                : 'product_cumulative',
            'ranges'   => [],
        ];
        if (isset($_POST['rule']['set_adjustments']['ranges']) && is_array($_POST['rule']['set_adjustments']['ranges'])) {
            foreach (wp_unslash($_POST['rule']['set_adjustments']['ranges']) as $range) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $rule['set_adjustments']['ranges'][] = [
                    'qty'          => absint($range['qty'] ?? 1),
                    'value'        => floatval($range['value'] ?? 0),
                    'type'         => sanitize_text_field($range['type'] ?? 'fixed_set_price'),
                    'label'        => sanitize_text_field($range['label'] ?? ''),
                    'is_recursive' => ! empty($range['is_recursive']) ? 1 : 0,
                ];
            }
        }
        // ===== BUY X GET X ADJUSTMENTS =====
        $rule['buyx_getx_adjustments'] = [
            'ranges' => [],
        ];
        if (isset($_POST['rule']['buyx_getx_adjustments']['ranges']) && is_array($_POST['rule']['buyx_getx_adjustments']['ranges'])) {
            foreach (wp_unslash($_POST['rule']['buyx_getx_adjustments']['ranges']) as $range) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $rule['buyx_getx_adjustments']['ranges'][] = [
                    'min_qty'      => absint($range['min_qty'] ?? 1),
                    'max_qty'      => (isset($range['max_qty']) && $range['max_qty'] !== '') ? absint($range['max_qty']) : '',
                    'free_qty'     => absint($range['free_qty'] ?? 1),
                    'free_type'    => sanitize_text_field($range['free_type'] ?? 'free_product'),
                    'amount'       => floatval($range['amount'] ?? 0),
                    'is_recursive' => ! empty($range['is_recursive']) ? 1 : 0,
                ];
            }
        }
        // ===== BUY X GET Y ADJUSTMENTS =====
        $rule['buyx_gety_adjustments'] = [
            'type'       => sanitize_text_field(wp_unslash($_POST['rule']['buyx_gety_adjustments']['type'] ?? '')),
            'operator'   => sanitize_text_field(wp_unslash($_POST['rule']['buyx_gety_adjustments']['operator'] ?? '')),
            'apply_mode' => sanitize_text_field(wp_unslash($_POST['rule']['buyx_gety_adjustments']['apply_mode'] ?? '')),
            'ranges'     => [],
        ];
        if (isset($_POST['rule']['buyx_gety_adjustments']['ranges']) && is_array($_POST['rule']['buyx_gety_adjustments']['ranges'])) {
            foreach (wp_unslash($_POST['rule']['buyx_gety_adjustments']['ranges']) as $range) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
                $rule['buyx_gety_adjustments']['ranges'][] = [
                    'min_qty'        => absint($range['min_qty'] ?? 1),
                    'max_qty'        => (isset($range['max_qty']) && $range['max_qty'] !== '') ? absint($range['max_qty']) : '',
                    'get_product'    => !empty($range['get_product']) ? array_map('absint', (array) $range['get_product']) : [],
                    'get_qty'        => absint($range['get_qty'] ?? 1),
                    'discount_type'  => sanitize_text_field($range['discount_type'] ?? 'free_product'),
                    'discount_value' => floatval($range['discount_value'] ?? 0),
                    'is_recursive'   => ! empty($range['is_recursive']) ? 1 : 0,
                ];
            }
        }
        // ===== CONDITIONS =====
        if (isset($_POST['rules']['type']) && is_array($_POST['rules']['type'])) {
            $types    = wp_unslash($_POST['rules']['type']); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $operators = wp_unslash($_POST['rules']['operator'] ?? []); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $values    = wp_unslash($_POST['rules']['value'] ?? []); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            foreach ($types as $i => $type) {
                $rule['conditions'][] = [
                    'type'     => sanitize_text_field($type),
                    'operator' => sanitize_text_field($operators[$i] ?? ''),
                    'value'    => is_array($values[$i] ?? null)
                        ? array_map('sanitize_text_field', (array) $values[$i])
                        : sanitize_text_field($values[$i] ?? ''),
                ];
            }
        }

        // ===== CLEANUP BASED ON TYPE =====
        // Only keep relevance keys to avoid mixing settings
        $type = $rule['type'];

        // Define keys for each type
        $keys_map = [
            'yeekit_simple_discount' => ['discount'],
            'yeekit_bulk_discount'   => ['bulk_adjustments', 'bulk_tiers'],
            'yeekit_set_discount'    => ['set_adjustments'],
            'yeekit_buy_x_get_x'     => ['buyx_getx_adjustments'],
            'yeekit_buy_x_get_y'     => ['buyx_gety_adjustments']
        ];

        // List of all exclusive keys to check/unset
        $all_special_keys = ['discount', 'bulk_adjustments', 'bulk_tiers', 'set_adjustments', 'buyx_getx_adjustments', 'buyx_gety_adjustments'];

        // Get allowed keys for current type
        $allowed_keys = $keys_map[$type] ?? [];

        // Special Case: Simple Discount logic might be shared or default?
        // Actually simple discount uses 'discount' array.
        // Others might NOT use 'discount' array (e.g. BOGO stores amount in ranges).
        // Let's check:
        // Bulk: uses bulk_tiers.
        // Set: uses set_adjustments.
        // BOGO: uses ranges.
        // So 'discount' key is effectively only for Simple Discount?
        // Wait, 'discount' key stores "Title" sometimes? No, title covers whole rule.
        // Checking `blocks/discount.php` usually handles simple discount inputs.
        // So yes, strictly cleaning up is safe.

        foreach ($all_special_keys as $key) {
            if (!in_array($key, $allowed_keys)) {
                unset($rule[$key]);
            }
        }
        // ===== SAVE =====
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_rules();
        if (isset($_POST['rule_id']) && $_POST['rule_id'] !== '') {
            $rule_id = sanitize_text_field(wp_unslash($_POST['rule_id']));
            $rules[$rule_id] = $rule;
            $redirect_rule_id = $rule_id;
        } else {
            // GENERATE RANDOM ID (6 chars, alphanumeric)
            do {
                $new_id = wp_generate_password(6, false);
            } while (isset($rules[$new_id]));

            $rules[$new_id] = $rule;
            $redirect_rule_id = $new_id;
        }
        Yeekit_Dynamic_Discounts_Rules_Manager::save_rules($rules);
        // Build redirect args to stay on edit/add page after save
        $args = [
            'tab'     => self::TAB_ID,
            'section' => 'rules',
            'saved'   => 1,
        ];

        // Redirect to edit mode
        $args['action']  = 'edit';
        $args['rule_id'] = $redirect_rule_id;
        wp_safe_redirect(
            add_query_arg($args, admin_url('admin.php?page=wc-settings'))
        );
        exit;
    }
    public function render_tab()
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading GET parameters for display only, no data modification.
        $section = isset($_GET['section']) ? sanitize_text_field(wp_unslash($_GET['section'])) : 'rules';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading GET parameters for display only.
        $action  = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        // Handle save BEFORE rendering anything
        if ($section === 'rules') {
            $this->handle_save_rule();
            $this->handle_delete_rule();
        }
    ?>
        <h2><?php esc_html_e('Dynamic Discounts', 'yeediscounts'); ?></h2>
        <?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display notice based on GET parameter.
        if (isset($_GET['saved']) && ! empty(sanitize_text_field(wp_unslash($_GET['saved'])))) : ?>
            <div class="notice notice-success is-dismissible">
                <p><?php esc_html_e('Rule saved successfully.', 'yeediscounts'); ?></p>
            </div>
        <?php endif; ?>
        <?php // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Display notice based on GET parameter.
        if (isset($_GET['deleted']) && ! empty(sanitize_text_field(wp_unslash($_GET['deleted'])))) : ?>
            <div class="notice notice-success is-dismissible">
                <p><?php esc_html_e('Rule deleted successfully.', 'yeediscounts'); ?></p>
            </div>
        <?php endif; ?>
        <ul class="subsubsub">
            <li>
                <a href="<?php echo esc_url(remove_query_arg(['action', 'rule_id', 'saved', 'deleted'], add_query_arg(['section' => 'rules']))); ?>"
                    class="<?php echo esc_attr($section === 'rules' ? 'current' : ''); ?>">
                    <?php esc_html_e('Rules', 'yeediscounts'); ?>
                </a> |
            </li>
            <li>
                <a href="<?php echo esc_url(add_query_arg(['section' => 'settings'])); ?>"
                    class="<?php echo esc_attr($section === 'settings' ? 'current' : ''); ?>">
                    <?php esc_html_e('Settings', 'yeediscounts'); ?>
                </a> |
            </li>
            <li>
                <a href="<?php echo esc_url(add_query_arg(['section' => 'import_export'])); ?>"
                    class="<?php echo esc_attr($section === 'import_export' ? 'current' : ''); ?>">
                    <?php esc_html_e('Import/Export', 'yeediscounts'); ?>
                </a> |
            </li>
            <li>
                <a href="<?php echo esc_url(add_query_arg(['section' => 'reports'])); ?>"
                    class="<?php echo esc_attr($section === 'reports' ? 'current' : ''); ?>">
                    <?php esc_html_e('Reports', 'yeediscounts'); ?>
                </a>
            </li>
        </ul>
        <br class="clear" />
        <?php
        // SETTINGS
        if ($section === 'settings') {
        ?>
            <div class="yeekit-settings-wrapper">
                <?php
                woocommerce_admin_fields($this->get_settings_fields());
                ?></div>
        <?php
            return;
        }
        // IMPORT/EXPORT
        if ($section === 'import_export') {
            require_once __DIR__ . '/import_export.php';
            settings_errors('yeekit_dd_messages');
        ?>
            <div class="yeekit-settings-wrapper">
                <h2><?php esc_html_e('Import/Export Rules', 'yeediscounts'); ?></h2>
                <!-- Export -->
                <div class="card" style="padding: 20px; margin-bottom: 20px; max-width: 600px;">
                    <h3><?php esc_html_e('Export Rules', 'yeediscounts'); ?></h3>
                    <p><?php esc_html_e('Download all your current discount rules as a JSON file.', 'yeediscounts'); ?></p>
                    <form method="post">
                        <?php wp_nonce_field('yeekit_dd_export', 'yeekit_dd_export_nonce'); ?>
                        <input type="hidden" name="yeekit_action" value="export_rules">
                        <button type="submit" class="button button-primary">
                            <?php esc_html_e('Export Rules', 'yeediscounts'); ?>
                        </button>
                    </form>
                </div>
                <!-- Import -->
                <div class="card" style="padding: 20px; max-width: 600px;">
                    <h3><?php esc_html_e('Import Rules', 'yeediscounts'); ?></h3>
                    <p><?php esc_html_e('Upload a JSON file to import rules. New rules will be added to your existing ones.', 'yeediscounts'); ?></p>
                    <form method="post" enctype="multipart/form-data">
                        <?php wp_nonce_field('yeekit_dd_import', 'yeekit_dd_import_nonce'); ?>
                        <input type="hidden" name="yeekit_action" value="import_rules">
                        <p>
                            <input type="file" name="import_file" accept=".json" required>
                        </p>
                        <p>
                            <button type="submit" class="button button-secondary">
                                <?php esc_html_e('Import Rules', 'yeediscounts'); ?>
                            </button>
                        </p>
                    </form>
                </div>
            </div>
        <?php
            return;
        }
        // REPORTS
        if ($section === 'reports') {
            require_once __DIR__ . '/reports.php';
            $reports = new Yeekit_Dynamic_Discounts_Reports();
            $reports->render();
            return;
        }
        // ADD/EDIT RULE FORM
        if ($section === 'rules' && in_array($action, ['add', 'edit'], true)) {
            $rule = null;
            $rule_id = null;
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading rule_id for display only.
            if ($action === 'edit' && isset($_GET['rule_id'])) {
                $rule_id = sanitize_text_field(wp_unslash($_GET['rule_id'])); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
                $rule = $this->get_rule_by_id($rule_id);
            }
            $this->render_add_rule_form($rule, $rule_id);
            return;
        }
        // ===== DEFAULT: RULES LIST TABLE =====
        $table = new Yeekit_Dynamic_Discounts_Rules_Table();
        $table->prepare_items();
        ?>
        <p>
            <a href="<?php echo esc_url(add_query_arg([
                            'tab'     => self::TAB_ID,
                            'section' => 'rules',
                            'action'  => 'add',
                        ])); ?>" class="button button-primary">
                <?php esc_html_e('Add Rule', 'yeediscounts'); ?>
            </a>
        </p>
        <form method="post">
            <?php $table->display(); ?>
        </form>
<?php
    }
    public function handle_delete_rule()
    {
        if (
            isset($_GET['action']) &&
            sanitize_text_field(wp_unslash($_GET['action'])) === 'delete' &&
            isset($_GET['rule_id'])
        ) {
            $rule_id = sanitize_text_field(wp_unslash($_GET['rule_id']));
            check_admin_referer('yeekit_delete_rule_' . $rule_id);
            if (! current_user_can('manage_woocommerce')) {
                return;
            }
            $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_rules();
            if (isset($rules[$rule_id])) {
                unset($rules[$rule_id]);
                // We do NOT re-index values to preserve IDs for reports/orders
                Yeekit_Dynamic_Discounts_Rules_Manager::save_rules($rules);
                wp_safe_redirect(add_query_arg([
                    'tab' => self::TAB_ID,
                    'section' => 'rules',
                    'deleted' => 1
                ], admin_url('admin.php?page=wc-settings')));
                exit;
            }
        }
    }
    public function save_settings()
    {
        woocommerce_update_options($this->get_settings_fields());
    }
    private function get_settings_fields()
    {
        return [
            // ===== GENERAL SETTINGS =====
            [
                'title' => __('General Settings', 'yeediscounts'),
                'type'  => 'title',
                'id'    => 'yk_dd_general_section',
            ],
            [
                'title'    => __('Choose how the discount rules should apply', 'yeediscounts'),
                'desc'     => __('Select which discount to apply when multiple product rules match.', 'yeediscounts'),
                'id'       => 'yk_dd_apply_product_discount_to',
                'default'  => 'biggest_discount',
                'type'     => 'select',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'biggest_discount' => __('Biggest one from matched rules', 'yeediscounts'),
                    'lowest_discount'  => __('Lowest one from matched rules', 'yeediscounts'),
                    'first'            => __('First matched rules', 'yeediscounts'),
                    'all'              => __('All matched rules', 'yeediscounts'),
                ],
            ],
            [
                'title'    => __('Choose discount rules behaviour with default WooCommerce Coupons', 'yeediscounts'),
                'desc'     => __('Control how dynamic discounts interact with standard WooCommerce coupons.', 'yeediscounts'),
                'id'       => 'yk_dd_disable_coupon_when_rule_applied',
                'default'  => 'run_both',
                'type'     => 'select',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'run_both'       => __('Let both coupons and discount rules run together', 'yeediscounts'),
                    'disable_coupon' => __('Disable the coupons (discount rules will work)', 'yeediscounts'),
                    'disable_rules'  => __('Disable the discount rules (coupons will work)', 'yeediscounts'),
                ],
            ],
            [
                'type' => 'sectionend',
                'id'   => 'yk_dd_general_section',
            ],
            // ===== PRODUCT SETTINGS =====
            [
                'title' => __('Product Settings', 'yeediscounts'),
                'type'  => 'title',
                'id'    => 'yk_dd_product_section',
            ],
            [
                'title'   => __('Show discounted price (strikeout)', 'yeediscounts'),
                'id'      => 'yk_dd_show_discounted_price',
                'default' => 'at_least_has_any_rules',
                'type'    => 'select',
                'class'   => 'wc-enhanced-select',
                'options' => [
                    'when_condition_matches' => __('Show only after a rule condition is matched exactly', 'yeediscounts'),
                    'at_least_has_any_rules' => __('Show on products that are covered under any discount rule in the plugin', 'yeediscounts'),
                    'disabled'               => __('Do not show', 'yeediscounts'),
                ],
                'desc'    => __('Show the calculated discounted price on shop and product pages (strikethrough regular price).', 'yeediscounts'),
            ],
            [
                'title'    => __('Show on pages', 'yeediscounts'),
                'desc'     => __('Select which pages to display the discounted price on.', 'yeediscounts'),
                'id'       => 'yk_dd_show_discounted_price_pages',
                'default'  => ['shop', 'category', 'product'],
                'type'     => 'multiselect',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'shop'     => __('Shop Page', 'yeediscounts'),
                    'category' => __('Category / Archive Pages', 'yeediscounts'),
                    'product'  => __('Single Product Page', 'yeediscounts'),
                ],
            ],
            [
                'title'    => __('On-sale badge', 'yeediscounts'),
                'desc'     => __('Choose when to display the sale badge on products affected by dynamic discounts.', 'yeediscounts'),
                'id'       => 'yk_dd_show_on_sale_badge',
                'default'  => 'at_least_has_any_rules',
                'type'     => 'select',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'when_condition_matches' => __('Show only after a rule condition is matched exactly', 'yeediscounts'),
                    'at_least_has_any_rules' => __('Show on products that are covered under any discount rule in the plugin', 'yeediscounts'),
                    'disabled'               => __('Do not show', 'yeediscounts'),
                ],
            ],
            [
                'title'   => __('Do you want to customize the sale badge?', 'yeediscounts'),
                'id'      => 'yk_dd_customize_sale_badge',
                'default' => 'no',
                'type'    => 'checkbox',
                'desc'    => __('Yes, I would like to customize the sale badge', 'yeediscounts'),
            ],
            [
                'title'   => __('Force override label', 'yeediscounts'),
                'id'      => 'yk_dd_force_override_sale_badge',
                'default' => 'no',
                'type'    => 'checkbox',
                'desc'    => __('Force override the label for sale badge (useful when your theme has override for sale badge).', 'yeediscounts'),
            ],
            [
                'title'   => __('Display percentage', 'yeediscounts'),
                'id'      => 'yk_dd_show_percentage_in_sale_badge',
                'default' => 'no',
                'type'    => 'checkbox',
                'desc'    => __('I would like to display percentage in sale badge (Displays only when rule matches else displays default sale badge content).', 'yeediscounts'),
            ],
            [
                'title'    => __('Enter the sale badge text', 'yeediscounts'),
                'desc'     => __('Custom text to show on the validation badge. Leave empty to use default.', 'yeediscounts'),
                'id'       => 'yk_dd_sale_badge_text',
                'type'     => 'text',
                'default'  => '',
            ],

            [
                'title'    => __('Show bulk discount table', 'yeediscounts'),
                'desc'     => __('Display a table of bulk discount tiers on the product page.', 'yeediscounts'),
                'id'       => 'yk_dd_show_bulk_discount_table',
                'default'  => 'yes',
                'type'     => 'radio',
                'options'  => [
                    'yes' => __('Yes', 'yeediscounts'),
                    'no'  => __('No', 'yeediscounts'),
                ],
            ],
            [
                'title'    => __('Discount table location', 'yeediscounts'),
                'desc'     => __('Select where the bulk discount table should appear on the product page.', 'yeediscounts'),
                'id'       => 'yk_dd_position_to_show_bulk_table',
                'default'  => 'woocommerce_before_add_to_cart_form',
                'type'     => 'select',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'woocommerce_before_add_to_cart_form'      => 'Woocommerce before add to cart form',
                    'woocommerce_product_meta_end'             => 'Woocommerce product meta end',
                    'woocommerce_product_meta_start'           => 'Woocommerce product meta start',
                    'woocommerce_after_add_to_cart_form'       => 'Woocommerce after add to cart form',
                    'woocommerce_after_single_product'         => 'Woocommerce after single product',
                    'woocommerce_before_single_product'        => 'Woocommerce before single product',
                    'woocommerce_after_single_product_summary' => 'Woocommerce after single product summary',
                    'woocommerce_before_single_product_summary' => 'Woocommerce before single product summary',
                ],
            ],
            [
                'title'    => __('Discount bar location', 'yeediscounts'),
                'desc'     => __('Select where the discount bar (notification) should appear.', 'yeediscounts'),
                'id'       => 'yk_dd_position_to_show_discount_bar',
                'default'  => 'woocommerce_before_add_to_cart_form',
                'type'     => 'select',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'woocommerce_before_add_to_cart_form'      => 'Woocommerce before add to cart form',
                    'woocommerce_product_meta_end'             => 'Woocommerce product meta end',
                    'woocommerce_product_meta_start'           => 'Woocommerce product meta start',
                    'woocommerce_after_add_to_cart_form'       => 'Woocommerce after add to cart form',
                    'woocommerce_after_single_product'         => 'Woocommerce after single product',
                    'woocommerce_before_single_product'        => 'Woocommerce before single product',
                    'woocommerce_after_single_product_summary' => 'Woocommerce after single product summary',
                    'woocommerce_before_single_product_summary' => 'Woocommerce before single product summary',
                ],
            ],
            [
                'type' => 'sectionend',
                'id'   => 'yk_dd_product_section',
            ],
            // ===== CART SETTINGS =====
            [
                'title' => __('Cart Settings', 'yeediscounts'),
                'type'  => 'title',
                'id'    => 'yk_dd_cart_section',
            ],
            [
                'title'   => __('Show notifications on the cart page', 'yeediscounts'),
                'id'      => 'yk_dd_show_cart_notifications',
                'default' => 'yes',
                'type'    => 'checkbox',
                'desc'    => __('Display success messages ("Discount X has been applied") on the cart and checkout pages.', 'yeediscounts'),
            ],
            [
                'title'   => __('Notification text', 'yeediscounts'),
                'id'      => 'yk_dd_cart_notification_text',
                'default' => 'Discount {discount_title} has been applied to your cart.',
                'type'    => 'text',
                'desc'    => __('Customize the success message. Use <code>{discount_title}</code> for the rule name.', 'yeediscounts'),
            ],
            [
                'title'   => __('Show strikeout on cart page', 'yeediscounts'),
                'desc'    => __('Display the original price struck out next to the discounted price in the cart.', 'yeediscounts'),
                'id'      => 'yk_dd_show_strikeout_on_cart_page',
                'default' => 'yes',
                'type'    => 'checkbox',
            ],
            [
                'title'    => __('Apply cart discount as', 'yeediscounts'),
                'desc'     => __('Choose whether cart discounts appear as a coupon line item or a fee.', 'yeediscounts'),
                'id'       => 'yk_dd_apply_cart_discount_as',
                'default'  => 'coupon',
                'type'     => 'select',
                'class'    => 'wc-enhanced-select',
                'options'  => [
                    'fee'    => __('Fee', 'yeediscounts'),
                    'coupon' => __('Coupon', 'yeediscounts'),
                ],
            ],
            [
                'title'    => __('Free shipping method title', 'yeediscounts'),
                'desc'     => __('Title for the free shipping method when applied by a rule.', 'yeediscounts'),
                'id'       => 'yk_dd_free_shipping_method_title',
                'type'     => 'text',
                'default'  => 'free shipping',
            ],
            [
                'type' => 'sectionend',
                'id'   => 'yk_dd_cart_section',
            ],
        ];
    }
    /**
     * Returns the available condition groups for rules.
     */
    public function get_condition_groups()
    {
        $gateways = WC()->payment_gateways->payment_gateways();
        $payment_options = [];
        foreach ($gateways as $gateway) {
            if ('yes' === $gateway->enabled) {
                $payment_options[$gateway->id] = $gateway->get_title();
            }
        }
        return [
            'cart' => [
                'label' => __('Cart', 'yeediscounts'),
                'conditions' => [
                    'cart_subtotal' => [
                        'label' => __('Subtotal', 'yeediscounts'),
                        'desc'  => __('Enter the subtotal amount (Number)', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                    'cart_items_quantity' => [
                        'label' => __('Item quantity', 'yeediscounts'),
                        'desc'  => __('Enter the total number of items (Number)', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                    'cart_coupon' => [
                        'label' => __('Coupons', 'yeediscounts'),
                        'desc'  => __('Search and select coupons (List)', 'yeediscounts'),
                        'value_type' => 'coupon_select',
                        'operators' => ['in', 'not_in'],
                    ],
                    'cart_items_weight' => [
                        'label' => __('Total weight', 'yeediscounts'),
                        'desc'  => __('Enter total weight', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                    'cart_payment_method' => [
                        'label' => __('Payment Method', 'yeediscounts'),
                        'desc'  => __('Select payment methods', 'yeediscounts'),
                        'value_type' => 'static_select',
                        'operators' => ['in', 'not_in'],
                        'options' => $payment_options,
                    ],
                    'cart_line_items_count' => [
                        'label' => __('Line Item Count', 'yeediscounts'),
                        'desc'  => __('Enter distinct line item count', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                ],
            ],
            'cart_items' => [
                'label' => __('Cart Items', 'yeediscounts'),
                'conditions' => [
                    'cart_item_product_attributes' => [
                        'label' => __('Attributes', 'yeediscounts'),
                        'desc'  => __('Search and select attributes', 'yeediscounts'),
                        'value_type' => 'attribute_select',
                        'operators' => ['in', 'not_in'],
                    ],
                    'cart_item_product_category' => [
                        'label' => __('Category', 'yeediscounts'),
                        'desc'  => __('Select categories', 'yeediscounts'),
                        'value_type' => 'taxonomy_select',
                        'taxonomy' => 'product_cat',
                        'operators' => ['in', 'not_in'],
                    ],
                    'cart_item_product_combination' => [
                        'label' => __('Product combination', 'yeediscounts'),
                        'desc'  => __('Select combined products', 'yeediscounts'),
                        'value_type' => 'product_select',
                        'operators' => ['in'],
                    ],
                    'cart_item_product_onsale' => [
                        'label' => __('On sale products', 'yeediscounts'),
                        'desc'  => __('Use the operator to choose Yes/No', 'yeediscounts'),
                        'value_type' => 'bool',
                        'operators' => ['yes', 'no'],
                    ],
                    'cart_item_product_sku' => [
                        'label' => __('SKU', 'yeediscounts'),
                        'desc'  => __('Enter SKUs', 'yeediscounts'),
                        'value_type' => 'text',
                        'operators' => ['in', 'not_in'],
                    ],
                    'cart_item_product_tags' => [
                        'label' => __('Product Tags', 'yeediscounts'),
                        'desc'  => __('Select tags', 'yeediscounts'),
                        'value_type' => 'taxonomy_select',
                        'taxonomy' => 'product_tag',
                        'operators' => ['in', 'not_in'],
                    ],
                    'cart_item_products' => [
                        'label' => __('Products', 'yeediscounts'),
                        'desc'  => __('Search and select products', 'yeediscounts'),
                        'value_type' => 'product_select',
                        'operators' => ['in', 'not_in'],
                    ],
                    'cart_item_category_combination' => [
                        'label' => __('Category combination', 'yeediscounts'),
                        'desc'  => __('Select category combination', 'yeediscounts'),
                        'value_type' => 'taxonomy_select',
                        'taxonomy' => 'product_cat',
                        'operators' => ['in'],
                    ],
                ],
            ],
            'date_time' => [
                'label' => __('Date & Time', 'yeediscounts'),
                'conditions' => [
                    'order_date' => [
                        'label' => __('Date', 'yeediscounts'),
                        'desc'  => __('Select date', 'yeediscounts'),
                        'value_type' => 'date',
                        'operators' => ['=', '!=', '>=', '<='],
                    ],
                    'order_date_and_time' => [
                        'label' => __('Date and Time', 'yeediscounts'),
                        'desc'  => __('Select date and time', 'yeediscounts'),
                        'value_type' => 'datetime',
                        'operators' => ['=', '!=', '>=', '<='],
                    ],
                    'order_days' => [
                        'label' => __('Days', 'yeediscounts'),
                        'desc'  => __('Select weekdays', 'yeediscounts'),
                        'value_type' => 'weekday',
                        'operators' => ['in', 'not_in'],
                    ],
                    'order_time' => [
                        'label' => __('Time', 'yeediscounts'),
                        'desc'  => __('Select time', 'yeediscounts'),
                        'value_type' => 'time',
                        'operators' => ['=', '!=', '>=', '<='],
                    ],
                ],
            ],
            'purchase' => [
                'label' => __('Purchase History', 'yeediscounts'),
                'conditions' => [
                    'purchase_first_order' => [
                        'label' => __('First order', 'yeediscounts'),
                        'desc'  => __('Is this the first order?', 'yeediscounts'),
                        'value_type' => 'bool',
                        'operators' => ['yes', 'no'],
                    ],
                    'purchase_last_order' => [
                        'label' => __('Last order', 'yeediscounts'),
                        'desc'  => __('Condition on last order (requires logic)', 'yeediscounts'),
                        'value_type' => 'bool',
                        'operators' => ['yes', 'no'],
                    ],
                    'purchase_last_order_amount' => [
                        'label' => __('Last order amount', 'yeediscounts'),
                        'desc'  => __('Amount of the previous order', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                    'purchase_previous_orders' => [
                        'label' => __('Number of orders made', 'yeediscounts'),
                        'desc'  => __('Count of previous orders', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                    'purchase_previous_orders_for_specific_product' => [
                        'label' => __('Number of orders made with following products', 'yeediscounts'),
                        'desc'  => __('Select products to check previous orders', 'yeediscounts'),
                        'value_type' => 'product_select',
                        'operators' => ['>='],
                    ],
                    'purchase_quantities_for_specific_product' => [
                        'label' => __('Number of quantities made with following products', 'yeediscounts'),
                        'desc'  => __('Select products to check previous quantities', 'yeediscounts'),
                        'value_type' => 'product_select',
                        'operators' => ['>='],
                    ],
                    'purchase_spent' => [
                        'label' => __('Total spent', 'yeediscounts'),
                        'desc'  => __('Total amount spent by customer', 'yeediscounts'),
                        'value_type' => 'number',
                        'operators' => ['>=', '>', '<=', '<'],
                    ],
                ],
            ],
            'shipping' => [
                'label' => __('Shipping', 'yeediscounts'),
                'conditions' => [
                    'shipping_city' => [
                        'label' => __('City', 'yeediscounts'),
                        'desc'  => __('Shipping City', 'yeediscounts'),
                        'value_type' => 'text',
                        'operators' => ['in', 'not_in', '='],
                    ],
                    'shipping_state' => [
                        'label' => __('State', 'yeediscounts'),
                        'desc'  => __('Shipping State/Province code', 'yeediscounts'),
                        'value_type' => 'text',
                        'operators' => ['in', 'not_in', '='],
                    ],
                    'shipping_zipcode' => [
                        'label' => __('Zipcode', 'yeediscounts'),
                        'desc'  => __('Shipping Zipcode', 'yeediscounts'),
                        'value_type' => 'text',
                        'operators' => ['in', 'not_in', '='],
                    ],
                    'shipping_country' => [
                        'label' => __('Country', 'yeediscounts'),
                        'desc'  => __('Select Shipping Country', 'yeediscounts'),
                        'value_type' => 'static_select',
                        'operators' => ['in', 'not_in'],
                        'options' => WC()->countries->get_countries(),
                    ],
                ],
            ],
            'billing' => [
                'label' => __('Billing', 'yeediscounts'),
                'conditions' => [
                    'billing_city' => [
                        'label' => __('City', 'yeediscounts'),
                        'desc'  => __('Billing City', 'yeediscounts'),
                        'value_type' => 'text',
                        'operators' => ['in', 'not_in', '='],
                    ],
                ],
            ],
            'customer' => [
                'label' => __('Customer', 'yeediscounts'),
                'conditions' => [
                    'user_email' => [
                        'label' => __('Email', 'yeediscounts'),
                        'desc'  => __('Enter emails', 'yeediscounts'),
                        'value_type' => 'text',
                        'operators' => ['in', 'not_in'],
                    ],
                    'user_role' => [
                        'label' => __('User role', 'yeediscounts'),
                        'desc'  => __('Select user roles', 'yeediscounts'),
                        'value_type' => 'static_select',
                        'operators' => ['in', 'not_in'],
                        'options' => wp_roles()->get_names(),
                    ],
                    'user_logged_in' => [
                        'label' => __('Is logged in', 'yeediscounts'),
                        'desc'  => __('Use the operator to choose Yes/No', 'yeediscounts'),
                        'value_type' => 'bool',
                        'operators' => ['yes', 'no'],
                    ],
                ],
            ],
        ];
    }
}
new Yeekit_Dynamic_Discounts_Settings_Tab();
