<?php
if (! defined('ABSPATH')) exit;
class Yeekit_Dynamic_Discounts_Rules_Manager
{
    const OPTION_KEY = 'yeekit_dynamic_discounts_rules';
    public static function get_rules(): array
    {
        $rules = get_option(self::OPTION_KEY, []);
        return is_array($rules) ? $rules : [];
    }
    public static function save_rules(array $rules): void
    {
        update_option(self::OPTION_KEY, $rules, false);
    }
    public static function get_active_rules(): array
    {
        $rules = self::get_rules();
        return array_filter($rules, function ($rule) {
            return !empty($rule['enabled']);
        });
    }
    /**
     * Get all matched rules (not just the first one)
     */
    public static function get_all_matched_rules()
    {
        $rules = self::get_rules();
        $matched_rules = [];

        foreach ($rules as $id => $rule) {
            if (empty($rule['enabled'])) {
                continue;
            }
            if (self::match_conditions($rule['conditions'] ?? [])) {
                $rule['id'] = $id; // Inject ID
                $matched_rules[] = $rule;
            }
        }

        // Check for exclusive rules
        foreach ($matched_rules as $rule) {
            if (!empty($rule['ignore_other_rules'])) {
                // If an exclusive rule is found, return ONLY that one.
                return [$rule];
            }
        }

        return $matched_rules;
    }

    /**
     * Check all rules and return matched rule(s) based on settings.
     * Returns single rule or null (based on yk_dd_apply_product_discount_to setting)
     */
    public static function get_matched_rule()
    {
        $matched_rules = self::get_all_matched_rules();

        if (empty($matched_rules)) {
            return null;
        }

        // Get setting for how to apply discount
        $apply_mode = get_option('yk_dd_apply_product_discount_to', 'biggest_discount');

        // For 'first' mode, return first matched rule (fastest)
        if ($apply_mode === 'first') {
            return $matched_rules[0];
        }

        // For 'all' mode, we can't return multiple rules in current architecture
        // This needs refactoring in discounts.php to handle multiple rules
        // For now, apply biggest discount as fallback
        if ($apply_mode === 'all') {
            // TODO: Refactor to support applying all rules simultaneously
            $apply_mode = 'biggest_discount';
        }

        // Calculate discount amounts for each rule to compare based on ACTUAL cart content
        if ($apply_mode === 'biggest_discount' || $apply_mode === 'lowest_discount') {
            $best_rule = null;
            $best_discount_value = -1.0;

            foreach ($matched_rules as $rule) {
                // Calculate actual savings for this rule on the current cart
                $discount_value = self::calculate_potential_savings($rule);

                if ($best_discount_value === -1.0) {
                    $best_rule = $rule;
                    $best_discount_value = $discount_value;
                    continue;
                }

                if ($apply_mode === 'biggest_discount' && $discount_value > $best_discount_value) {
                    $best_rule = $rule;
                    $best_discount_value = $discount_value;
                } elseif ($apply_mode === 'lowest_discount' && $discount_value < $best_discount_value) {
                    $best_rule = $rule;
                    $best_discount_value = $discount_value;
                }
            }

            // If the best rule has 0 savings, it might be that no items matched the filter.
            // If ALL rules have 0 savings, we might still return the "first" one or null?
            // If we strictly want to apply valid rules, maybe return null if value is 0?
            // For now, consistent with legacy, we return the one we found, but 0 value implies no effect.
            return $best_rule;
        }

        // Fallback: return first matched rule
        return $matched_rules[0];
    }

    /**
     * Calculate potential savings for a rule based on current cart
     */
    public static function calculate_potential_savings($rule)
    {
        $total_savings = 0.0;
        $cart = WC()->cart;

        if (!$cart) return 0.0;

        $type = $rule['type'] ?? 'yeekit_simple_discount';

        // Strategy: Iterate items, check filter, apply simplistic calc for sorting.
        // Complex rules (Bulk, BOGO) are hard to perfectly predict without running their full engine.
        // But we MUST check Filter::validate_item at minimum.

        foreach ($cart->get_cart() as $cart_item) {
            if (!Yeekit_Dynamic_Discounts_Filter::validate_item($cart_item, $rule)) {
                continue;
            }

            $qty = $cart_item['quantity'];
            $price = $cart_item['data']->get_price(); // Unit price
            $savings_item = 0;

            // SIMPLE DISCOUNT
            if ($type === 'yeekit_simple_discount') {
                $d_type = $rule['discount']['type'] ?? 'percentage';
                $d_val  = floatval($rule['discount']['value'] ?? 0);

                if ($d_type === 'percentage') {
                    $savings_item = $price * ($d_val / 100);
                } elseif ($d_type === 'fixed') { // Fixed amount off per item
                    $savings_item = $d_val;
                } elseif ($d_type === 'fixed_price') { // Set price to X
                    $savings_item = max(0, $price - $d_val);
                }
            }
            // BULK DISCOUNT (Approximation)
            elseif ($type === 'yeekit_bulk_discount') {
                // Check tiers
                $tiers = $rule['bulk_tiers'] ?? [];
                // This is rough because bulk applies to 'cumulative' quantities usually.
                // We'll just assume specific product qty for now for estimation.
                // If we want perfection, we need to sum up qtys first. Aggregation is expensive here?
                // Let's just find the best tier for THIS item's qty as a heuristic.
                $match_tier = null;
                foreach ($tiers as $tier) {
                    if ($qty >= $tier['min_qty'] && (empty($tier['max_qty']) || $qty <= $tier['max_qty'])) {
                        $match_tier = $tier;
                        // Keep looking? Tiers usually distinct.
                        break;
                    }
                }

                if ($match_tier) {
                    $t_val = floatval($match_tier['value']);
                    if ($match_tier['type'] === 'percentage') {
                        $savings_item = $price * ($t_val / 100);
                    } elseif ($match_tier['type'] === 'flat' || $match_tier['type'] === 'fixed') {
                        $savings_item = $t_val;
                    } elseif ($match_tier['type'] === 'fixed_price') {
                        $savings_item = max(0, $price - $t_val);
                    }
                }
            }
            // BOGO / OTHERS
            else {
                // For BOGO/Sets, it's very hard to estimate without running full logic.
                // But at least we checked Filter::validate_item!
                // If item validates, give it a score based on price so it beats non-matches.
                // Just assume 10% value as generic weight?
                $savings_item = $price * 0.1;
            }

            $total_savings += $savings_item * $qty;
        }

        return $total_savings;
    }

    /**
     * Estimate discount value for comparison purposes
     * Returns a numeric value representing discount magnitude
     */
    public static function estimate_rule_discount_value($rule)
    {
        // Simple estimation based on discount settings
        $discount = $rule['discount'] ?? [];
        $type = $discount['type'] ?? 'percentage';
        $value = floatval($discount['value'] ?? 0);

        // Normalize to comparable values
        // For percentage, use value as-is (10% = 10)
        // For fixed, multiply by 10 to make it comparable (e.g. $5 off = 50)
        // This is rough estimation, actual discount varies by cart
        if ($type === 'percentage') {
            return $value;
        } elseif ($type === 'fixed' || $type === 'fixed_price') {
            return $value * 10; // Weight fixed discounts higher
        }

        return 0;
    }
    /**
     * Match all conditions (AND logic)
     */
    protected static function match_conditions(array $conditions): bool
    {
        foreach ($conditions as $cond) {
            if (! self::match_condition($cond)) {
                return false;
            }
        }
        return true;
    }
    /**
     * Match a single condition
     */
    protected static function match_condition(array $cond): bool
    {
        $type     = $cond['type'] ?? '';
        $operator = $cond['operator'] ?? '';
        $value    = $cond['value'] ?? null;
        switch ($type) {
            /* ================= CART ================= */
            case 'cart_subtotal':
                // In 'before_calculate_totals', subtotal might be 0. Calculate it manually.
                $subtotal = WC()->cart->get_subtotal();
                if (empty($subtotal)) {
                    $subtotal = 0;
                    foreach (WC()->cart->get_cart() as $item) {
                        $price = $item['data']->get_price();
                        $subtotal += $price * $item['quantity'];
                    }
                }

                return self::compare(
                    $subtotal,
                    $operator,
                    floatval($value)
                );
            case 'cart_items_quantity':
                return self::compare(
                    WC()->cart->get_cart_contents_count(),
                    $operator,
                    intval($value)
                );
            case 'cart_coupon':
                $applied_coupons = WC()->cart->get_applied_coupons();
                $coupons_to_match = (array) $value; // Now array from select

                // WC coupons are usually case-insensitive storage but case-sensitive check? 
                // Best to normalize both to lowercase for comparison
                $applied_coupons = array_map('strtolower', $applied_coupons);
                $coupons_to_match = array_map('strtolower', $coupons_to_match);

                $found = !empty(array_intersect($applied_coupons, $coupons_to_match));
                return ($operator === 'in') ? $found : ! $found;
                /* ================= CART ITEMS ================= */
            case 'cart_item_products':

                return self::match_cart_items_by_product($operator, (array) $value);
            case 'cart_item_product_category':
                return self::match_cart_items_by_taxonomy('product_cat', $operator, (array) $value);
            case 'cart_item_product_tags':
                return self::match_cart_items_by_taxonomy('product_tag', $operator, (array) $value);
            case 'cart_item_product_attributes':
                return self::match_cart_items_by_attribute($operator, (array) $value);
            case 'cart_item_product_onsale':
                return self::match_cart_items_on_sale($operator);
            case 'cart_item_product_sku':
                $skus = is_array($value) ? $value : preg_split('/[\r\n,]+/', (string) $value, -1, PREG_SPLIT_NO_EMPTY);
                $skus = array_map('trim', $skus);
                return self::match_cart_items_by_sku($operator, $skus);
                /* ================= CUSTOMER ================= */
            case 'user_role':
                return self::match_user_role($operator, (array) $value);
        }
        return false;
    }
    /* ================= HELPERS ================= */
    protected static function compare($left, $operator, $right): bool
    {
        switch ($operator) {
            case '>=':
                return $left >= $right;
            case '>':
                return $left > $right;
            case '<=':
                return $left <= $right;
            case '<':
                return $left < $right;
            case '=':
                return $left == $right;
            case '!=':
                return $left != $right;
        }
        return false;
    }
    protected static function match_cart_items_by_product($operator, array $product_ids): bool
    {
        // Normalize rule IDs to string and unique
        $product_ids = array_unique(array_map('strval', $product_ids));

        // Gather all cart IDs (products and variations)
        $cart_ids = [];
        foreach (WC()->cart->get_cart() as $item) {
            $cart_ids[] = (string) $item['product_id'];
            if (!empty($item['variation_id'])) {
                $cart_ids[] = (string) $item['variation_id'];
            }
        }
        $cart_ids = array_unique($cart_ids);

        if ($operator === 'in') {
            // User requirement: MUST contain ALL selected products
            // If there are any product_ids NOT in cart_ids, then returns false
            $missing = array_diff($product_ids, $cart_ids);
            return empty($missing);
        } else {
            // not_in: MUST NOT contain ANY of the selected products
            $matches = array_intersect($cart_ids, $product_ids);
            return empty($matches);
        }
    }
    protected static function match_cart_items_by_taxonomy($taxonomy, $operator, array $term_ids): bool
    {
        if ($operator === 'in') {
            // AND LOGIC: For every selected term, there must be at least one item in the cart that matches it.
            foreach ($term_ids as $term_id) {
                $term_found_in_cart = false;
                foreach (WC()->cart->get_cart() as $item) {
                    if (has_term($term_id, $taxonomy, $item['product_id'])) {
                        $term_found_in_cart = true;
                        break;
                    }
                }
                if (!$term_found_in_cart) {
                    return false; // This term is missing from cart
                }
            }
            return true; // All terms found
        } else {
            // NOT IN: None of the selected terms should be present in cart?
            // Usually NOT IN means "Cart should not contain any item from these categories".
            // If any item matches any term, return false.
            foreach (WC()->cart->get_cart() as $item) {
                if (has_term($term_ids, $taxonomy, $item['product_id'])) {
                    return false; // Found a forbidden term
                }
            }
            return true;
        }
    }
    protected static function match_cart_items_by_attribute($operator, array $attributes): bool
    {
        if ($operator === 'in') {
            // AND logic: Cart must contain items matching ALL selected attributes
            foreach ($attributes as $attr) {
                $found_attr_in_cart = false;
                [$taxonomy, $slug] = explode(':', $attr);

                foreach (WC()->cart->get_cart() as $item) {
                    $product = $item['data'];
                    if (
                        $product->has_attribute($taxonomy) &&
                        in_array($slug, wc_get_product_terms($product->get_id(), $taxonomy, ['fields' => 'slugs']), true)
                    ) {
                        $found_attr_in_cart = true;
                        break;
                    }
                }

                if (!$found_attr_in_cart) {
                    return false; // Missing this attribute
                }
            }
            return true;
        } else {
            // NOT IN: Cart should NOT contain any of the selected attributes
            foreach (WC()->cart->get_cart() as $item) {
                $product = $item['data'];
                foreach ($attributes as $attr) {
                    [$taxonomy, $slug] = explode(':', $attr);
                    if (
                        $product->has_attribute($taxonomy) &&
                        in_array($slug, wc_get_product_terms($product->get_id(), $taxonomy, ['fields' => 'slugs']), true)
                    ) {
                        return false; // Found a forbidden attribute
                    }
                }
            }
            return true;
        }
    }
    protected static function match_cart_items_on_sale($operator): bool
    {
        $found = false;
        foreach (WC()->cart->get_cart() as $item) {
            if ($item['data']->is_on_sale()) {
                $found = true;
                break;
            }
        }
        return ($operator === 'yes') ? $found : ! $found;
    }
    protected static function match_cart_items_by_sku($operator, array $skus): bool
    {
        // Gather cart SKUs
        $cart_skus = [];
        foreach (WC()->cart->get_cart() as $item) {
            $cart_skus[] = $item['data']->get_sku();
        }
        $cart_skus = array_unique($cart_skus);
        $skus = array_unique($skus);

        if ($operator === 'in') {
            // AND LOGIC: Cart must contain ALL selected SKUs
            $missing = array_diff($skus, $cart_skus);
            return empty($missing);
        } else {
            // NOT IN: Cart must NOT contain ANY of the selected SKUs
            $matches = array_intersect($cart_skus, $skus);
            return empty($matches);
        }
    }
    protected static function match_user_role($operator, array $roles): bool
    {
        if (! is_user_logged_in()) {
            return false;
        }
        $user = wp_get_current_user();
        $found = array_intersect($user->roles, $roles);
        return ($operator === 'in') ? ! empty($found) : empty($found);
    }

    /**
     * Get potential matches (rules that are close to matching)
     * Currently supports: cart_subtotal, cart_items_quantity
     */
    public static function get_potential_matches()
    {
        $rules = self::get_rules();
        $potential_matches = [];

        foreach ($rules as $id => $rule) {
            if (empty($rule['enabled'])) {
                continue;
            }

            // check if rule is already matched (fully)
            if (self::match_conditions($rule['conditions'] ?? [])) {
                continue; // Already matched, not a potential match
            }

            // Check if "Basic" filters match (Product, Category, User Role, etc.)
            // We only want to show progress if the USER matches the eligibility criteria, just not the QUANTITY/AMOUNT criteria.
            if (!self::match_basic_filters($rule['conditions'] ?? [])) {
                continue;
            }

            // Now check "Progress" conditions (Subtotal, Quantity)
            $progress = self::check_progress_conditions($rule['conditions'] ?? []);
            if ($progress) {
                $rule['id'] = $id;
                $rule['progress'] = $progress; // ['type' => 'subtotal', 'missing' => 50, 'current' => 50, 'required' => 100]
                $potential_matches[] = $rule;
            }
        }

        // Return the "closest" match (smallest % missing)
        usort($potential_matches, function ($a, $b) {
            $pct_a = $a['progress']['missing'] / ($a['progress']['required'] ?: 1);
            $pct_b = $b['progress']['missing'] / ($b['progress']['required'] ?: 1);
            return $pct_a <=> $pct_b;
        });

        return !empty($potential_matches) ? $potential_matches[0] : null;
    }

    /**
     * Check if basic filters match (everything EXCEPT subtotal/quantity)
     */
    protected static function match_basic_filters($conditions)
    {
        $progress_types = ['cart_subtotal', 'cart_items_quantity'];

        foreach ($conditions as $cond) {
            if (in_array($cond['type'], $progress_types)) {
                continue; // Skip progress conditions
            }
            if (!self::match_condition($cond)) {
                return false; // specific requirement failed
            }
        }
        return true;
    }

    /**
     * Check progress for Subtotal / Quantity
     * Returns array with details or false
     */
    protected static function check_progress_conditions($conditions)
    {
        foreach ($conditions as $cond) {
            $type = $cond['type'];
            $value = floatval($cond['value']);
            $operator = $cond['operator']; // We only care about >= or >

            if ($type === 'cart_subtotal' && in_array($operator, ['>=', '>'])) {
                $subtotal = WC()->cart->get_subtotal();
                if (empty($subtotal)) {
                    $subtotal = 0;
                    foreach (WC()->cart->get_cart() as $item) {
                        $price = $item['data']->get_price();
                        $subtotal += $price * $item['quantity'];
                    }
                }

                if ($subtotal < $value) {
                    return [
                        'type' => 'amount',
                        'current' => $subtotal,
                        'required' => $value,
                        'missing' => $value - $subtotal
                    ];
                }
            } elseif ($type === 'cart_items_quantity' && in_array($operator, ['>=', '>'])) {
                $count = WC()->cart->get_cart_contents_count();
                if ($count < $value) {
                    return [
                        'type' => 'qty',
                        'current' => $count,
                        'required' => $value,
                        'missing' => $value - $count
                    ];
                }
            }
        }
        return false;
    }
}
