<?php
if (! defined('ABSPATH')) {
    exit;
}
/**
 * YeeDiscounts Reports Dashboard
 * Professional analytics and statistics for discount rules
 */
class Yeekit_Dynamic_Discounts_Reports
{
    /**
     * Constructor - Register AJAX hooks
     */
    public function __construct()
    {
        add_action('wp_ajax_yeekit_get_reports_data', [$this, 'ajax_get_reports_data']);
    }

    /**
     * AJAX handler to get reports data
     */
    public function ajax_get_reports_data()
    {
        // Verify nonce for security
        check_ajax_referer('yeekit-reports-nonce', 'nonce');

        // Get filter parameters
        $start_date = isset($_POST['start_date']) ? sanitize_text_field(wp_unslash($_POST['start_date'])) : current_time('Y-m-01');
        $end_date = isset($_POST['end_date']) ? sanitize_text_field(wp_unslash($_POST['end_date'])) : current_time('Y-m-d');
        $filter_rule_id = isset($_POST['filter_rule_id']) ? sanitize_text_field(wp_unslash($_POST['filter_rule_id'])) : '';

        // Get metrics
        $summary = $this->get_summary_metrics($start_date, $end_date, $filter_rule_id);
        $rule_performance = $this->get_rule_performance($start_date, $end_date, $filter_rule_id);
        $trend_data = $this->get_trend_data($start_date, $end_date, $filter_rule_id);
        $top_rules = $this->get_top_rules($start_date, $end_date, 5);
        $type_distribution = $this->get_type_distribution($rule_performance);

        // Prepare response
        $response = [
            'success' => true,
            'summary' => $summary,
            'trendData' => [
                'labels' => array_keys($trend_data),
                'values' => array_values($trend_data),
            ],
            'rulePerformance' => [
                'labels' => array_column($rule_performance, 'title'),
                'values' => array_column($rule_performance, 'order_count'),
            ],
            'typeDistribution' => [
                'labels' => array_keys($type_distribution),
                'values' => array_values($type_distribution),
            ],
            'topRules' => $top_rules,
        ];

        wp_send_json($response);
    }

    /**
     * Enqueue scripts and styles for reports
     */
    private function enqueue_scripts()
    {
        // Enqueue Chart.js library
        wp_enqueue_script(
            'yeekit-chartjs',
            YEEKIT_DYNAMIC_DISCOUNTS_WOOCOMMERCE_URL . 'assets/chart.min.js',
            [],
            YEEKIT_DYNAMIC_DISCOUNTS_VERSION,
            true
        );

        // Enqueue custom reports JS
        wp_enqueue_script(
            'yeekit-reports',
            YEEKIT_DYNAMIC_DISCOUNTS_WOOCOMMERCE_URL . 'assets/reports.js',
            ['jquery', 'yeekit-chartjs'],
            YEEKIT_DYNAMIC_DISCOUNTS_VERSION,
            true
        );

        // Pass configuration data only (no chart data - will load via AJAX)
        wp_localize_script('yeekit-reports', 'yeekitReportsData', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('yeekit-reports-nonce'),
            'initialFilters' => [
                'start_date' => isset($_GET['start_date']) ? sanitize_text_field(wp_unslash($_GET['start_date'])) : current_time('Y-m-01'),
                'end_date' => isset($_GET['end_date']) ? sanitize_text_field(wp_unslash($_GET['end_date'])) : current_time('Y-m-d'),
                'filter_rule_id' => isset($_GET['filter_rule_id']) ? sanitize_text_field(wp_unslash($_GET['filter_rule_id'])) : '',
            ],
            'i18n' => [
                'dailyDiscount' => __('Daily Discount Amount', 'yeediscounts'),
                'orderCount' => __('Order Count', 'yeediscounts'),
                'loading' => __('Loading...', 'yeediscounts'),
                'filter' => __('Filter', 'yeediscounts'),
            ],
        ]);
    }

    /**
     * Render the reports dashboard
     */
    public function render()
    {
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_rules();

        // Get filter values from URL (for form inputs only)
        $start_date = isset($_GET['start_date']) ? sanitize_text_field(wp_unslash($_GET['start_date'])) : current_time('Y-m-01'); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $end_date = isset($_GET['end_date']) ? sanitize_text_field(wp_unslash($_GET['end_date'])) : current_time('Y-m-d'); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $filter_rule_id = isset($_GET['filter_rule_id']) ? sanitize_text_field(wp_unslash($_GET['filter_rule_id'])) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

        // Enqueue scripts (data will be loaded via AJAX)
        $this->enqueue_scripts();
?>
        <div class="yeekit-settings-wrapper yeekit-reports">
            <h2><?php esc_html_e('Discount Reports', 'yeediscounts'); ?></h2>
            <!-- Filters -->
            <div class="yeekit-reports-filters">
                <div class="yeekit-filter-form">
                    <div class="yeekit-date-range-group">
                        <div class="yeekit-filter-group">
                            <label for="start_date"><?php esc_html_e('Start Date', 'yeediscounts'); ?></label>
                            <input type="date" id="start_date" name="start_date" value="<?php echo esc_attr($start_date); ?>" />
                        </div>
                        <div class="yeekit-filter-group">
                            <label for="end_date"><?php esc_html_e('End Date', 'yeediscounts'); ?></label>
                            <input type="date" id="end_date" name="end_date" value="<?php echo esc_attr($end_date); ?>" />
                        </div>
                        <div class="yeekit-filter-group">
                            <label for="filter_rule_id"><?php esc_html_e('Rule', 'yeediscounts'); ?></label>
                            <select id="filter_rule_id" name="filter_rule_id">
                                <option value=""><?php esc_html_e('All Rules', 'yeediscounts'); ?></option>
                                <?php foreach ($rules as $id => $r): ?>
                                    <option value="<?php echo esc_attr($id); ?>" <?php selected($filter_rule_id, (string)$id); ?>>
                                        <?php echo esc_html($r['title']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="yeekit-filter-actions">
                            <button type="button" id="yeekit-filter-btn" class="button button-primary"><?php esc_html_e('Filter', 'yeediscounts'); ?></button>
                            <a href="<?php echo esc_url(remove_query_arg(['start_date', 'end_date', 'filter_rule_id'])); ?>" class="button"><?php esc_html_e('Reset', 'yeediscounts'); ?></a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <!-- Summary Metric Cards -->
        <div class="yeekit-metrics-grid">
            <div class="yeekit-metric-card yeekit-metric-primary">
                <div class="yeekit-metric-icon">💰</div>
                <div class="yeekit-metric-content">
                    <div class="yeekit-metric-label"><?php esc_html_e('Total Discount', 'yeediscounts'); ?></div>
                    <div class="yeekit-metric-value yeekit-loading-skeleton">--</div>
                </div>
            </div>
            <div class="yeekit-metric-card yeekit-metric-success">
                <div class="yeekit-metric-icon">🛒</div>
                <div class="yeekit-metric-content">
                    <div class="yeekit-metric-label"><?php esc_html_e('Orders with Discount', 'yeediscounts'); ?></div>
                    <div class="yeekit-metric-value yeekit-loading-skeleton">--</div>
                </div>
            </div>
            <div class="yeekit-metric-card yeekit-metric-info">
                <div class="yeekit-metric-icon">📦</div>
                <div class="yeekit-metric-content">
                    <div class="yeekit-metric-label"><?php esc_html_e('Items Discounted', 'yeediscounts'); ?></div>
                    <div class="yeekit-metric-value yeekit-loading-skeleton">--</div>
                </div>
            </div>
            <div class="yeekit-metric-card yeekit-metric-warning">
                <div class="yeekit-metric-icon">📈</div>
                <div class="yeekit-metric-content">
                    <div class="yeekit-metric-label"><?php esc_html_e('Avg Discount/Order', 'yeediscounts'); ?></div>
                    <div class="yeekit-metric-value yeekit-loading-skeleton">--</div>
                </div>
            </div>
        </div>
        <!-- Charts Row 1 -->
        <div class="yeekit-charts-row">
            <div class="yeekit-chart-container">
                <h3><?php esc_html_e('Discount Trends', 'yeediscounts'); ?></h3>
                <canvas id="yeekit-trend-chart"></canvas>
            </div>
            <div class="yeekit-chart-container">
                <h3><?php esc_html_e('Rule Performance', 'yeediscounts'); ?></h3>
                <canvas id="yeekit-performance-chart"></canvas>
            </div>
        </div>
        <!-- Charts Row 2 -->
        <div class="yeekit-charts-row">
            <div class="yeekit-chart-container yeekit-chart-small">
                <h3><?php esc_html_e('Discount Type Distribution', 'yeediscounts'); ?></h3>
                <canvas id="yeekit-type-chart"></canvas>
            </div>
            <div class="yeekit-chart-container yeekit-chart-large">
                <h3><?php esc_html_e('Top Performing Rules', 'yeediscounts'); ?></h3>
                <div class="yeekit-top-rules-table">
                    <?php if (empty($top_rules)): ?>
                        <p class="yeekit-no-data"><?php esc_html_e('No data available for this period.', 'yeediscounts'); ?></p>
                    <?php else: ?>
                        <table class="widefat striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e('Rank', 'yeediscounts'); ?></th>
                                    <th><?php esc_html_e('Rule', 'yeediscounts'); ?></th>
                                    <th><?php esc_html_e('Type', 'yeediscounts'); ?></th>
                                    <th><?php esc_html_e('Orders', 'yeediscounts'); ?></th>
                                    <th><?php esc_html_e('Total Discount', 'yeediscounts'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($top_rules as $rank => $rule_data): ?>
                                    <tr>
                                        <td><strong><?php echo esc_html($rank + 1); ?></strong></td>
                                        <td><?php echo esc_html($rule_data['title']); ?></td>
                                        <td><code><?php echo esc_html($rule_data['type']); ?></code></td>
                                        <td><?php echo esc_html(number_format_i18n($rule_data['order_count'])); ?></td>
                                        <td><?php echo wp_kses_post(wc_price($rule_data['total_discount'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        </div>
<?php
    }
    /**
     * Get summary metrics with High Performance (HPOS & Transient support)
     *
     * @param string $start_date YYYY-MM-DD
     * @param string $end_date   YYYY-MM-DD
     * @param string $filter_rule_id
     * @return array
     */
    /**
     * Get summary metrics with High Performance (HPOS & Transient support)
     */
    private function get_summary_metrics($start_date, $end_date, $filter_rule_id = '')
    {
        global $wpdb;
        // 1. Generate a unique cache key based on the filters
        $cache_key = 'yeekit_report_sum_' . md5($start_date . $end_date . $filter_rule_id);
        // 2. Try to get data from Transient first (IMPORTANT: Uncommented for performance)
        $cached = get_transient($cache_key);
        if (false !== $cached) {
            return $cached;
        }
        // 3. Detect HPOS or Legacy storage
        $using_hpos  = \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        $order_table = $using_hpos ? $wpdb->prefix . 'wc_orders' : $wpdb->posts;
        $id_column   = $using_hpos ? 'id' : 'ID';
        $status_col  = $using_hpos ? 'status' : 'post_status';
        // Use GMT date column for HPOS consistency
        $date_col    = $using_hpos ? 'date_created_gmt' : 'post_date';
        // 4. Optimized SQL Query
        // Join itemmeta three times: rule_id, discount_amount, and qty
        // Use CAST for precision math
        $query = "
            SELECT 
                COUNT(DISTINCT o.{$id_column}) as order_count,
                COUNT(rm.meta_id) as item_count,
                SUM(CAST(dm.meta_value AS DECIMAL(10,2)) * CAST(qm.meta_value AS DECIMAL(10,2))) as total_discount
            FROM {$order_table} AS o
            INNER JOIN {$wpdb->prefix}woocommerce_order_items AS i ON o.{$id_column} = i.order_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS rm ON i.order_item_id = rm.order_item_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS dm ON i.order_item_id = dm.order_item_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS qm ON i.order_item_id = qm.order_item_id
            WHERE rm.meta_key = '_yeekit_rule_id'
              AND dm.meta_key = '_yeekit_discount_amount'
              AND qm.meta_key = '_qty'
              AND o.{$status_col} IN ('wc-completed', 'wc-processing', 'wc-on-hold')
        ";
        $params = [];
        // Date Filtering
        if ($start_date) {
            $query .= " AND o.{$date_col} >= %s";
            $params[] = $start_date . ' 00:00:00';
        }
        if ($end_date) {
            $query .= " AND o.{$date_col} <= %s";
            $params[] = $end_date . ' 23:59:59';
        }
        // Rule ID Filtering
        if ($filter_rule_id) {
            $query .= " AND rm.meta_value = %s";
            $params[] = $filter_rule_id;
        }
        // 5. Execute Query
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $result = $wpdb->get_row($wpdb->prepare($query, $params));
        // 6. Prepare data and calculate average
        $summary = [
            'order_count'    => (int) ($result->order_count ?? 0),
            'item_count'     => (int) ($result->item_count ?? 0),
            'total_discount' => (float) ($result->total_discount ?? 0),
            'avg_discount'   => 0,
        ];
        if ($summary['order_count'] > 0) {
            $summary['avg_discount'] = $summary['total_discount'] / $summary['order_count'];
        }
        // 7. Store in Transient for 6 hours
        set_transient($cache_key, $summary, 6 * HOUR_IN_SECONDS);
        return $summary;
    }
    /**
     * Get per-rule performance data
     *
     * @param string $start_date Start date.
     * @param string $end_date End date.
     * @param string $filter_rule_id Optional rule ID filter.
     * @return array Rule performance data.
     */
    /**
     * Get per-rule performance data with HPOS and Transient support
     */
    private function get_rule_performance($start_date, $end_date, $filter_rule_id = '')
    {
        global $wpdb;
        // 1. Generate a unique cache key
        $cache_key = 'yeekit_report_perf_' . md5($start_date . $end_date . $filter_rule_id);
        // 2. Try to get data from Transient
        $cached = get_transient($cache_key);
        if (false !== $cached) {
            return $cached;
        }
        // 3. Detect HPOS or Legacy storage
        $using_hpos  = \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        $order_table = $using_hpos ? $wpdb->prefix . 'wc_orders' : $wpdb->posts;
        $id_column   = $using_hpos ? 'id' : 'ID';
        $status_col  = $using_hpos ? 'status' : 'post_status';
        $date_col    = $using_hpos ? 'date_created_gmt' : 'post_date';
        // 4. Build Optimized SQL
        $query = "
            SELECT 
                rm.meta_value as rule_id,
                COUNT(DISTINCT o.{$id_column}) as order_count,
                COUNT(rm.meta_id) as item_count
            FROM {$order_table} AS o
            INNER JOIN {$wpdb->prefix}woocommerce_order_items AS i ON o.{$id_column} = i.order_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS rm ON i.order_item_id = rm.order_item_id
            WHERE rm.meta_key = %s
              AND o.{$status_col} IN ('wc-completed', 'wc-processing', 'wc-on-hold')
        ";
        $params = ['_yeekit_rule_id'];
        // Date Filtering
        if ($start_date) {
            $query .= " AND o.{$date_col} >= %s";
            $params[] = $start_date . ' 00:00:00';
        }
        if ($end_date) {
            $query .= " AND o.{$date_col} <= %s";
            $params[] = $end_date . ' 23:59:59';
        }
        // Specific Rule ID Filtering
        if ($filter_rule_id) {
            $query .= " AND rm.meta_value = %s";
            $params[] = $filter_rule_id;
        }
        $query .= " GROUP BY rm.meta_value ORDER BY order_count DESC";
        // 5. Execute Query
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $results = $wpdb->get_results($wpdb->prepare($query, $params));
        // 6. Map results with Rule Manager data
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_rules();
        $performance = [];
        foreach ($results as $row) {
            $rule_id = $row->rule_id;
            if (isset($rules[$rule_id])) {
                $performance[] = [
                    'rule_id'     => $rule_id,
                    'title'       => $rules[$rule_id]['title'] ?? 'Rule #' . $rule_id,
                    'type'        => $rules[$rule_id]['type'] ?? 'unknown',
                    'order_count' => (int) $row->order_count,
                    'item_count'  => (int) $row->item_count,
                ];
            }
        }
        // 7. Store in Transient for 6 hours
        set_transient($cache_key, $performance, 6 * HOUR_IN_SECONDS);
        return $performance;
    }
    /**
     * Get trend data for time-series chart
     *
     * @param string $start_date Start date.
     * @param string $end_date End date.
     * @param string $filter_rule_id Optional rule ID filter.
     * @return array Trend data [date => amount].
     */
    /**
     * Get trend data for time-series chart with HPOS and Transient support
     */
    private function get_trend_data($start_date, $end_date, $filter_rule_id = '')
    {
        global $wpdb;
        // 1. Generate a unique cache key
        $cache_key = 'yeekit_report_trend_' . md5($start_date . $end_date . $filter_rule_id);
        // 2. Try to get data from Transient
        $cached = get_transient($cache_key);
        // if (false !== $cached) {
        //     return $cached;
        // }
        // 3. Detect HPOS or Legacy storage
        $using_hpos  = \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        $order_table = $using_hpos ? $wpdb->prefix . 'wc_orders' : $wpdb->posts;
        $id_column   = $using_hpos ? 'id' : 'ID';
        $status_col  = $using_hpos ? 'status' : 'post_status';
        $date_col    = $using_hpos ? 'date_created_gmt' : 'post_date';
        // 4. Build Optimized SQL
        // Group by Date and Sum (Discount * Qty)
        $query = "
            SELECT 
                DATE(o.{$date_col}) as date_only,
                SUM(CAST(dm.meta_value AS DECIMAL(10,2)) * CAST(qm.meta_value AS DECIMAL(10,2))) as daily_discount
            FROM {$order_table} AS o
            INNER JOIN {$wpdb->prefix}woocommerce_order_items AS i ON o.{$id_column} = i.order_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS rm ON i.order_item_id = rm.order_item_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS dm ON i.order_item_id = dm.order_item_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS qm ON i.order_item_id = qm.order_item_id
            WHERE rm.meta_key = '_yeekit_rule_id'
              AND dm.meta_key = '_yeekit_discount_amount'
              AND qm.meta_key = '_qty'
              AND o.{$status_col} IN ('wc-completed', 'wc-processing', 'wc-on-hold')
        ";
        $params = [];
        // Date Filtering
        if ($start_date) {
            $query .= " AND o.{$date_col} >= %s";
            $params[] = $start_date . ' 00:00:00';
        }
        if ($end_date) {
            $query .= " AND o.{$date_col} <= %s";
            $params[] = $end_date . ' 23:59:59';
        }
        // Rule ID Filtering
        if ($filter_rule_id) {
            $query .= " AND rm.meta_value = %s";
            $params[] = $filter_rule_id;
        }
        $query .= " GROUP BY DATE(o.{$date_col}) ORDER BY date_only ASC";
        // 5. Execute Query
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $results = $wpdb->get_results($wpdb->prepare($query, $params));

        // 6. Format data for Chart.js [ 'YYYY-MM-DD' => total ]
        $trend = [];
        foreach ($results as $row) {
            $trend[$row->date_only] = (float) $row->daily_discount;
        }
        // 7. Store in Transient for 6 hours
        set_transient($cache_key, $trend, 6 * HOUR_IN_SECONDS);

        return $trend;
    }
    /**
     * Get top performing rules
     *
     * @param string $start_date Start date.
     * @param string $end_date End date.
     * @param int    $limit Number of top rules to return.
     * @return array Top rules data.
     */
    /**
     * Get top performing rules with HPOS and Transient support
     */
    private function get_top_rules($start_date, $end_date, $limit = 5)
    {
        global $wpdb;
        // 1. Generate a unique cache key based on the filters
        $cache_key = 'yeekit_report_top_' . md5($start_date . $end_date . $limit);
        // 2. Try to get data from Transient first
        $cached = get_transient($cache_key);
        if (false !== $cached) {
            return $cached;
        }
        // 3. Detect HPOS or Legacy storage
        $using_hpos  = \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        $order_table = $using_hpos ? $wpdb->prefix . 'wc_orders' : $wpdb->posts;
        $id_column   = $using_hpos ? 'id' : 'ID';
        $status_col  = $using_hpos ? 'status' : 'post_status';
        $date_col    = $using_hpos ? 'date_created_gmt' : 'post_date';
        // 4. Optimized SQL Query
        // Join itemmeta for discount amount, rule ID, and quantity
        $query = "
            SELECT 
                rm.meta_value as rule_id,
                COUNT(DISTINCT o.{$id_column}) as order_count,
                SUM(CAST(dm.meta_value AS DECIMAL(10,2)) * CAST(qm.meta_value AS DECIMAL(10,2))) as total_discount
            FROM {$order_table} AS o
            INNER JOIN {$wpdb->prefix}woocommerce_order_items AS i ON o.{$id_column} = i.order_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS rm ON i.order_item_id = rm.order_item_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS dm ON i.order_item_id = dm.order_item_id
            INNER JOIN {$wpdb->prefix}woocommerce_order_itemmeta AS qm ON i.order_item_id = qm.order_item_id
            WHERE rm.meta_key = '_yeekit_rule_id'
              AND dm.meta_key = '_yeekit_discount_amount'
              AND qm.meta_key = '_qty'
              AND o.{$status_col} IN ('wc-completed', 'wc-processing', 'wc-on-hold')
        ";
        $params = [];
        // Date Filtering
        if ($start_date) {
            $query .= " AND o.{$date_col} >= %s";
            $params[] = $start_date . ' 00:00:00';
        }
        if ($end_date) {
            $query .= " AND o.{$date_col} <= %s";
            $params[] = $end_date . ' 23:59:59';
        }
        $query .= " GROUP BY rm.meta_value ORDER BY total_discount DESC LIMIT %d";
        $params[] = (int) $limit;
        // 5. Execute Query
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $results = $wpdb->get_results($wpdb->prepare($query, $params));
        // 6. Map results with Rule Manager
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_rules();
        $top_rules = [];
        foreach ($results as $row) {
            $rule_id = $row->rule_id;
            if (isset($rules[$rule_id])) {
                $top_rules[] = [
                    'rule_id'        => $rule_id,
                    'title'          => $rules[$rule_id]['title'] ?? 'Rule #' . $rule_id,
                    'type'           => $rules[$rule_id]['type'] ?? 'unknown',
                    'order_count'    => (int) $row->order_count,
                    'total_discount' => (float) $row->total_discount,
                ];
            }
        }
        // 7. Store in Transient for 6 hours
        set_transient($cache_key, $top_rules, 6 * HOUR_IN_SECONDS);
        return $top_rules;
    }
    /**
     * Get type distribution for doughnut chart
     *
     * @param array $rule_performance Rule performance data.
     * @return array Type distribution [type => count].
     */
    private function get_type_distribution($rule_performance)
    {
        $distribution = [];
        foreach ($rule_performance as $rule) {
            $type = $rule['type'];
            if (!isset($distribution[$type])) {
                $distribution[$type] = 0;
            }
            $distribution[$type] += $rule['order_count'];
        }
        return $distribution;
    }
}
