<?php
if (! defined('ABSPATH')) exit;
class Yeekit_Dynamic_Discounts_Orders
{
    public function __construct()
    {
        // 1. Capture Rule IDs during Order Creation
        // NOTE: backend/order_tracker.php handles saving _yeekit_rule_id to items.
        // We only need to aggregate them to the Order Level for filtering.
        add_action('woocommerce_checkout_create_order', [$this, 'aggregate_order_rule_ids'], 20);
        add_action('woocommerce_store_api_checkout_order_processed', [$this, 'aggregate_order_rule_ids'], 20);
        // 2. Admin Filter UI (Legacy Post Edit)
        add_action('restrict_manage_posts', [$this, 'render_rule_filter_legacy'], 20);
        add_action('pre_get_posts', [$this, 'handle_filter_query_legacy']);
        // 3. Admin Filter UI (HPOS)
        add_action('woocommerce_order_list_table_restrict_manage_orders', [$this, 'render_rule_filter_hpos'], 20);
        add_filter('woocommerce_order_query_args', [$this, 'handle_filter_query_hpos']);

        // 4. Display Meta in Order Items
        add_filter('woocommerce_order_item_get_formatted_meta_data', [$this, 'format_order_item_meta'], 10, 2);
        add_filter('woocommerce_hidden_order_itemmeta', [$this, 'hide_order_item_meta']);
    }
    /**
     * Aggregate all rule IDs from items and save to Order Meta
     */
    public function aggregate_order_rule_ids($order, $data = null)
    {
        $rule_ids = [];

        // Check Line Items
        foreach ($order->get_items() as $item) {
            $rid = $item->get_meta('_yeekit_rule_id');
            // Check strictly for not empty string but allow '0'
            if ($rid !== '' && $rid !== false && $rid !== null) {
                $rule_ids[] = $rid;
            }
        }

        // Check Fees
        foreach ($order->get_fees() as $item) {
            $rid = $item->get_meta('_yeekit_rule_id');
            if ($rid !== '' && $rid !== false && $rid !== null) {
                $rule_ids[] = $rid;
            }
        }
        if (!empty($rule_ids)) {
            $unique_ids = array_unique($rule_ids);

            foreach ($unique_ids as $id) {
                // Save for efficient filtering
                $order->add_meta_data('_yeekit_applied_rule_id', $id, false);
            }
        }
    }
    // ===== FILTER UI =====
    private function get_rules_for_filter()
    {
        $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();
        $options = [];
        foreach ($rules as $id => $rule) {
            // Ensure ID is available. rules.php usually returns keyed by ID or array with injected ID?
            // get_active_rules() returns array_filter(get_rules()). 
            // get_rules() uses get_option which returns array.
            // We need to verify if $rule['id'] is set or if key is ID.
            // Rules Logic: $id is the key in the main array.
            $rule_id = $rule['id'] ?? $id;
            $options[$rule_id] = $rule['title'] ?? __('Rule', 'yeediscounts') . " #$rule_id";
        }
        return $options;
    }
    public function render_rule_filter_legacy($post_type)
    {
        if ($post_type !== 'shop_order') return;
        $this->render_dropdown();
    }
    public function render_rule_filter_hpos($which)
    {
        $this->render_dropdown();
    }
    private function render_dropdown()
    {
        $rules = $this->get_rules_for_filter();
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $current = isset($_GET['yeekit_filter_rule']) ? sanitize_text_field(wp_unslash($_GET['yeekit_filter_rule'])) : '';
?>
        <select name="yeekit_filter_rule">
            <option value=""><?php esc_html_e('Filter by Discount Rule', 'yeediscounts'); ?></option>
            <?php foreach ($rules as $id => $title) : ?>
                <option value="<?php echo esc_attr($id); ?>" <?php selected($current, $id); ?>>
                    <?php echo esc_html($title); ?>
                </option>
            <?php endforeach; ?>
        </select>
<?php
    }
    // ===== QUERY HANDLING =====
    public function handle_filter_query_legacy($query)
    {
        global $pagenow;
        if ($pagenow !== 'edit.php' || $query->get('post_type') !== 'shop_order') {
            return;
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $rule_id = isset($_GET['yeekit_filter_rule']) ? sanitize_text_field(wp_unslash($_GET['yeekit_filter_rule'])) : '';
        if (empty($rule_id)) return;
        $meta_query = $query->get('meta_query');
        if (empty($meta_query)) $meta_query = [];
        $meta_query[] = [
            'key'     => '_yeekit_applied_rule_id',
            'value'   => $rule_id,
            'compare' => '='
        ];

        $query->set('meta_query', $meta_query);
    }
    public function handle_filter_query_hpos($args)
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $rule_id = isset($_GET['yeekit_filter_rule']) ? sanitize_text_field(wp_unslash($_GET['yeekit_filter_rule'])) : '';
        if ($rule_id == "") return $args;

        $meta_query = $args['meta_query'] ?? [];
        $meta_query[] = [
            'key'     => '_yeekit_applied_rule_id',
            'value'   => $rule_id,
            'compare' => '='
        ];
        $args['meta_query'] = $meta_query;
        return $args;
    }

    /**
     * Display Yeekit Meta in Order Items table (Admin + Frontend)
     */
    public function format_order_item_meta($formatted_meta, $item)
    {
        // 1. Discount Rule
        if (!is_admin()) {
            return $formatted_meta;
        }
        $rule_id = $item->get_meta('_yeekit_rule_id');
        if ($rule_id) {
            $rules = Yeekit_Dynamic_Discounts_Rules_Manager::get_active_rules();
            $rule_title = isset($rules[$rule_id]['title']) ? $rules[$rule_id]['title'] . " - " . $rule_id : __('Rule', 'yeediscounts') . " #$rule_id";
            $formatted_meta[] = (object) [
                'key'           => 'yeekit_rule_title',
                'label'         => __('Discount Rule', 'yeediscounts'),
                'value'         => $rule_title,
                'display_key'   => __('Discount Rule', 'yeediscounts'),
                'display_value' => $rule_title,
            ];



            // 1b. Discount Type
            if (!empty($rules[$rule_id]['type'])) {
                $type_key = $rules[$rule_id]['type'];
                $type_labels = [
                    'yeekit_simple_discount' => __('Product Adjustment', 'yeediscounts'),
                    'yeekit_cart_discount'   => __('Cart Adjustment', 'yeediscounts'),
                    'yeekit_free_shipping'   => __('Free Shipping', 'yeediscounts'),
                    'yeekit_bulk_discount'   => __('Bulk Discount', 'yeediscounts'),
                    'yeekit_set_discount'    => __('Bundle (Set) Discount', 'yeediscounts'),
                    'yeekit_buy_x_get_x'     => __('Buy X get X', 'yeediscounts'),
                    'yeekit_buy_x_get_y'     => __('Buy X get Y', 'yeediscounts'),
                ];
                $type_label = isset($type_labels[$type_key]) ? $type_labels[$type_key] : $type_key;

                $formatted_meta[] = (object) [
                    'key'           => 'yeekit_rule_type',
                    'label'         => __('Type', 'yeediscounts'),
                    'value'         => $type_label,
                    'display_key'   => __('Type', 'yeediscounts'),
                    'display_value' => $type_label,
                ];
                $_yeekit_discount_type = $item->get_meta('_yeekit_discount_type');
                $_yeekit_discount_value = $item->get_meta('_yeekit_discount_value');

                $formatted_value = $_yeekit_discount_value;
                if ($_yeekit_discount_type === 'percentage') {
                    $formatted_value .= '%';
                } elseif (in_array($_yeekit_discount_type, ['fixed', 'fixed_amount', 'fixed_price', 'flat'])) {
                    $formatted_value = wc_price($_yeekit_discount_value);
                }

                // Map value type to label based on Rule Type
                $val_type_labels = [];
                if ($type_key === 'yeekit_buy_x_get_x') {
                    $val_type_labels = [
                        'free_product' => __('Free', 'yeediscounts'),
                        'percentage'   => __('Percentage discount', 'yeediscounts'),
                        'flat'         => __('Fixed discount', 'yeediscounts'),
                    ];
                } else {
                    $val_type_labels = [
                        'percentage'      => __('Percentage discount', 'yeediscounts'),
                        'fixed'           => __('Fixed discount', 'yeediscounts'),
                        'fixed_per_item'  => __('Fixed discount per item', 'yeediscounts'),
                        'fixed_price'     => __('Fixed price per item', 'yeediscounts'),
                    ];
                }

                $val_type_label = isset($val_type_labels[$_yeekit_discount_type]) ? $val_type_labels[$_yeekit_discount_type] : $_yeekit_discount_type;

                $formatted_meta[] = (object) [
                    'key'           => 'yeekit_discount_type',
                    'label'         => __('Discount Type', 'yeediscounts'),
                    'value'         => $val_type_label,
                    'display_key'   => __('Discount Type', 'yeediscounts'),
                    'display_value' => $val_type_label,
                ];
                $formatted_meta[] = (object) [
                    'key'           => 'yeekit_discount_value',
                    'label'         => __('Discount Value', 'yeediscounts'),
                    'value'         => $formatted_value,
                    'display_key'   => __('Discount Value', 'yeediscounts'),
                    'display_value' => $formatted_value,
                ];
            }
        }

        // 2. Discount Value / Amount
        $discount_amount = $item->get_meta('_yeekit_discount_amount');
        if ($discount_amount) {
            $_yeekit_original_price = $item->get_meta('_yeekit_original_price');
            if ($_yeekit_original_price) {
                $formatted_meta[] = (object) [
                    'key'           => 'yeekit_original_price',
                    'label'         => __('Original Price', 'yeediscounts'),
                    'value'         => wc_price($_yeekit_original_price),
                    'display_key'   => __('Original Price', 'yeediscounts'),
                    'display_value' => wc_price($_yeekit_original_price),
                ];
            }
            $qty = $item->get_quantity();
            $total_discount = floatval($discount_amount) * $qty;

            $formatted_meta[] = (object) [
                'key'           => 'yeekit_discount_amount',
                'label'         => __('Discount', 'yeediscounts'),
                'value'         => wc_price($discount_amount),
                'display_key'   => __('Discount', 'yeediscounts'),
                'display_value' => wc_price($discount_amount),
            ];
            $formatted_meta[] = (object) [
                'key'           => 'yeekit_discount_total',
                'label'         => __('Discount Total', 'yeediscounts'),
                'value'         => wc_price($total_discount),
                'display_key'   => __('Discount Total', 'yeediscounts'),
                'display_value' => wc_price($total_discount),
            ];
        }

        return $formatted_meta;
    }

    /**
     * Explicitly hide raw Yeekit meta keys
     */
    public function hide_order_item_meta($hidden_keys)
    {
        $keys_to_hide = [
            '_yeekit_rule_id',
            '_yeekit_discount_amount',
            '_yeekit_rule_type',
            '_yeekit_is_discounted',
            '_yeekit_discount_type',
            '_yeekit_discount_value',
            '_yeekit_original_price',
            '_yeekit_final_price'
        ];
        return array_merge($hidden_keys, $keys_to_hide);
    }
}
new Yeekit_Dynamic_Discounts_Orders();
