<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Render Rules (Conditions) Block
 *
 * @param array  $rule     The discount rule data.
 * @param object $settings The settings object to get condition groups.
 */
function yeekit_render_rules_block($rule, $settings)
{
?>
    <div class="yeekit-dd-block">
        <h3 class="yeekit-dd-block-title"><?php esc_html_e('Rules', 'yeediscounts'); ?></h3>
        <div class="yeekit-dd-block-content">
            <p class="description yeekit-block-desc">
                <?php esc_html_e('Define conditions that must be met for the discount to apply. E.g. Subtotal > 100.', 'yeediscounts'); ?>
            </p>
            <?php
            $condition_groups = $settings->get_condition_groups();
            $condition_map = [];
            foreach ($condition_groups as $group) {
                foreach ($group['conditions'] as $key => $cfg) {
                    $condition_map[$key] = $cfg;
                }
            }
            ?>
            <div id="yeekit-dd-conditions-wrap">
                <?php
                $conditions = (! empty($rule['conditions']) && is_array($rule['conditions'])) ? $rule['conditions'] : [];
                if (count($conditions) > 0) {
                    foreach ($conditions as $index => $condition):
                ?>
                        <div class="yeekit-cond-row" data-index="<?php echo esc_attr($index); ?>">
                            <select name="rules[type][<?php echo esc_attr($index); ?>]" class="yeekit-cond-type">
                                <?php foreach ($settings->get_condition_groups() as $group): ?>
                                    <optgroup label="<?php echo esc_attr($group['label']); ?>">
                                        <?php foreach ($group['conditions'] as $type_key => $cfg): ?>
                                            <option value="<?php echo esc_attr($type_key); ?>"
                                                data-desc="<?php echo esc_attr($cfg['desc']); ?>"
                                                <?php selected($condition['type'], $type_key); ?>>
                                                <?php echo esc_html($cfg['label']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endforeach; ?>
                            </select>
                            <?php
                            $current_type = $condition['type'] ?? '';
                            $cfg = $condition_map[$current_type] ?? [];
                            $operators = $cfg['operators'] ?? [];
                            ?>
                            <select name="rules[operator][<?php echo esc_attr($index); ?>]" class="yeekit-cond-operator">
                                <?php foreach ($operators as $op): ?>
                                    <option value="<?php echo esc_attr($op); ?>" <?php selected($condition['operator'], $op); ?>>
                                        <?php echo esc_html($op); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <?php
                            $type  = $condition['type'];
                            $value = $condition['value'] ?? '';
                            $cfg   = $condition_map[$type] ?? [];
                            ?>
                            <div class="yeekit-cond-value">
                                <?php switch ($cfg['value_type'] ?? 'text'):
                                    case 'number': ?>
                                        <input type="number" step="0.01" name="rules[value][<?php echo esc_attr($index); ?>]" value="<?php echo esc_attr($value); ?>">
                                    <?php break;
                                    case 'bool':
                                    case 'boolean': ?>
                                        <input type="hidden" name="rules[value][<?php echo esc_attr($index); ?>]" value="">
                                    <?php break;
                                    case 'product_select': ?>
                                        <select class="yeekit-product-ajax-select" name="rules[value][<?php echo esc_attr($index); ?>][]" multiple>
                                            <?php foreach ((array) $value as $pid):
                                                $product = wc_get_product($pid);
                                                if (! $product) continue; ?>
                                                <option value="<?php echo esc_attr($pid); ?>" selected><?php echo esc_html($product->get_name()); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php break;
                                    case 'coupon_select': ?>
                                        <select name="rules[value][<?php echo esc_attr($index); ?>][]" multiple>
                                            <?php foreach ((array) $value as $code): ?>
                                                <option value="<?php echo esc_attr($code); ?>" selected><?php echo esc_html($code); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php break;
                                    case 'attribute_select': ?>
                                        <select class="yeekit-attribute-search" name="rules[value][<?php echo esc_attr($index); ?>][]" multiple>
                                            <?php foreach ((array) $value as $slug):
                                                $term = get_term_by('slug', $slug, $slug);
                                                $label = $term ? $term->name : $slug; ?>
                                                <option value="<?php echo esc_attr($slug); ?>" selected><?php echo esc_html($label); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php break;
                                    case 'taxonomy_select':
                                        $taxonomy = $cfg['taxonomy'] ?? '';
                                        $terms = get_terms(['taxonomy' => $taxonomy, 'hide_empty' => false]); ?>
                                        <select class="wc-enhanced-select" name="rules[value][<?php echo esc_attr($index); ?>][]" multiple>
                                            <?php foreach ($terms as $term): ?>
                                                <option value="<?php echo esc_attr($term->term_id); ?>" <?php if (in_array($term->term_id, (array) $value)) echo wp_kses_post(' selected="selected"'); ?>><?php echo esc_html($term->name); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php break;
                                    case 'date': ?>
                                        <input type="date" name="rules[value][<?php echo esc_attr($index); ?>]" value="<?php echo esc_attr($value); ?>">
                                    <?php break;
                                    case 'datetime': ?>
                                        <input type="datetime-local" name="rules[value][<?php echo esc_attr($index); ?>]" value="<?php echo esc_attr($value); ?>">
                                    <?php break;
                                    case 'time': ?>
                                        <input type="time" name="rules[value][<?php echo esc_attr($index); ?>]" value="<?php echo esc_attr($value); ?>">
                                    <?php break;
                                    case 'weekday':
                                        $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday']; ?>
                                        <select class="wc-enhanced-select" name="rules[value][<?php echo esc_attr($index); ?>][]" multiple style="width: 200px">
                                            <?php foreach ($days as $day): ?>
                                                <option value="<?php echo esc_attr($day); ?>" <?php if (in_array($day, (array)$value)) echo wp_kses_post(' selected="selected"'); ?>><?php echo esc_html(ucfirst($day)); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php break;
                                    case 'static_select':
                                        $opts = $cfg['options'] ?? []; ?>
                                        <select class="wc-enhanced-select" name="rules[value][<?php echo esc_attr($index); ?>][]" multiple style="width: 200px">
                                            <?php foreach ($opts as $k => $label): ?>
                                                <option value="<?php echo esc_attr($k); ?>" <?php if (in_array($k, (array)$value)) echo wp_kses_post(' selected="selected"'); ?>><?php echo esc_html($label); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                    <?php break;
                                    default: ?>
                                        <input type="text" name="rules[value][<?php echo esc_attr($index); ?>]" value="<?php echo esc_attr($value); ?>">
                                <?php endswitch; ?>
                            </div>
                            <span class="yeekit-cond-remove dashicons dashicons-no-alt" title="<?php esc_attr_e('Remove condition', 'yeediscounts'); ?>"></span>
                            <?php if (!empty($cfg['desc'])): ?>
                                <p class="description yeekit-cond-desc"><?php echo esc_html($cfg['desc']); ?></p>
                            <?php endif; ?>
                        </div>
                <?php endforeach;
                } ?>
            </div>
            <p class="yeekit-dd-cond-actions">
                <button type="button" class="button yeekit-dd-add-condition" id="yeekit-dd-add-condition">
                    <?php esc_html_e('Add condition', 'yeediscounts'); ?>
                </button>
            </p>
        </div>
    </div>
<?php
}

// Call the render function.
yeekit_render_rules_block($rule, $settings);
