/**
 * YeeDiscounts Reports Dashboard - Chart.js Initialization with AJAX
 * 
 * @package YeeDiscounts
 */

(function ($) {
    'use strict';

    // Store chart instances globally
    let trendChart, perfChart, typeChart;

    /**
     * Initialize charts with data
     */
    function initCharts(data) {
        // Trend Chart (Line Chart)
        const trendCtx = document.getElementById('yeekit-trend-chart');
        if (trendCtx && data.trendData) {
            // Destroy existing chart if exists
            if (trendChart) {
                trendChart.destroy();
            }

            trendChart = new Chart(trendCtx, {
                type: 'line',
                data: {
                    labels: data.trendData.labels,
                    datasets: [{
                        label: data.i18n.dailyDiscount,
                        data: data.trendData.values,
                        borderColor: '#2271b1',
                        backgroundColor: 'rgba(34, 113, 177, 0.1)',
                        tension: 0.3,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }

        // Performance Chart (Bar Chart)
        const perfCtx = document.getElementById('yeekit-performance-chart');
        if (perfCtx && data.rulePerformance && data.rulePerformance.labels.length > 0) {
            // Destroy existing chart if exists
            if (perfChart) {
                perfChart.destroy();
            }

            perfChart = new Chart(perfCtx, {
                type: 'bar',
                data: {
                    labels: data.rulePerformance.labels,
                    datasets: [{
                        label: data.i18n.orderCount,
                        data: data.rulePerformance.values,
                        backgroundColor: '#00a32a'
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true
                        }
                    }
                }
            });
        }

        // Type Distribution Chart (Doughnut)
        const typeCtx = document.getElementById('yeekit-type-chart');
        if (typeCtx && data.typeDistribution) {
            // Destroy existing chart if exists
            if (typeChart) {
                typeChart.destroy();
            }

            typeChart = new Chart(typeCtx, {
                type: 'doughnut',
                data: {
                    labels: data.typeDistribution.labels,
                    datasets: [{
                        data: data.typeDistribution.values,
                        backgroundColor: [
                            '#2271b1', '#00a32a', '#d63638', '#f0b849',
                            '#826eb4', '#c9356e', '#00a0d2', '#72aee6'
                        ]
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom'
                        }
                    }
                }
            });
        }
    }

    /**
     * Update summary metrics cards
     */
    function updateSummaryMetrics(summary) {
        const metrics = document.querySelectorAll('.yeekit-metric-value');

        // Remove skeleton class and update values
        // Order: Total Discount, Orders, Items, Avg Discount
        if (metrics[0]) {
            metrics[0].classList.remove('yeekit-loading-skeleton');
            metrics[0].textContent = '$' + new Intl.NumberFormat('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(summary.total_discount || 0);
        }

        if (metrics[1]) {
            metrics[1].classList.remove('yeekit-loading-skeleton');
            metrics[1].textContent = new Intl.NumberFormat().format(summary.order_count || 0);
        }

        if (metrics[2]) {
            metrics[2].classList.remove('yeekit-loading-skeleton');
            metrics[2].textContent = new Intl.NumberFormat().format(summary.item_count || 0);
        }

        if (metrics[3]) {
            metrics[3].classList.remove('yeekit-loading-skeleton');
            metrics[3].textContent = '$' + new Intl.NumberFormat('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            }).format(summary.avg_discount || 0);
        }
    }

    /**
     * Fetch reports data via AJAX
     */
    function fetchReportsData(startDate, endDate, ruleId) {
        // Show loading state
        $('.yeekit-reports').addClass('loading');
        $('.yeekit-filter-btn').prop('disabled', true).text(yeekitReportsData.i18n.loading);
        $.ajax({
            url: yeekitReportsData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'yeekit_get_reports_data',
                nonce: yeekitReportsData.nonce,
                start_date: startDate,
                end_date: endDate,
                filter_rule_id: ruleId
            },
            success: function (response) {
                if (response.success) {
                    // Merge i18n data from localized script
                    response.i18n = yeekitReportsData.i18n;

                    // Update charts
                    initCharts(response);

                    // Update summary metrics
                    updateSummaryMetrics(response.summary);

                    // Update URL without reload
                    const url = new URL(window.location);
                    url.searchParams.set('start_date', startDate);
                    url.searchParams.set('end_date', endDate);
                    if (ruleId) {
                        url.searchParams.set('filter_rule_id', ruleId);
                    } else {
                        url.searchParams.delete('filter_rule_id');
                    }
                    window.history.pushState({}, '', url);
                }
            },
            error: function () {
                alert('Failed to load reports data. Please try again.');
            },
            complete: function () {
                // Remove loading state
                $('.yeekit-reports').removeClass('loading');
                $('.yeekit-filter-btn').prop('disabled', false).text('Filter');
            }
        });
    }

    $(document).ready(function () {
        // Check if report data exists
        if (typeof yeekitReportsData === 'undefined') {
            return;
        }

        // Auto-load data on page load with initial filters
        const initialFilters = yeekitReportsData.initialFilters;
        fetchReportsData(initialFilters.start_date, initialFilters.end_date, initialFilters.filter_rule_id);

        // Handle filter button click
        $('#yeekit-filter-btn').on('click', function (e) {
            e.preventDefault();

            const startDate = $('#start_date').val();
            const endDate = $('#end_date').val();
            const ruleId = $('#filter_rule_id').val();

            // Fetch new data via AJAX
            fetchReportsData(startDate, endDate, ruleId);
        });
    });

})(jQuery);
