<?php
/**
 * Plugin Name: XTND Table Of Content
 * Description: Adds a dynamic, customizable table of contents block to your WordPress posts and pages. Automatically generates anchor links for headings, improving navigation and SEO. Supports both RTL and LTR languages for optimal accessibility.
 * Version: 1.0.0
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Author: Xtnd
 * Author URI: https://xtnd.net
 * Text Domain: xtnd-table-of-content
 *
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @package XTND_TABLE_OF_CONTENT
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// phpcs:ignoreFile WordPress.Files.FileName.InvalidClassFileName -- File name kept for plugin consistency.
if ( ! class_exists( 'XTND_TABLE_OF_CONTENT' ) ) {
	/**
	 * Main plugin class for the XTND Table of Contents.
	 *
	 * Initializes hooks, registers the block, and processes heading IDs.
	 *
	 * @package XTND_TABLE_OF_CONTENT
	 */
	class XTND_TABLE_OF_CONTENT {
		/**
		 * The single instance of the class.
		 *
		 * @var XTND_TABLE_OF_CONTENT|null
		 */
		private static $instance = null;

		/**
		 * Main XTND_TABLE_OF_CONTENT Instance.
		 *
		 * @return XTND_TABLE_OF_CONTENT
		 */
		public static function get_instance() {
			if ( null === self::$instance ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * XTND_TABLE_OF_CONTENT constructor.
		 */
		private function __construct() {
			$this->init_hooks();
		}

		/**
		 * Initialize WordPress hooks.
		 */
		private function init_hooks() {
			add_action( 'init', array( $this, 'register_blocks' ) );
			add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
			add_action( 'the_post', array( $this, 'conditionally_setup_heading_ids' ) );
		}

		/**
		 * Register block(s) using blocks-manifest.php for performance.
		 *
		 * @return void
		 */
		public function register_blocks() {
			register_block_type( __DIR__ . '/build/xtnd-table-of-content-block' );
		}

		/**
		 * Extract headings by allowed tags from blocks.
		 *
		 * @param array $blocks       The parsed blocks array.
		 * @param array $allowed_tags The allowed heading tags (e.g., array( 'h2', 'h3' )).
		 * @return array              Array of heading strings.
		 */
		public function extract_headings_by_tags( $blocks, $allowed_tags = array( 'h2' ) ) {
			$headings = array();
			foreach ( $blocks as $block ) {
				if ( isset( $block['blockName'] ) && 'core/heading' === $block['blockName'] && ! empty( $block['innerHTML'] ) ) {
					if ( preg_match( '/<\s*(h[1-6])\b/i', $block['innerHTML'], $matches ) ) {
						$tag = strtolower( $matches[1] );
						if ( in_array( $tag, $allowed_tags, true ) ) {
							$text       = wp_strip_all_tags( $block['innerHTML'] );
							$headings[] = $text;
						}
					}
				}
				if ( ! empty( $block['innerBlocks'] ) ) {
					$child_headings = $this->extract_headings_by_tags( $block['innerBlocks'], $allowed_tags );
					$headings       = array_merge( $headings, $child_headings );
				}
			}
			return $headings;
		}

		/**
		 * Add ID to heading block if not present.
		 *
		 * @param string $block_content The block HTML content.
		 * @param array  $block         The block data array.
		 * @return string               The filtered block content.
		 */
		public function auto_add_id_to_heading( $block_content, $block ) {
			if ( ! isset( $block['blockName'] ) || 'core/heading' !== $block['blockName'] ) {
				return $block_content;
			}
			$slug = isset( $block['attrs']['anchor'] ) ? $block['attrs']['anchor'] : '';
			if ( ! $slug && ! empty( $block_content ) ) {
				$text = wp_strip_all_tags( $block_content );
				$slug = sanitize_title( $text );
			}
			if ( strpos( $block_content, 'id="' ) !== false || ! $slug ) {
				return $block_content;
			}
			return preg_replace(
				'/<h([1-6])([^>]*)>/i',
				'<h$1$2 id="' . esc_attr( $slug ) . '">',
				$block_content
			);
		}

		/**
		 * Set up heading IDs if xtnd/xtnd-table-of-content block is present.
		 *
		 * @param WP_Post $post The post object.
		 * @return void
		 */
		public function conditionally_setup_heading_ids( $post ) {
			$blocks = parse_blocks( $post->post_content );
			if ( $this->has_block_type_recursive( $blocks, 'xtnd/xtnd-table-of-content' ) ) {
				add_filter( 'render_block', array( $this, 'auto_add_id_to_heading' ), 10, 2 );
			}
		}

		/**
		 * Recursively check for block type.
		 *
		 * @param array  $blocks       The parsed blocks array.
		 * @param string $target_block The block name to search for.
		 * @return bool               True if found, false otherwise.
		 */
		private function has_block_type_recursive( $blocks, $target_block ) {
			foreach ( $blocks as $block ) {
				if ( isset( $block['blockName'] ) && $block['blockName'] === $target_block ) {
					return true;
				}
				if ( ! empty( $block['innerBlocks'] ) ) {
					if ( $this->has_block_type_recursive( $block['innerBlocks'], $target_block ) ) {
						return true;
					}
				}
			}
			return false;
		}

        /**
         * Get the full URL to a plugin asset.
         *
         * @param string $type       Asset type (e.g., 'css', 'js', 'image').
         * @param string $asset_name Asset file name (e.g., 'style.css', 'app.js').
         * @return string            Full URL to the asset.
         */
        public function asset_url( $type, $asset_name ) {
            return sprintf( '%s/assets/%s/%s', plugin_dir_url( __FILE__ ), $type, $asset_name );
        }

        /**
         * Load plugin text domain for translations.
         *
         * This allows WordPress to load the translation files from the 'languages' directory.
         *
         * @return void
         */
		public function load_textdomain() {
			load_plugin_textdomain( 'xtnd-table-of-content', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
		}
	}
}

// Bootstrap the plugin.
XTND_TABLE_OF_CONTENT::get_instance();
