<?php
/**
 * Template 1 View (with Background Mode + Color Pickers wired to frontend)
 *
 * - Keeps original logic and structure
 * - Applies admin color pickers to: title, subtitle, countdown, nav/menu, toggle label, copyright
 * - Provides plugin-asset fallbacks for images so frontend always looks populated
 *
 * @package MDXCSP_Coming_Soon_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * IMPORTANT:
 * Ensure $template_key is defined BEFORE it's used when fetching settings.
 */
$template_key = 'template-1';

// Fetch template-specific settings.
$all_settings = get_option( 'mdxcsp_template_settings', array() );
$opts         = isset( $all_settings[ $template_key ] ) && is_array( $all_settings[ $template_key ] ) ? $all_settings[ $template_key ] : array();

// Helper to ensure hex color is safe for inline CSS.
$hex = static function( $c ): string {
	$c = (string) $c;
	if ( '' === trim( $c ) ) {
		return '';
	}
	if ( function_exists( 'sanitize_hex_color' ) ) {
		$san = sanitize_hex_color( $c );
		return $san ? $san : '';
	}
	return preg_match( '/^#([A-Fa-f0-9]{3}){1,2}$/', $c ) ? $c : '';
};

// ORIGINAL variables (kept)
$logo_id   = isset( $opts['logo_id'] ) ? absint( $opts['logo_id'] ) : 0;
$bg_id     = isset( $opts['bg_id'] ) ? absint( $opts['bg_id'] ) : 0;
$bg_mode   = isset( $opts['bg_mode'] ) && 'color' === $opts['bg_mode'] ? 'color' : 'image';
$bg_color  = isset( $opts['bg_color'] ) ? (string) $opts['bg_color'] : '';
$title     = isset( $opts['title'] ) ? (string) $opts['title'] : '';
$subtitle  = isset( $opts['subtitle'] ) ? (string) $opts['subtitle'] : '';

$target_ts = isset( $opts['target_timestamp'] ) && (int) $opts['target_timestamp'] > 0
	? (int) $opts['target_timestamp']
	: strtotime( '+10 days 5 hours 30 minutes', current_time( 'timestamp' ) );

$menu_items = array(
	array(
		'label' => isset( $opts['menu_1_label'] ) ? (string) $opts['menu_1_label'] : '',
		'url'   => isset( $opts['menu_1_url'] ) ? (string) $opts['menu_1_url'] : '#',
	),
	array(
		'label' => isset( $opts['menu_2_label'] ) ? (string) $opts['menu_2_label'] : '',
		'url'   => isset( $opts['menu_2_url'] ) ? (string) $opts['menu_2_url'] : '#',
	),
	array(
		'label' => isset( $opts['menu_3_label'] ) ? (string) $opts['menu_3_label'] : '',
		'url'   => isset( $opts['menu_3_url'] ) ? (string) $opts['menu_3_url'] : '#',
	),
);

$menu_toggle_icon_id = isset( $opts['menu_toggle_icon_id'] ) ? absint( $opts['menu_toggle_icon_id'] ) : 0;
$menu_toggle_label   = isset( $opts['menu_toggle_label'] ) ? (string) $opts['menu_toggle_label'] : '';
$menu_toggle_icon    = $menu_toggle_icon_id ? wp_get_attachment_image_url( $menu_toggle_icon_id, 'full' ) : '';

$logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'full' ) : '';
$bg_url   = $bg_id ? wp_get_attachment_image_url( $bg_id, 'full' ) : '';

$social_1_id  = isset( $opts['social_1_id'] ) ? absint( $opts['social_1_id'] ) : 0;
$social_1_url = isset( $opts['social_1_url'] ) ? (string) $opts['social_1_url'] : '#';
$social_1_img = $social_1_id ? wp_get_attachment_image_url( $social_1_id, 'full' ) : '';

$social_2_id  = isset( $opts['social_2_id'] ) ? absint( $opts['social_2_id'] ) : 0;
$social_2_url = isset( $opts['social_2_url'] ) ? (string) $opts['social_2_url'] : '#';
$social_2_img = $social_2_id ? wp_get_attachment_image_url( $social_2_id, 'full' ) : '';

// ADDED: text and nav colors from admin + safe fallbacks
$title_color             = isset( $opts['title_color'] ) ? $hex( $opts['title_color'] ) : '';
$subtitle_color          = isset( $opts['subtitle_color'] ) ? $hex( $opts['subtitle_color'] ) : '';
$countdown_color         = isset( $opts['countdown_color'] ) ? $hex( $opts['countdown_color'] ) : '';
$nav_color               = isset( $opts['nav_color'] ) ? $hex( $opts['nav_color'] ) : '';
$menu_1_color            = isset( $opts['menu_1_color'] ) ? $hex( $opts['menu_1_color'] ) : '';
$menu_2_color            = isset( $opts['menu_2_color'] ) ? $hex( $opts['menu_2_color'] ) : '';
$menu_3_color            = isset( $opts['menu_3_color'] ) ? $hex( $opts['menu_3_color'] ) : '';
$menu_toggle_label_color = isset( $opts['menu_toggle_label_color'] ) ? $hex( $opts['menu_toggle_label_color'] ) : '';
$copyright_color         = isset( $opts['copyright_color'] ) ? $hex( $opts['copyright_color'] ) : '';

$title_color             = $title_color ?: '#ffffff';
$subtitle_color          = $subtitle_color ?: '#E5E7EB';
$countdown_color         = $countdown_color ?: '#ffffff';
$nav_color               = $nav_color ?: '#ffffff';
$menu_1_color            = $menu_1_color ?: $nav_color;
$menu_2_color            = $menu_2_color ?: $nav_color;
$menu_3_color            = $menu_3_color ?: $nav_color;
$menu_toggle_label_color = $menu_toggle_label_color ?: $nav_color;
$copyright_color         = $copyright_color ?: '#9CA3AF';

// ADDED: plugin-asset fallbacks so frontend always shows images
$asset_logo = MDXCSP_PLUGIN_URL . 'assets/images/logo.png';
$asset_bg   = MDXCSP_PLUGIN_URL . 'assets/images/bg-1.png';
$asset_tog  = MDXCSP_PLUGIN_URL . 'assets/images/toggle.png';
$asset_s1   = MDXCSP_PLUGIN_URL . 'assets/images/social-1.png';
$asset_s2   = MDXCSP_PLUGIN_URL . 'assets/images/social-2.png';

if ( ! $logo_url ) {
	$logo_url = $asset_logo;
}
if ( 'image' === $bg_mode && ! $bg_url ) {
	$bg_url = $asset_bg;
}
if ( '' === trim( $menu_toggle_label ) && '' === (string) $menu_toggle_icon ) {
	$menu_toggle_label = '☰ ' . __( 'Menu', 'xpertcodes-coming-soon-maintenance' );
	$menu_toggle_icon  = $asset_tog;
}
if ( ! $social_1_img ) {
	$social_1_img = $asset_s1;
}
if ( ! $social_2_img ) {
	$social_2_img = $asset_s2;
}

// ORIGINAL: detect menu presence
$has_menu_items = array_filter(
	wp_list_pluck( $menu_items, 'label' ),
	static function( $l ) {
		return '' !== trim( (string) $l );
	}
);

// ORIGINAL: inline background style (kept)
$inline_style = '';
if ( 'color' === $bg_mode && $bg_color ) {
	$inline_style = 'background: ' . esc_attr( $bg_color ) . ';';
} elseif ( 'image' === $bg_mode && $bg_url ) {
	$inline_style = "background: url('" . esc_url( $bg_url ) . "') no-repeat center center / cover;";
}

/**
 * Proper enqueue + inline CSS for color styles.
 *
 * Note:
 * - Use plugin_dir_url( __FILE__ ) relative to this template file to build CSS URL.
 * - Use __DIR__ for filemtime() to avoid relying on MDXCSP_PLUGIN_PATH constant.
 */
add_action( 'wp_enqueue_scripts', function() use ( $title_color, $subtitle_color, $countdown_color, $nav_color, $menu_toggle_label_color, $copyright_color ) {
	// URL to the template-1 css file (relative to this template directory)
	$css_url = plugin_dir_url( __FILE__ ) . 'assets/template-1.css';

	// Local filesystem path to the CSS for filemtime (use __DIR__)
	$css_path = rtrim( __DIR__, DIRECTORY_SEPARATOR ) . DIRECTORY_SEPARATOR . 'assets' . DIRECTORY_SEPARATOR . 'template-1.css';

	$ver = is_file( $css_path ) ? filemtime( $css_path ) : '1.0.0';

	wp_enqueue_style(
		'mdxcsp-template-1',
		$css_url,
		array(),
		$ver
	);

	$inline_css = "
		body.mdxcsp-coming-soon.template-1 .mdxcsp-title { color: {$title_color} !important; }
		body.mdxcsp-coming-soon.template-1 .mdxcsp-subtitle { color: {$subtitle_color} !important; }
		body.mdxcsp-coming-soon.template-1 .mdxcsp-counter { color: {$countdown_color} !important; }
		body.mdxcsp-coming-soon.template-1 .mdxcsp-top-bar,
		body.mdxcsp-coming-soon.template-1 .mdxcsp-top-bar a { color: {$nav_color} !important; }
		body.mdxcsp-coming-soon.template-1 .mdxcsp-menu-toggle { color: {$menu_toggle_label_color} !important; }
		body.mdxcsp-coming-soon.template-1 .mdxcsp-footer-left { color: {$copyright_color} !important; }
	";

	wp_add_inline_style( 'mdxcsp-template-1', $inline_css );
}, 20 );

// ADDED: title/subtitle sensible defaults
if ( '' === trim( $title ) ) {
	$title = __( 'We\'re launching soon', 'xpertcodes-coming-soon-maintenance' );
}
if ( '' === trim( $subtitle ) ) {
	$subtitle = __( 'Our site is getting ready. Stay tuned for something amazing!', 'xpertcodes-coming-soon-maintenance' );
}
?>
<!DOCTYPE html>
<html <?php language_attributes(); ?>>
<head>
	<meta charset="<?php bloginfo( 'charset' ); ?>">
	<meta name="viewport" content="width=device-width, initial-scale=1.0">
	<meta name="robots" content="noindex, nofollow, noarchive">
	<title><?php echo esc_html( '' !== $title ? $title : __( 'Coming Soon', 'xpertcodes-coming-soon-maintenance' ) ); ?></title>
	<?php wp_head(); ?>
</head>
<body <?php body_class( 'mdxcsp-coming-soon template-1' ); ?><?php echo $inline_style ? ' style="' . wp_kses( $inline_style, array() ) . '"' : ''; ?>>

	<?php
	if ( function_exists( 'wp_body_open' ) ) {
		wp_body_open();
	}
	?>

	<div class="mdxcsp-coming-container">
		<header class="mdxcsp-top-bar">
			<?php if ( $logo_url ) : ?>
				<div class="mdxcsp-logo">
					<img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php esc_attr_e( 'Logo', 'xpertcodes-coming-soon-maintenance' ); ?>">
				</div>
			<?php endif; ?>

			<?php if ( $has_menu_items && ( $menu_toggle_icon || '' !== trim( $menu_toggle_label ) ) ) : ?>
				<button type="button"
					class="mdxcsp-menu-toggle"
					aria-label="<?php esc_attr_e( 'Toggle Menu', 'xpertcodes-coming-soon-maintenance' ); ?>"
					aria-expanded="false"
					aria-controls="mdxcsp-dropdown-menu">
					<?php if ( $menu_toggle_icon ) : ?>
						<img src="<?php echo esc_url( $menu_toggle_icon ); ?>" alt="<?php esc_attr_e( 'Menu', 'xpertcodes-coming-soon-maintenance' ); ?>" style="width:28px;height:auto;">
					<?php endif; ?>
					<span><?php echo esc_html( $menu_toggle_label ); ?></span>
				</button>
			<?php endif; ?>

			<?php if ( $has_menu_items ) : ?>
				<nav id="mdxcsp-dropdown-menu" class="mdxcsp-dropdown-menu" aria-hidden="true">
					<ul>
						<?php
						$colors = array( $menu_1_color, $menu_2_color, $menu_3_color );
						foreach ( $menu_items as $idx => $item ) :
							$label = trim( (string) $item['label'] );
							if ( '' === $label ) {
								continue;
							}
							$link_color = $hex( $colors[ $idx ] ?? '' ) ?: $nav_color;
							?>
							<li><a href="<?php echo esc_url( (string) $item['url'] ); ?>" style="color:<?php echo esc_attr( $link_color ); ?>"><?php echo esc_html( $label ); ?></a></li>
						<?php endforeach; ?>
					</ul>
				</nav>
			<?php endif; ?>
		</header>

		<main class="mdxcsp-main-content">
			<?php if ( '' !== trim( $title ) ) : ?>
				<h1 class="mdxcsp-title"><?php echo esc_html( $title ); ?></h1>
			<?php endif; ?>

			<div class="mdxcsp-counter" data-target-date="<?php echo esc_attr( date_i18n( 'c', $target_ts ) ); ?>">
				<div class="mdxcsp-counter-box"><span class="days">00</span><small><?php esc_html_e( 'Days', 'xpertcodes-coming-soon-maintenance' ); ?></small></div>
				<div class="mdxcsp-counter-box"><span class="hours">00</span><small><?php esc_html_e( 'Hours', 'xpertcodes-coming-soon-maintenance' ); ?></small></div>
				<div class="mdxcsp-counter-box"><span class="minutes">00</span><small><?php esc_html_e( 'Minutes', 'xpertcodes-coming-soon-maintenance' ); ?></small></div>
				<div class="mdxcsp-counter-box"><span class="seconds">00</span><small><?php esc_html_e( 'Seconds', 'xpertcodes-coming-soon-maintenance' ); ?></small></div>
			</div>

			<?php if ( '' !== trim( $subtitle ) ) : ?>
				<p class="mdxcsp-subtitle"><?php echo esc_html( $subtitle ); ?></p>
			<?php endif; ?>
		</main>

		<footer class="mdxcsp-bottom-bar">
			<div class="mdxcsp-footer-left">
				<?php if ( '' !== trim( (string) ( $opts['copyright_text'] ?? '' ) ) ) : ?>
					<?php
					$year = date_i18n( 'Y' );
					echo esc_html( sprintf( __( '© %1$s %2$s', 'xpertcodes-coming-soon-maintenance' ), $year, (string) $opts['copyright_text'] ) );
					?>
				<?php else : ?>
					<?php
					$year = date_i18n( 'Y' );
					echo esc_html( sprintf( __( '© %1$s Your Company. All rights reserved.', 'xpertcodes-coming-soon-maintenance' ), $year ) );
					?>
				<?php endif; ?>
			</div>
			<div class="mdxcsp-footer-right">
				<?php if ( $social_1_img ) : ?>
					<a href="<?php echo esc_url( $social_1_url ); ?>"><img src="<?php echo esc_url( $social_1_img ); ?>" alt="<?php esc_attr_e( 'Social link 1', 'xpertcodes-coming-soon-maintenance' ); ?>"></a>
				<?php endif; ?>
				<?php if ( $social_2_img ) : ?>
					<a href="<?php echo esc_url( $social_2_url ); ?>"><img src="<?php echo esc_url( $social_2_img ); ?>" alt="<?php esc_attr_e( 'Social link 2', 'xpertcodes-coming-soon-maintenance' ); ?>"></a>
				<?php endif; ?>
			</div>
		</footer>
	</div>

	<?php wp_footer(); ?>
</body>
</html>
