<?php
/**
 * Settings Page View (WP-standard UX, progressive menu items, sane defaults).
 *
 * - Shows only one default "Home" menu item initially in admin.
 * - Provides an "Add menu item" button to add as many items as the user wants.
 * - Items 2..10 are present in markup (to preserve data) but hidden by default.
 * - Social, branding, countdown, and color pickers remain unchanged and WP-standard.
 *
 * @package MDXCSP_Coming_Soon_Pro
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

$plugin_enabled    = get_option( 'mdxcsp_enable_coming_soon', '0' );
$selected_template = sanitize_key( (string) get_option( 'mdxcsp_selected_template', 'template-1' ) );
$template_settings = get_option( 'mdxcsp_template_settings', array() );

$templates = array(
	'template-1' => array(
		'title' => __( 'Template 1', 'xpertcodes-coming-soon-maintenance' ),
		'image' => MDXCSP_PLUGIN_URL . 'assets/images/template-1-view.png',
	),
	'template-2' => array(
		'title' => __( 'Template 2', 'xpertcodes-coming-soon-maintenance' ),
		'image' => MDXCSP_PLUGIN_URL . 'assets/images/template-2-view.png',
	),
);

/**
 * Format datetime for local input.
 *
 * @param int $timestamp Unix timestamp.
 * @return string
 */
function mdxcsp_format_datetime_local( $timestamp ): string {
	if ( ! $timestamp ) {
		return '';
	}
	return function_exists( 'wp_date' )
		? wp_date( 'Y-m-d\TH:i', (int) $timestamp )
		: date_i18n( 'Y-m-d\TH:i', (int) $timestamp );
}

// Default images paths.
$default_logo_url    = MDXCSP_PLUGIN_URL . 'assets/images/logo.png';
$default_toggle_url  = MDXCSP_PLUGIN_URL . 'assets/images/toggle.png';
$default_social1_url = MDXCSP_PLUGIN_URL . 'assets/images/social-1.png';
$default_social2_url = MDXCSP_PLUGIN_URL . 'assets/images/social-2.png';

// Template-specific default background images.
$default_bg_images = array(
	'template-1' => MDXCSP_PLUGIN_URL . 'assets/images/bg-1.jpeg',
	'template-2' => MDXCSP_PLUGIN_URL . 'assets/images/bg-2.jpg',
);
?>
<div class="wrap">
	<h1><?php echo esc_html__( 'Coming Soon Pro Settings', 'xpertcodes-coming-soon-maintenance' ); ?></h1>
	<?php settings_errors(); ?>

	<div class="mdxcsp-settings-container">
		<form method="post" action="options.php" id="mdxcsp-settings-form">
			<?php settings_fields( 'mdxcsp_settings_group' ); ?>

			<!-- Status Row -->
			<div class="mdxcsp-form-row mdxcsp-row-status">
				<div class="mdxcsp-status-left">
					<span class="mdxcsp-status-pill <?php echo '1' === (string) $plugin_enabled ? 'is-on' : 'is-off'; ?>">
						<?php echo '1' === (string) $plugin_enabled ? esc_html__( 'Active', 'xpertcodes-coming-soon-maintenance' ) : esc_html__( 'Inactive', 'xpertcodes-coming-soon-maintenance' ); ?>
					</span>
					<span class="mdxcsp-status-text"><?php echo esc_html__( 'Enable Coming Soon Mode', 'xpertcodes-coming-soon-maintenance' ); ?></span>
				</div>
				<div class="mdxcsp-status-right">
					<input type="hidden" name="mdxcsp_enable_coming_soon" value="0" />
					<label class="mdxcsp-toggle-switch">
						<input type="checkbox" name="mdxcsp_enable_coming_soon" id="mdxcsp_enable_coming_soon" value="1" <?php checked( (string) $plugin_enabled, '1' ); ?> />
						<span class="mdxcsp-slider"></span>
					</label>
				</div>
			</div>

			<!-- Template selector -->
			<div class="mdxcsp-form-row mdxcsp-row-title">
				<label class="mdxcsp-required-label"><?php echo esc_html__( 'Select a Template', 'xpertcodes-coming-soon-maintenance' ); ?></label>
				<p class="description"><?php echo esc_html__( 'Pick a layout. You will get tailored options below.', 'xpertcodes-coming-soon-maintenance' ); ?></p>
			</div>

			<div class="mdxcsp-templates-grid" id="mdxcsp-templates-grid">
				<?php foreach ( $templates as $template_key => $template_data ) : ?>
					<label class="mdxcsp-template-card <?php echo $selected_template === $template_key ? 'is-selected' : ''; ?>">
						<input type="radio" name="mdxcsp_selected_template" value="<?php echo esc_attr( $template_key ); ?>" <?php checked( $selected_template, $template_key ); ?> />
						<span class="mdxcsp-badge"><?php echo esc_html__( 'Selected', 'xpertcodes-coming-soon-maintenance' ); ?></span>
						<img src="<?php echo esc_url( $template_data['image'] ); ?>" alt="<?php echo esc_attr( $template_data['title'] ); ?>">
						<span class="mdxcsp-template-title"><?php echo esc_html( $template_data['title'] ); ?></span>
					</label>
				<?php endforeach; ?>
			</div>

			<?php if ( count( $templates ) > 3 ) : ?>
				<button type="button" class="button button-secondary" id="mdxcsp-load-more">
					<?php echo esc_html__( 'Load More', 'xpertcodes-coming-soon-maintenance' ); ?>
				</button>
			<?php endif; ?>

			<hr class="mdxcsp-divider" />

			<h2 class="mdxcsp-section-title"><?php echo esc_html__( 'Template Options', 'xpertcodes-coming-soon-maintenance' ); ?></h2>
			<p class="mdxcsp-section-desc"><?php echo esc_html__( 'Configure your coming soon page. Fields marked with * are required for optimal display.', 'xpertcodes-coming-soon-maintenance' ); ?></p>

			<div class="mdxcsp-template-options-wrap" id="mdxcsp-template-options-wrap">
				<?php foreach ( $templates as $template_key => $template_data ) : ?>
					<?php
					$tkey = sanitize_key( $template_key );
					$opts = isset( $template_settings[ $tkey ] ) && is_array( $template_settings[ $tkey ] ) ? $template_settings[ $tkey ] : array();

					// Get template-specific default background image.
					$default_bg_url = $default_bg_images[ $tkey ] ?? $default_bg_images['template-1'];

					$logo_id   = isset( $opts['logo_id'] ) ? absint( $opts['logo_id'] ) : 0;
					$bg_id     = isset( $opts['bg_id'] ) ? absint( $opts['bg_id'] ) : 0;
					$bg_mode   = isset( $opts['bg_mode'] ) && in_array( $opts['bg_mode'], array( 'image', 'color' ), true ) ? $opts['bg_mode'] : 'image';
					$bg_color  = isset( $opts['bg_color'] ) ? (string) $opts['bg_color'] : '';

					$title           = isset( $opts['title'] ) ? (string) $opts['title'] : '';
					$title_color     = isset( $opts['title_color'] ) ? (string) $opts['title_color'] : '';
					$subtitle        = isset( $opts['subtitle'] ) ? (string) $opts['subtitle'] : '';
					$subtitle_color  = isset( $opts['subtitle_color'] ) ? (string) $opts['subtitle_color'] : '';
					$target_ts       = isset( $opts['target_timestamp'] ) ? (int) $opts['target_timestamp'] : 0;
					$countdown_color = isset( $opts['countdown_color'] ) ? (string) $opts['countdown_color'] : '';

					$logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'full' ) : '';
					$bg_url   = $bg_id ? wp_get_attachment_image_url( $bg_id, 'full' ) : '';

					$menu_1_label = isset( $opts['menu_1_label'] ) ? (string) $opts['menu_1_label'] : '';
					$menu_1_url   = isset( $opts['menu_1_url'] ) ? (string) $opts['menu_1_url'] : '#';
					$menu_1_color = isset( $opts['menu_1_color'] ) ? (string) $opts['menu_1_color'] : '';

					$menu_2_label = isset( $opts['menu_2_label'] ) ? (string) $opts['menu_2_label'] : '';
					$menu_2_url   = isset( $opts['menu_2_url'] ) ? (string) $opts['menu_2_url'] : '#';
					$menu_2_color = isset( $opts['menu_2_color'] ) ? (string) $opts['menu_2_color'] : '';

					$menu_3_label = isset( $opts['menu_3_label'] ) ? (string) $opts['menu_3_label'] : '';
					$menu_3_url   = isset( $opts['menu_3_url'] ) ? (string) $opts['menu_3_url'] : '#';
					$menu_3_color = isset( $opts['menu_3_color'] ) ? (string) $opts['menu_3_color'] : '';

					$nav_color = isset( $opts['nav_color'] ) ? (string) $opts['nav_color'] : '';

					$menu_toggle_icon_id     = isset( $opts['menu_toggle_icon_id'] ) ? absint( $opts['menu_toggle_icon_id'] ) : 0;
					$menu_toggle_label       = isset( $opts['menu_toggle_label'] ) ? (string) $opts['menu_toggle_label'] : '';
					$menu_toggle_label_color = isset( $opts['menu_toggle_label_color'] ) ? (string) $opts['menu_toggle_label_color'] : '';
					$menu_toggle_icon        = $menu_toggle_icon_id ? wp_get_attachment_image_url( $menu_toggle_icon_id, 'full' ) : '';

					$social_1_id    = isset( $opts['social_1_id'] ) ? absint( $opts['social_1_id'] ) : 0;
					$social_1_url   = isset( $opts['social_1_url'] ) ? (string) $opts['social_1_url'] : '#';
					$social_2_id    = isset( $opts['social_2_id'] ) ? absint( $opts['social_2_id'] ) : 0;
					$social_2_url   = isset( $opts['social_2_url'] ) ? (string) $opts['social_2_url'] : '#';
					$social_1_img   = $social_1_id ? wp_get_attachment_image_url( $social_1_id, 'full' ) : '';
					$social_2_img   = $social_2_id ? wp_get_attachment_image_url( $social_2_id, 'full' ) : '';

					$copyright_text  = isset( $opts['copyright_text'] ) ? (string) $opts['copyright_text'] : '';
					$copyright_color = isset( $opts['copyright_color'] ) ? (string) $opts['copyright_color'] : '';

					// Admin-friendly defaults (prefill for a good first impression).
					if ( 0 === $logo_id ) {
						$default_logo_id = absint( get_theme_mod( 'custom_logo' ) );
						if ( $default_logo_id ) {
							$logo_id  = $default_logo_id;
							$logo_url = wp_get_attachment_image_url( $logo_id, 'full' );
						}
						if ( empty( $logo_url ) ) {
							$logo_url = $default_logo_url;
						}
					}
					if ( '' === $title ) {
						$title = __( 'We are launching soon', 'xpertcodes-coming-soon-maintenance' );
					}
					if ( '' === $title_color ) {
						$title_color = '#ffffff';
					}
					if ( '' === $subtitle ) {
						$subtitle = __( 'Our site is getting ready. Stay tuned for something amazing!', 'xpertcodes-coming-soon-maintenance' );
					}
					if ( '' === $subtitle_color ) {
						$subtitle_color = '#E5E7EB';
					}
					if ( 0 === $target_ts ) {
						$tz  = function_exists( 'wp_timezone' ) ? wp_timezone() : new DateTimeZone( get_option( 'timezone_string' ) ?: 'UTC' );
						$now = new DateTime( 'now', $tz );
						$now->setTime( 0, 0 );
						$now->add( new DateInterval( 'P30D' ) ); // +30 days at midnight.
						$target_ts = $now->getTimestamp();
					}
					if ( '' === $countdown_color ) {
						$countdown_color = '#ffffff';
					}
					// IMPORTANT: Only prefill Menu 1; do NOT prefill Menu 2/3 so only the first appears by default.
					if ( '' === $menu_1_label ) {
						$menu_1_label = __( 'Home', 'xpertcodes-coming-soon-maintenance' );
						$menu_1_url   = '#home';
					}
					if ( '' === $nav_color ) {
						$nav_color = '#ffffff';
					}
					if ( '' === $menu_1_color ) {
						$menu_1_color = $nav_color;
					}
					if ( '' === $menu_2_color ) {
						$menu_2_color = $nav_color;
					}
					if ( '' === $menu_3_color ) {
						$menu_3_color = $nav_color;
					}
					if ( '' === $menu_toggle_label ) {
						$menu_toggle_label = '☰ ' . __( 'Menu', 'xpertcodes-coming-soon-maintenance' );
					}
					if ( '' === $menu_toggle_label_color ) {
						$menu_toggle_label_color = $nav_color;
					}
					if ( '' === $bg_color ) {
						$bg_color = '#111827';
					}
					if ( 0 === $menu_toggle_icon_id && empty( $menu_toggle_icon ) ) {
						$default_site_icon_id = absint( get_option( 'site_icon' ) );
						if ( $default_site_icon_id ) {
							$menu_toggle_icon_id = $default_site_icon_id;
							$menu_toggle_icon    = wp_get_attachment_image_url( $menu_toggle_icon_id, 'full' );
						}
						if ( empty( $menu_toggle_icon ) ) {
							$menu_toggle_icon = $default_toggle_url;
						}
					}
					if ( 0 === $social_1_id && empty( $social_1_img ) ) {
						$default_site_icon_id = absint( get_option( 'site_icon' ) );
						if ( $default_site_icon_id ) {
							$social_1_id  = $default_site_icon_id;
							$social_1_img = wp_get_attachment_image_url( $social_1_id, 'full' );
						}
						if ( empty( $social_1_img ) ) {
							$social_1_img = $default_social1_url;
						}
					}
					if ( '' === $social_1_url ) {
						$social_1_url = '#social';
					}
					if ( 0 === $social_2_id && empty( $social_2_img ) ) {
						$default_site_icon_id = absint( get_option( 'site_icon' ) );
						if ( $default_site_icon_id ) {
							$social_2_id  = $default_site_icon_id;
							$social_2_img = wp_get_attachment_image_url( $social_2_id, 'full' );
						}
						if ( empty( $social_2_img ) ) {
							$social_2_img = $default_social2_url;
						}
					}
					if ( '' === $social_2_url ) {
						$social_2_url = '#social-2';
					}
					if ( '' === $copyright_text ) {
						$copyright_text = __( 'Your Company. All rights reserved.', 'xpertcodes-coming-soon-maintenance' );
					}
					if ( '' === $copyright_color ) {
						$copyright_color = '#9CA3AF';
					}

					// Extra menu items (4..10): keep values, but HIDE them by default (user adds via button).
					$extra_menu = array();
					for ( $i = 4; $i <= 10; $i++ ) {
						$li = isset( $opts[ "menu_{$i}_label" ] ) ? (string) $opts[ "menu_{$i}_label" ] : '';
						$lu = isset( $opts[ "menu_{$i}_url" ] ) ? (string) $opts[ "menu_{$i}_url" ] : '#';
						$lc = isset( $opts[ "menu_{$i}_color" ] ) ? (string) $opts[ "menu_{$i}_color" ] : '';
						if ( '' === $lc ) {
							$lc = $nav_color;
						}
						$extra_menu[ $i ] = array(
							'label' => $li,
							'url'   => $lu,
							'color' => $lc,
						);
					}
					?>
					<div class="mdxcsp-template-options" data-template="<?php echo esc_attr( $tkey ); ?>" style="display: <?php echo $selected_template === $tkey ? 'block' : 'none'; ?>;">
						<h3 class="mdxcsp-card-title">
							<?php
							/* translators: %s: Template title */
							echo esc_html( sprintf( __( '%s Options', 'xpertcodes-coming-soon-maintenance' ), $template_data['title'] ) );
							?>
						</h3>

						<!-- Branding -->
						<details class="mdxcsp-acc" open>
							<summary><?php echo esc_html__( 'Branding', 'xpertcodes-coming-soon-maintenance' ); ?></summary>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label"><?php echo esc_html__( 'Logo Image', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<div class="mdxcsp-image-field">
												<div class="mdxcsp-image-preview-wrap">
													<img class="mdxcsp-image-preview" data-target="logo_id" src="<?php echo esc_url( $logo_url ); ?>" alt="" style="<?php echo $logo_url ? '' : 'display:none;'; ?>max-width:150px;height:auto;">
												</div>
												<input type="hidden" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][logo_id]" value="<?php echo esc_attr( $logo_id ); ?>">
												<button type="button" class="button button-primary mdxcsp-upload" data-target="logo_id"><?php echo esc_html__( 'Upload/Select', 'xpertcodes-coming-soon-maintenance' ); ?></button>
												<button type="button" class="button-link-delete mdxcsp-remove" data-target="logo_id" <?php disabled( $logo_id, 0 ); ?>><?php echo esc_html__( 'Remove', 'xpertcodes-coming-soon-maintenance' ); ?></button>
												<p class="description"><?php echo esc_html__( 'Recommended for brand identity', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>

									<tr>
										<th scope="row"><label class="mdxcsp-required-label"><?php echo esc_html__( 'Background', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<fieldset class="mdxcsp-bg-mode">
												<label style="margin-right:12px;">
													<input type="radio" data-role="bg-mode" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][bg_mode]" value="image" <?php checked( $bg_mode, 'image' ); ?> />
													<?php echo esc_html__( 'Image', 'xpertcodes-coming-soon-maintenance' ); ?>
												</label>
												<label>
													<input type="radio" data-role="bg-mode" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][bg_mode]" value="color" <?php checked( $bg_mode, 'color' ); ?> />
													<?php echo esc_html__( 'Color', 'xpertcodes-coming-soon-maintenance' ); ?>
												</label>
											</fieldset>

											<div class="mdxcsp-bg-image-controls" aria-hidden="<?php echo 'image' === $bg_mode ? 'false' : 'true'; ?>" style="margin-top:8px;<?php echo 'image' === $bg_mode ? '' : 'display:none;'; ?>">
												<div class="mdxcsp-image-field">
													<div class="mdxcsp-image-preview-wrap">
														<img class="mdxcsp-image-preview" data-target="bg_id" src="<?php echo esc_url( $bg_url ?: $default_bg_url ); ?>" alt="" style="<?php echo ( $bg_url || $default_bg_url ) ? '' : 'display:none;'; ?>max-width:220px;height:auto;">
													</div>
													<input type="hidden" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][bg_id]" value="<?php echo esc_attr( $bg_id ); ?>" <?php echo 'image' === $bg_mode ? '' : 'disabled'; ?>>
													<button type="button" class="button button-primary mdxcsp-upload" data-target="bg_id" <?php echo 'image' === $bg_mode ? '' : 'disabled'; ?>><?php echo esc_html__( 'Upload/Select', 'xpertcodes-coming-soon-maintenance' ); ?></button>
													<button type="button" class="button-link-delete mdxcsp-remove" data-target="bg_id" <?php echo 'image' === $bg_mode ? '' : 'disabled'; ?> <?php disabled( $bg_id, 0 ); ?>><?php echo esc_html__( 'Remove', 'xpertcodes-coming-soon-maintenance' ); ?></button>
												</div>
												<p class="description"><?php echo esc_html__( 'Choose a background image', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>

											<div class="mdxcsp-bg-color-controls" aria-hidden="<?php echo 'color' === $bg_mode ? 'false' : 'true'; ?>" style="margin-top:8px;<?php echo 'color' === $bg_mode ? '' : 'display:none;'; ?>">
												<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="#111827" id="mdxcsp-bg-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][bg_color]" value="<?php echo esc_attr( $bg_color ?: '#111827' ); ?>" <?php echo 'color' === $bg_mode ? '' : 'disabled'; ?> />
												<p class="description"><?php echo esc_html__( 'Choose a background color', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>
								</tbody>
							</table>
						</details>

						<!-- Text Content -->
						<details class="mdxcsp-acc" open>
							<summary><?php echo esc_html__( 'Text Content', 'xpertcodes-coming-soon-maintenance' ); ?></summary>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label" for="mdxcsp-title-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Title', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<div class="mdxcsp-field-wrapper">
												<div class="mdxcsp-inline-field">
													<input type="text" class="regular-text" id="mdxcsp-title-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][title]" value="<?php echo esc_attr( $title ); ?>">
													<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="#ffffff" id="mdxcsp-title-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][title_color]" value="<?php echo esc_attr( $title_color ?: '#ffffff' ); ?>" />
												</div>
												<p class="description"><?php echo esc_html__( 'Main heading text for your page', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label" for="mdxcsp-subtitle-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Subtitle', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<div class="mdxcsp-field-wrapper">
												<div class="mdxcsp-inline-field">
													<textarea class="large-text" rows="3" id="mdxcsp-subtitle-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][subtitle]"><?php echo esc_textarea( $subtitle ); ?></textarea>
													<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="#E5E7EB" id="mdxcsp-subtitle-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][subtitle_color]" value="<?php echo esc_attr( $subtitle_color ?: '#E5E7EB' ); ?>" />
												</div>
												<p class="description"><?php echo esc_html__( 'Supporting text below the title', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>
								</tbody>
							</table>
						</details>

						<!-- Countdown -->
						<details class="mdxcsp-acc" open>
							<summary><?php echo esc_html__( 'Countdown', 'xpertcodes-coming-soon-maintenance' ); ?></summary>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label" for="mdxcsp-datetime-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Target Date & Time', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<div class="mdxcsp-field-wrapper">
												<div class="mdxcsp-inline-field">
													<input type="datetime-local" id="mdxcsp-datetime-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][target_datetime]" value="<?php echo esc_attr( mdxcsp_format_datetime_local( $target_ts ) ); ?>">
													<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="#ffffff" id="mdxcsp-countdown-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][countdown_color]" value="<?php echo esc_attr( $countdown_color ?: '#ffffff' ); ?>" />
												</div>
												<p class="description"><?php echo esc_html__( 'When your site will launch', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>
								</tbody>
							</table>
						</details>

						<!-- Menu Items -->
						<details class="mdxcsp-acc" open>
							<summary><?php echo esc_html__( 'Menu Items', 'xpertcodes-coming-soon-maintenance' ); ?></summary>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label"><?php echo esc_html__( 'Items', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<fieldset class="mdxcsp-menu-items-container">
												<!-- Item 1 (Home — always visible) -->
												<div class="mdxcsp-menu-item" data-menu-index="1">
													<div class="mdxcsp-menu-item-header">
														<label class="mdxcsp-required-label"><?php echo esc_html__( 'Menu Item 1', 'xpertcodes-coming-soon-maintenance' ); ?></label>
													</div>
													<div class="mdxcsp-menu-main-tnt">
													<div class="mdxcsp-inline-field">
														<label class="mdxcsp-url-label mdxcsp-required-label"><?php echo esc_html__( 'Name', 'xpertcodes-coming-soon-maintenance' ); ?></label>
														<input type="text" class="regular-text" placeholder="<?php echo esc_attr__( 'Menu Name', 'xpertcodes-coming-soon-maintenance' ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_1_label]" value="<?php echo esc_attr( $menu_1_label ); ?>">
														<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="<?php echo esc_attr( $nav_color ?: '#ffffff' ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_1_color]" value="<?php echo esc_attr( $menu_1_color ?: ( $nav_color ?: '#ffffff' ) ); ?>">
													</div>
													<div class="menu-url-input">
													<label class="mdxcsp-url-label mdxcsp-required-label"><?php echo esc_html__( 'URL', 'xpertcodes-coming-soon-maintenance' ); ?></label>
													<input type="text" class="regular-text code" placeholder="https://example.com or #section" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_1_url]" value="<?php echo esc_attr( $menu_1_url ); ?>">
													</div>
													</div>
												</div>

												<!-- Hidden Menu Items 2-10 -->
												<?php for ( $i = 2; $i <= 10; $i++ ) : ?>
													<?php
													$menu_label = isset( $opts[ "menu_{$i}_label" ] ) ? (string) $opts[ "menu_{$i}_label" ] : '';
													$menu_url   = isset( $opts[ "menu_{$i}_url" ] ) ? (string) $opts[ "menu_{$i}_url" ] : '#';
													$menu_color = isset( $opts[ "menu_{$i}_color" ] ) ? (string) $opts[ "menu_{$i}_color" ] : $nav_color;
													?>
													<div class="mdxcsp-menu-item mdxcsp-menu-item-hidden" data-menu-index="<?php echo esc_attr( $i ); ?>" style="display: none;">
														<div class="mdxcsp-menu-item-header">
															<label class="mdxcsp-required-label">
																<?php
																/* translators: %d: Menu item number */
																echo esc_html( sprintf( __( 'Menu Item %d', 'xpertcodes-coming-soon-maintenance' ), $i ) );
																?>
															</label>
															<button type="button" class="button-link mdxcsp-remove-menu-item"><?php echo esc_html__( 'Remove', 'xpertcodes-coming-soon-maintenance' ); ?></button>
														</div>
														<div class="mdxcsp-inline-field">
															<label class="mdxcsp-url-label mdxcsp-required-label"><?php echo esc_html__( 'Name', 'xpertcodes-coming-soon-maintenance' ); ?></label>
															<input type="text" class="regular-text" placeholder="<?php echo esc_attr__( 'Menu Name', 'xpertcodes-coming-soon-maintenance' ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_<?php echo esc_attr( $i ); ?>_label]" value="<?php echo esc_attr( $menu_label ); ?>">
															<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="<?php echo esc_attr( $nav_color ?: '#ffffff' ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_<?php echo esc_attr( $i ); ?>_color]" value="<?php echo esc_attr( $menu_color ); ?>">
														</div>
														<label class="mdxcsp-url-label mdxcsp-required-label"><?php echo esc_html__( 'URL', 'xpertcodes-coming-soon-maintenance' ); ?></label>
														<input type="text" class="regular-text code" placeholder="https://example.com or #section" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_<?php echo esc_attr( $i ); ?>_url]" value="<?php echo esc_attr( $menu_url ); ?>">
													</div>
												<?php endfor; ?>

												<!-- Default nav color -->
												<div class="mdxcsp-nav-color-section">
													<label class="mdxcsp-required-label" for="mdxcsp-nav-color-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Default Menu Color', 'xpertcodes-coming-soon-maintenance' ); ?></label><br>
													<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="#ffffff" id="mdxcsp-nav-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][nav_color]" value="<?php echo esc_attr( $nav_color ?: '#ffffff' ); ?>">
												</div>

												<!-- Add Button -->
												<button type="button" class="button mdxcsp-add-btn mdxcsp-add-menu-btn">
													<span class="mdxcsp-add-btn-icon">+</span> <?php echo esc_html__( 'Add menu item', 'xpertcodes-coming-soon-maintenance' ); ?>
												</button>

												<p class="description"><?php echo esc_html__( 'Add navigation items to your coming soon page', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</fieldset>
										</td>
									</tr>
								</tbody>
							</table>
						</details>

						<!-- Menu Toggle -->
						<details class="mdxcsp-acc" open>
							<summary><?php echo esc_html__( 'Menu Toggle', 'xpertcodes-coming-soon-maintenance' ); ?></summary>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label"><?php echo esc_html__( 'Toggle Button', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<div class="mdxcsp-field-wrapper">
												<div class="mdxcsp-image-field">
													<div class="mdxcsp-image-preview-wrap">
														<img class="mdxcsp-image-preview" data-target="menu_toggle_icon_id" src="<?php echo esc_url( $menu_toggle_icon ?: $default_toggle_url ); ?>" alt="" style="<?php echo ( $menu_toggle_icon || $default_toggle_url ) ? '' : 'display:none;'; ?>max-width:36px;height:auto;">
													</div>
													<input type="hidden" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_toggle_icon_id]" value="<?php echo esc_attr( $menu_toggle_icon_id ); ?>">
													<button type="button" class="button mdxcsp-upload" data-target="menu_toggle_icon_id"><?php echo esc_html__( 'Upload/Select Icon', 'xpertcodes-coming-soon-maintenance' ); ?></button>
													<button type="button" class="button-link-delete mdxcsp-remove" data-target="menu_toggle_icon_id" <?php disabled( $menu_toggle_icon_id, 0 ); ?>><?php echo esc_html__( 'Remove', 'xpertcodes-coming-soon-maintenance' ); ?></button>
												</div>

												<p style="margin-top:8px;">
													<label class="mdxcsp-required-label" for="mdxcsp-menu-toggle-label-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Toggle Text/Emoji', 'xpertcodes-coming-soon-maintenance' ); ?></label><br>
													<div class="mdxcsp-inline-field">
														<input type="text" class="regular-text" id="mdxcsp-menu-toggle-label-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_toggle_label]" value="<?php echo esc_attr( $menu_toggle_label ); ?>" placeholder="☰">
														<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="<?php echo esc_attr( $nav_color ?: '#ffffff' ); ?>" id="mdxcsp-menu-toggle-label-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][menu_toggle_label_color]" value="<?php echo esc_attr( $menu_toggle_label_color ?: ( $nav_color ?: '#ffffff' ) ); ?>">
													</div>
												</p>
												<p class="description"><?php echo esc_html__( 'Mobile menu toggle button settings', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>
								</tbody>
							</table>
						</details>

						<!-- Social & Footer -->
						<details class="mdxcsp-acc" open>
							<summary><?php echo esc_html__( 'Social & Footer', 'xpertcodes-coming-soon-maintenance' ); ?></summary>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><label class="mdxcsp-required-label"><?php echo esc_html__( 'Social Icons', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<fieldset>
												<p>
													<strong class="mdxcsp-required-label"><?php echo esc_html__( 'Icon 1', 'xpertcodes-coming-soon-maintenance' ); ?></strong><br>
													<div class="mdxcsp-image-field" style="margin:8px 0;">
														<div class="mdxcsp-image-preview-wrap">
															<img class="mdxcsp-image-preview" data-target="social_1_id" src="<?php echo esc_url( $social_1_img ?: $default_social1_url ); ?>" alt="" style="<?php echo ( $social_1_img || $default_social1_url ) ? '' : 'display:none;'; ?>max-width:36px;height:auto;">
														</div>
														<input type="hidden" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][social_1_id]" value="<?php echo esc_attr( $social_1_id ); ?>">
														<button type="button" class="button mdxcsp-upload" data-target="social_1_id"><?php echo esc_html__( 'Upload/Select', 'xpertcodes-coming-soon-maintenance' ); ?></button>
														<button type="button" class="button-link-delete mdxcsp-remove" data-target="social_1_id" <?php disabled( $social_1_id, 0 ); ?>><?php echo esc_html__( 'Remove', 'xpertcodes-coming-soon-maintenance' ); ?></button>
													</div>
													<div class="icons_url">
													<label class="mdxcsp-required-label" for="mdxcsp-social-1-url-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Icon 1 URL', 'xpertcodes-coming-soon-maintenance' ); ?></label><br>
													<input type="text" class="regular-text code" placeholder="https://example.com or #section" id="mdxcsp-social-1-url-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][social_1_url]" value="<?php echo esc_attr( $social_1_url ); ?>">
													</div>
												</p>

                                                    <div class="icon_btw-border"></div>

												<p>
													<strong class="mdxcsp-required-label"><?php echo esc_html__( 'Icon 2', 'xpertcodes-coming-soon-maintenance' ); ?></strong><br>
													<div class="mdxcsp-image-field" style="margin:8px 0;">
														<div class="mdxcsp-image-preview-wrap">
															<img class="mdxcsp-image-preview" data-target="social_2_id" src="<?php echo esc_url( $social_2_img ?: $default_social2_url ); ?>" alt="" style="<?php echo ( $social_2_img || $default_social2_url ) ? '' : 'display:none;'; ?>max-width:36px;height:auto;">
														</div>
														<input type="hidden" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][social_2_id]" value="<?php echo esc_attr( $social_2_id ); ?>">
														<button type="button" class="button mdxcsp-upload" data-target="social_2_id"><?php echo esc_html__( 'Upload/Select', 'xpertcodes-coming-soon-maintenance' ); ?></button>
														<button type="button" class="button-link-delete mdxcsp-remove" data-target="social_2_id" <?php disabled( $social_2_id, 0 ); ?>><?php echo esc_html__( 'Remove', 'xpertcodes-coming-soon-maintenance' ); ?></button>
													</div>
													<label class="mdxcsp-required-label" for="mdxcsp-social-2-url-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Icon 2 URL', 'xpertcodes-coming-soon-maintenance' ); ?></label><br>
													<input type="text" class="regular-text code" placeholder="https://example.com or #section" id="mdxcsp-social-2-url-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][social_2_url]" value="<?php echo esc_attr( $social_2_url ); ?>">
												</p>
											</fieldset>
											<div class="icon_btw-border"></div>
										</td>
									</tr>

									<tr>
										<th scope="row"><label class="mdxcsp-required-label" for="mdxcsp-copyright-<?php echo esc_attr( $tkey ); ?>"><?php echo esc_html__( 'Copyright Text', 'xpertcodes-coming-soon-maintenance' ); ?></label></th>
										<td>
											<div class="mdxcsp-field-wrapper">
												<div class="mdxcsp-inline-field">
													<input type="text" class="regular-text" id="mdxcsp-copyright-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][copyright_text]" value="<?php echo esc_attr( $copyright_text ); ?>" placeholder="<?php echo esc_attr__( 'Your Company. All rights reserved.', 'xpertcodes-coming-soon-maintenance' ); ?>">
													<input type="text" class="mdxcsp-color-picker wp-color-picker" data-default-color="#9CA3AF" id="mdxcsp-copyright-color-<?php echo esc_attr( $tkey ); ?>" name="mdxcsp_template_settings[<?php echo esc_attr( $tkey ); ?>][copyright_color]" value="<?php echo esc_attr( $copyright_color ?: '#9CA3AF' ); ?>">
												</div>
												<p class="description"><?php echo esc_html__( 'Footer copyright information', 'xpertcodes-coming-soon-maintenance' ); ?></p>
											</div>
										</td>
									</tr>
								</tbody>
							</table>
						</details>
					</div>
				<?php endforeach; ?>
			</div>

			<!-- Sticky Save Bar -->
			<div class="mdxcsp-sticky-actions">
				<div class="mdxcsp-sticky-left">
					<span class="mdxcsp-dirty-indicator" aria-hidden="true"></span>
					<span class="mdxcsp-sticky-text"><?php echo esc_html__( 'Unsaved changes', 'xpertcodes-coming-soon-maintenance' ); ?></span>
				</div>
				<div class="mdxcsp-sticky-right">
					<?php submit_button( __( 'Save Changes', 'xpertcodes-coming-soon-maintenance' ), 'primary', 'submit', false ); ?>
				</div>
			</div>
		</form>
	</div>
</div>