<?php
/**
 * Plugin Name: Coming Soon & Maintenance Mode
 * Plugin URI:  https://wordpress.org/plugins/xpertcodes-coming-soon-maintenance/
 * Description: Create coming soon or maintenance mode pages easily with templates and countdowns.
 * Version:     1.0.0
 * Requires PHP: 8.0
 * Author:      Adnan Hyder Pervez
 * Author URI:  https://profiles.wordpress.org/adnanhyder/
 * Text Domain: xpertcodes-coming-soon-maintenance
 * Domain Path: /languages
 * License:     GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @package MDXCSP_Coming_Soon_Pro
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * --------------------------------------------------
 * Define plugin constants (guarded).
 * --------------------------------------------------
 */
if ( ! defined( 'MDXCSP_PLUGIN_VERSION' ) ) {
	define( 'MDXCSP_PLUGIN_VERSION', '1.0.0' );
}

if ( ! defined( 'MDXCSP_MIN_PHP_VERSION' ) ) {
	define( 'MDXCSP_MIN_PHP_VERSION', '8.0' );
}

if ( ! defined( 'MDXCSP_PLUGIN_FILE' ) ) {
	define( 'MDXCSP_PLUGIN_FILE', __FILE__ );
}

if ( ! defined( 'MDXCSP_PLUGIN_BASENAME' ) ) {
	define( 'MDXCSP_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
}

if ( ! defined( 'MDXCSP_PLUGIN_DIR' ) ) {
	define( 'MDXCSP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}

if ( ! defined( 'MDXCSP_PLUGIN_URL' ) ) {
	define( 'MDXCSP_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}

/**
 * --------------------------------------------------
 * Activation/Deactivation hooks.
 * --------------------------------------------------
 */

/**
 * Runs on plugin activation.
 *
 * @return void
 * @since 1.0.0
 */
function mdxcsp_coming_soon_activate(): void {
	// Set default options on activation.
	$default_options = array(
		'mdxcsp_enable_coming_soon' => '0',
		'mdxcsp_selected_template'  => 'template-1',
		'mdxcsp_template_settings'  => array(),
	);

	foreach ( $default_options as $option_name => $default_value ) {
		if ( false === get_option( $option_name ) ) {
			add_option( $option_name, $default_value );
		}
	}

	// Flush rewrite rules for good measure.
	flush_rewrite_rules();
}

register_activation_hook( MDXCSP_PLUGIN_FILE, 'mdxcsp_coming_soon_activate' );

/**
 * Runs on plugin deactivation.
 *
 * @return void
 * @since 1.0.0
 */
function mdxcsp_coming_soon_deactivate(): void {
	// Clean up any temporary data if needed.
	flush_rewrite_rules();
}

register_deactivation_hook( MDXCSP_PLUGIN_FILE, 'mdxcsp_coming_soon_deactivate' );

/**
 * --------------------------------------------------
 * Bail out on unsupported PHP versions.
 * --------------------------------------------------
 */
if ( version_compare( PHP_VERSION, MDXCSP_MIN_PHP_VERSION, '<' ) ) {

	/**
	 * Deactivate plugin on admin init if PHP version is unsupported.
	 *
	 * @return void
	 */
	function mdxcsp_deactivate_self(): void {
		if ( current_user_can( 'activate_plugins' ) ) {
			deactivate_plugins( MDXCSP_PLUGIN_BASENAME );
		}
	}

	add_action( 'admin_init', 'mdxcsp_deactivate_self' );

	/**
	 * Show admin notice about PHP version.
	 *
	 * @return void
	 */
	function mdxcsp_admin_notice_php_version(): void {
		if ( ! current_user_can( 'activate_plugins' ) ) {
			return;
		}

		printf(
			'<div class="notice notice-error"><p>%s</p></div>',
			esc_html(
				sprintf(
					/* translators: 1: required PHP version, 2: current PHP version */
					__( 'Coming Soon Pro — Maintenance Mode requires PHP %1$s or higher. Your site is running PHP %2$s. The plugin has been deactivated.', 'xpertcodes-coming-soon-maintenance' ),
					MDXCSP_MIN_PHP_VERSION,
					PHP_VERSION
				)
			)
		);
	}

	add_action( 'admin_notices', 'mdxcsp_admin_notice_php_version' );

	// Do not load the rest of the plugin.
	return;
}




/**
 * Enqueue dynamic inline CSS for Template 2.
 */
function xpertcodes_enqueue_template_2_dynamic_css() {
    // Register a base style to attach inline CSS
    wp_register_style(
        'xpertcodes-template-2',
        plugin_dir_url( __FILE__ ) . 'assets/css/template-2.css',
        array(),
        '1.0.0'
    );
    wp_enqueue_style('xpertcodes-template-2');

    // Get your dynamic color variables
    $title_color             = get_option( 'title_color', '#000000' );
    $subtitle_color          = get_option( 'subtitle_color', '#000000' );
    $countdown_color         = get_option( 'countdown_color', '#000000' );
    $nav_color               = get_option( 'nav_color', '#000000' );
    $menu_toggle_label_color = get_option( 'menu_toggle_label_color', '#000000' );
    $copyright_color         = get_option( 'copyright_color', '#000000' );

    // Build inline CSS
    $custom_css = "
        body.mdxcsp-coming-soon.template-2 .mdxcsp-title,
        body.mdxcsp-coming-soon.template-2 .mdxcsp-title .accent { color: {$title_color} !important; }
        body.mdxcsp-coming-soon.template-2 .mdxcsp-subtitle { color: {$subtitle_color} !important; }
        body.mdxcsp-coming-soon.template-2 .mdxcsp-counter,
        body.mdxcsp-coming-soon.template-2 .mdxcsp-counter .num,
        body.mdxcsp-coming-soon.template-2 .mdxcsp-counter .lbl { color: {$countdown_color} !important; }
        body.mdxcsp-coming-soon.template-2 .mdxcsp-header,
        body.mdxcsp-coming-soon.template-2 .mdxcsp-header a { color: {$nav_color} !important; }
        body.mdxcsp-coming-soon.template-2 .mdxcsp-menu-toggle { color: {$menu_toggle_label_color} !important; }
        body.mdxcsp-coming-soon.template-2 .mdxcsp-footer-left { color: {$copyright_color} !important; }
    ";

    // Attach inline CSS properly
    wp_add_inline_style( 'xpertcodes-template-2', $custom_css );
}
add_action( 'wp_enqueue_scripts', 'xpertcodes_enqueue_template_2_dynamic_css' );

/**
 * --------------------------------------------------
 * Load required files.
 * --------------------------------------------------
 */
$admin_file    = MDXCSP_PLUGIN_DIR . 'src/admin/class-mdxcsp-admin.php';
$frontend_file = MDXCSP_PLUGIN_DIR . 'src/frontend/class-mdxcsp-frontend.php';

if ( file_exists( $admin_file ) ) {
	require_once $admin_file;
} else {
	error_log( 'Coming Soon Pro: Admin class file not found at ' . $admin_file ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
}

if ( file_exists( $frontend_file ) ) {
	require_once $frontend_file;
} else {
	error_log( 'Coming Soon Pro: Frontend class file not found at ' . $frontend_file ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
}

/**
 * --------------------------------------------------
 * Initialize plugin classes.
 * --------------------------------------------------
 */

/**
 * Initialize plugin classes.
 *
 * @return void
 */
function mdxcsp_init_plugin(): void {
	// Initialize admin functionality.
	if ( class_exists( 'MDXCSP_Admin' ) ) {
		new MDXCSP_Admin();
	}

	// Initialize frontend functionality.
	if ( class_exists( 'MDXCSP_Frontend' ) ) {
		new MDXCSP_Frontend();
	}
}

add_action( 'plugins_loaded', 'mdxcsp_init_plugin', 5 );

/**
 * --------------------------------------------------
 * Plugin uninstallation hook.
 * --------------------------------------------------
 */

/**
 * Runs on plugin uninstallation.
 *
 * @return void
 */
function mdxcsp_coming_soon_uninstall(): void {
	// Define options to delete on uninstall.
	$options_to_delete = array(
		'mdxcsp_enable_coming_soon',
		'mdxcsp_selected_template',
		'mdxcsp_template_settings',
	);

	// Delete options.
	foreach ( $options_to_delete as $option ) {
		delete_option( $option );
	}

	// Clear any cached data.
	wp_cache_flush();
}

register_uninstall_hook( MDXCSP_PLUGIN_FILE, 'mdxcsp_coming_soon_uninstall' );