<?php
namespace XFIVE_ACF_SYNC;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( '\XFIVE_ACF_SYNC\Plugin' ) ) {

	/**
	 * Plugin class.
	 */
	class Plugin extends \ACF_Admin_Internal_Post_Type_List {

		/**
		 * The class instance.
		 *
		 * @var class|null
		 */
		private static $instance = null;

		/**
		 * The acf groups post type.
		 *
		 * @var string
		 */
		public $post_type = 'acf-field-group';

		/**
		 * The acf groups sync data.
		 *
		 * @var array
		 */
		public $sync = array();

		/**
		 * The acf group single post id.
		 *
		 * @var null|int
		 */
		protected $post_id = null;

		/**
		 * The key of acf field group to sync.
		 *
		 * @var null|string
		 */
		protected $single_post_key = null;

		/**
		 * The acf groups sync page url.
		 *
		 * @var string
		 */
		protected $sync_page_url = '';

		/**
		 * The acf groups page url.
		 *
		 * @var string
		 */
		protected $fields_page_url = '';

		/**
		 * ACF section current page
		 *
		 * @var string
		 */
		public $view = '';

		/**
		 * The class constructor.
		 */
		private function __construct() {
			add_action( 'init', array( $this, 'i18n' ) );
			add_action( 'admin_notices', array( $this, 'acf_sync_notice' ) );
			add_action( 'admin_init', array( $this, 'setup' ) );

			add_action( 'admin_enqueue_scripts', array( $this, 'load_plugin_scripts' ) );
		}

		/**
		 * Load plugin translation strings
		 */
		public function i18n() {
			// Load user's custom translations from wp-content/languages/ folder.
			load_textdomain(
				'xfive-sync-watcher-for-acf',
				sprintf(
					'%s/%s-%s.mo',
					WP_LANG_DIR,
					XFIVE_ACF_SYNC_SLUG,
					get_locale(),
				)
			);
		}

		/**
		 * Setup the sync data.
		 */
		public function setup() {
			if ( wp_doing_ajax() || wp_doing_cron() ) {
				return;
			}

			// Stop if parent method doesn't exist.
			if ( ! method_exists( $this, 'setup_sync' ) ) {
				return;
			}

			global $pagenow;
			$this->view = acf_request_arg( 'post_status', '' );

			if ( $pagenow === 'edit.php' && isset( $_GET['post_type'] ) && sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) === $this->post_type ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$this->view = 'all';
			}

			if ( $pagenow === 'post.php' && isset( $_GET['post'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$post_id   = absint( $_GET['post'] ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				$post_type = get_post_type( $post_id );

				if ( $post_type === $this->post_type ) {
					$this->post_id = $post_id;
					$this->view    = 'single';
				}
			}

			if ( ! $this->view ) {
				return;
			}

			// Check for unsynced fields groups.
			$this->setup_sync();

			if ( $this->view === 'single' && $this->sync ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				foreach ( $this->sync as $key => $field_group ) {
					if ( $field_group['ID'] === $this->post_id ) {
						$this->single_post_key = $key;

						break;
					}
				}
			}

			$this->sync_page_url = add_query_arg(
				array(
					'post_type'   => $this->post_type,
					'post_status' => 'sync',
				),
				admin_url( 'edit.php' )
			);

			$this->fields_page_url = add_query_arg(
				array(
					'post_type'   => $this->post_type,
				),
				admin_url( 'edit.php' )
			);
		}

		/**
		 * Enqueue plugin scripts
		 *
		 * @return void
		 */
		public function load_plugin_scripts() {
			if ( $this->view ) {
				wp_enqueue_style( XFIVE_ACF_SYNC_SLUG, XFIVE_ACF_SYNC_URL . '/dist/css/plugin.css', array(), '1.0.0' );
				wp_enqueue_script( XFIVE_ACF_SYNC_SLUG, XFIVE_ACF_SYNC_URL . '/dist/js/plugin.js', array(), '1.0.0', true );
				wp_localize_script(
					XFIVE_ACF_SYNC_SLUG,
					'xfiveAcfSyncData',
					array(
						'get' => array_map( 'sanitize_text_field', wp_unslash( $_GET ) ), // phpcs:ignore -- Verified elsewhere.
					)
				);
			}
		}

		/**
		 * Check if sync notice needs to be displayed.
		 *
		 * @return void
		 */
		public function acf_sync_notice() {
			if ( $this->post_id && $this->single_post_key ) { // We're on a single page.
				$this->display_modal_notice();
			} elseif ( $this->sync ) {
				$this->display_admin_notice();
			}
		}

		/**
		 * Display ACF sync admin notice.
		 *
		 * @return void
		 */
		protected function display_admin_notice() {
			?>
			<div class="error">
				<p><?php esc_html_e( 'You have unsynced ACF fields.', 'xfive-sync-watcher-for-acf' ); ?> <a href="<?php echo esc_url( $this->sync_page_url ); ?>"><?php esc_html_e( ' View fields available for sync', 'xfive-sync-watcher-for-acf' ); ?></a></p>
			</div>
			<?php
		}

		/**
		 * Display ACF sync modal notice.
		 *
		 * @return void
		 */
		protected function display_modal_notice() {
			$sync_url   = $this->get_admin_url( '&acfsync=' . $this->single_post_key . '&_wpnonce=' . wp_create_nonce( 'bulk-posts' ) );
			$review_url = add_query_arg( array( 'review' => $this->post_id ), $this->fields_page_url );
			?>
			<div class="xfive-acf-sync-modal js-xfive-acf-sync-modal is-open" role="dialog">
				<div class="xfive-acf-sync-modal__box">
					<div class="xfive-acf-sync-modal__content">
						<div class="xfive-acf-sync-modal__icon">
							<svg aria-hidden="true" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 20 20">
								<path stroke="#d63638" stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 11V6m0 8h.01M19 10a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z"></path>
							</svg>
						</div>

						<h3>
							<?php esc_html_e( 'The field group you are editing has not been synced with your local acf json file.', 'xfive-sync-watcher-for-acf' ); ?>
							<?php esc_html_e( 'Making any changes will overwrite the local json file and result in losing data.', 'xfive-sync-watcher-for-acf' ); ?>
							<?php esc_html_e( 'This is irreversible.', 'xfive-sync-watcher-for-acf' ); ?>
						</h3>

						<div class="xfive-acf-sync-modal__actions">
							<a href="<?php echo esc_url( $review_url ); ?>" class="acf-btn"><?php esc_html_e( 'View changes', 'xfive-sync-watcher-for-acf' ); ?></a>
							<a href="<?php echo esc_url( $sync_url ); ?>" class="acf-btn"><?php esc_html_e( 'Sync data', 'xfive-sync-watcher-for-acf' ); ?></a>
							<button type="button" class="acf-btn acf-btn-secondary js-xfive-acf-sync-modal-close"><?php esc_html_e( 'Ignore', 'xfive-sync-watcher-for-acf' ); ?></button>
						</div>
					</div>
				</div>
			</div>
			<?php
		}

		/**
		 * Get the instance of the class.
		 *
		 * @return class
		 */
		public static function get_instance() {
			if ( self::$instance === null ) {
					self::$instance = new self();
			}

			return self::$instance;
		}
	}

	\XFIVE_ACF_SYNC\Plugin::get_instance();

}
