<?php

namespace XfiveSentry\WP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use XfiveSentry\Trait\Singleton;
use XfiveSentry\Helper\PluginHelpers;

class Settings {

	use Singleton;

	/**
	 * Settings data.
	 *
	 * @var array
	 */
	private $settings = array();

	/**
	 * Settings slug.
	 *
	 * @var string
	 */
	private $settings_slug = 'xfive-sentry-integration';

	/**
	 * Settings cache.
	 *
	 * @var array
	 */
	private static $settings_cache = array();

	/**
	 * Settings constructor.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'set_settings' ) );
		add_action( 'admin_menu', array( $this, 'plugin_settings_page' ) );
		add_action( 'admin_init', array( $this, 'plugin_settings' ) );
	}

	/**
	 * Set settings data.
	 *
	 * @return void
	 */
	public function set_settings() {
		$this->settings = apply_filters(
			'xfive_sentry_integration_settings',
			array(
				'config'           => array(
					'title'  => __( 'Settings', 'xfive-sentry-integration' ),
					'fields' => array(
						'dsn'                    => array(
							'type'        => 'text',
							'label'       => __( 'DSN', 'xfive-sentry-integration' ),
							'description' => __( 'The DSN tells the SDK where to send the events to.', 'xfive-sentry-integration' ),
						),
						'env'                    => array(
							'type'        => 'text',
							'label'       => __( 'Environment', 'xfive-sentry-integration' ),
							'description' => __( 'Sets the environment. Useful for filtering in Sentry.', 'xfive-sentry-integration' ),
							'placeholder' => PluginHelpers::get_wp_env(),
						),
						'tracing'                => array(
							'type'        => 'select',
							'label'       => __( 'Enable Tracing', 'xfive-sentry-integration' ),
							'description' => __( 'Automatic performance issue detection across services and context on who is impacted, outliers, regressions, and the root cause of your slowdown.', 'xfive-sentry-integration' ) . sprintf( ' <a target="_blank" rel="noreferrer noopener" href="https://docs.sentry.io/platforms/php/tracing/">%s</a>', __( 'Learn more', 'xfive-sentry-integration' ) ),
							'options'     => array(
								'yes' => __( 'Yes', 'xfive-sentry-integration' ),
								'no'  => __( 'No', 'xfive-sentry-integration' ),
							),
							'default'     => 'no',
						),
						'traces_sample_rate_php' => array(
							'type'        => 'number',
							'label'       => __( 'Traces Sample Rate - PHP', 'xfive-sentry-integration' ),
							'description' => __( 'Value between 0 and 1. Use lower values in production. Tracing must be enabled.', 'xfive-sentry-integration' ),
							'placeholder' => '0.2',
							'default'     => '0.2',
							'min'         => '0',
							'max'         => '1',
							'step'        => '0.1',
						),
						'traces_sample_rate_js'  => array(
							'type'        => 'number',
							'label'       => __( 'Traces Sample Rate - JavaScript', 'xfive-sentry-integration' ),
							'description' => __( 'Value between 0 and 1. Use lower values in production. Tracing must be enabled.', 'xfive-sentry-integration' ),
							'placeholder' => '0.2',
							'default'     => '0.2',
							'min'         => '0',
							'max'         => '1',
							'step'        => '0.1',
						),
					),
				),
				'errors_filtering' => array(
					'title'  => __( 'Errors Filtering', 'xfive-sentry-integration' ),
					'fields' => array(
						'ef_instructions'    => array(
							'type'        => 'html',
							'label'       => '',
							'html'        => __( 'Use with caution as enabling these options can result in a lot of errors being sent to Sentry and use up your quota.', 'xfive-sentry-integration' ),
							'description' => '',
						),
						'admin_errors_php'   => array(
							'type'        => 'select',
							'label'       => __( 'Admin area errors - PHP', 'xfive-sentry-integration' ),
							'description' => __( 'Choose whether to capture or ignore PHP errors occurred in the admin area.', 'xfive-sentry-integration' ),
							'options'     => array(
								'capture' => __( 'Capture', 'xfive-sentry-integration' ),
								'ignore'  => __( 'Ignore', 'xfive-sentry-integration' ),
							),
							'default'     => 'ignore',
						),
						'admin_errors_js'    => array(
							'type'        => 'select',
							'label'       => __( 'Admin area errors - JavaScript', 'xfive-sentry-integration' ),
							'description' => __( 'Choose whether to capture or ignore JS errors occurred in the admin area.', 'xfive-sentry-integration' ),
							'options'     => array(
								'capture' => __( 'Capture', 'xfive-sentry-integration' ),
								'ignore'  => __( 'Ignore', 'xfive-sentry-integration' ),
							),
							'default'     => 'ignore',
						),
						'plugins_errors_php' => array(
							'type'        => 'select',
							'label'       => __( 'Plugins errors - PHP', 'xfive-sentry-integration' ),
							'description' => __( 'Choose whether to capture or ignore PHP errors occured in plugins.', 'xfive-sentry-integration' ),
							'options'     => array(
								'capture' => __( 'Capture', 'xfive-sentry-integration' ),
								'ignore'  => __( 'Ignore', 'xfive-sentry-integration' ),
							),
							'default'     => 'ignore',
						),
						'plugins_errors_js'  => array(
							'type'        => 'select',
							'label'       => __( 'Plugins errors - JavaScript', 'xfive-sentry-integration' ),
							'description' => __( 'Choose whether to capture or ignore JS errors occured in plugins.', 'xfive-sentry-integration' ),
							'options'     => array(
								'capture' => __( 'Capture', 'xfive-sentry-integration' ),
								'ignore'  => __( 'Ignore', 'xfive-sentry-integration' ),
							),
							'default'     => 'ignore',
						),
						'ignore_errors_php'  => array(
							'type'        => 'textarea',
							'label'       => __( 'Ignore errors by message - PHP', 'xfive-sentry-integration' ),
							'description' => __( 'Add list of error messages to ignore. Can be a part of a message to match. Enter one match per line.', 'xfive-sentry-integration' ) . sprintf( ' <a target="_blank" rel="noreferrer noopener" href="https://docs.sentry.io/platforms/javascript/configuration/filtering/#using-ignore-errors">%s</a>', __( 'Learn more', 'xfive-sentry-integration' ) ),
						),
						'ignore_errors_js'   => array(
							'type'        => 'textarea',
							'label'       => __( 'Ignore errors by message - JavaScript', 'xfive-sentry-integration' ),
							'description' => __( 'Add list of error messages to ignore. Can be a part of a message to match. Enter one match per line.', 'xfive-sentry-integration' ) . sprintf( ' <a target="_blank" rel="noreferrer noopener" href="https://docs.sentry.io/platforms/javascript/configuration/filtering/#using-ignore-errors">%s</a>', __( 'Learn more', 'xfive-sentry-integration' ) ),
						),
						'allow_urls'         => array(
							'type'        => 'textarea',
							'label'       => __( 'Allow URLs - JavaScript', 'xfive-sentry-integration' ),
							'description' => __( 'Enter a list of  URLs to allow sending errors that originate in scripts from certain domains. Enter one url per line.', 'xfive-sentry-integration' ) . sprintf( ' <a target="_blank" rel="noreferrer noopener" href="https://docs.sentry.io/platforms/javascript/configuration/filtering/#using-allowurls-and-denyurls">%s</a>', __( 'Learn more', 'xfive-sentry-integration' ) ),
						),
						'deny_urls'          => array(
							'type'        => 'textarea',
							'label'       => __( 'Deny URLs - JavaScript', 'xfive-sentry-integration' ),
							'description' => __( 'Enter a list of URLs to avoid sending errors that originate in scripts from certain domains. Enter one url per line.', 'xfive-sentry-integration' ) . sprintf( ' <a target="_blank" rel="noreferrer noopener" href="https://docs.sentry.io/platforms/javascript/configuration/filtering/#using-allowurls-and-denyurls">%s</a>', __( 'Learn more', 'xfive-sentry-integration' ) ),
						),
					),
				),
				'tests'            => array(
					'title'  => __( 'Testing', 'xfive-sentry-integration' ),
					'fields' => array(
						'test' => array(
							'type'        => 'html',
							'label'       => __( 'Test Error Event', 'xfive-sentry-integration' ),
							'html'        => sprintf(
								'<a class="button" href="%s" rel="noreferrer noopener">%s</a>',
								add_query_arg( array( 'error' => '1' ), admin_url( 'options-general.php?page=xfive-sentry-integration', false ) ),
								__( 'Trigger an error', 'xfive-sentry-integration' ),
							),
							'description' => __( 'Trigger an error to test the Sentry integration. Refresh the page after the error has been displayed.', 'xfive-sentry-integration' ),
						),
					),
				),
			)
		);
	}

	/**
	 * Create settings page.
	 *
	 * @return void
	 */
	public function plugin_settings_page() {
		add_submenu_page(
			'options-general.php',
			__( 'xfive Aristotle - Monitor Site Errors with Sentry', 'xfive-sentry-integration' ),
			__( 'Monitor Site Errors with Sentry', 'xfive-sentry-integration' ),
			'manage_options',
			$this->settings_slug,
			array( $this, 'plugin_settings_page_html' ),
		);
	}

	/**
	 * Create settings sections and fields
	 *
	 * @return void
	 */
	public function plugin_settings() {
		foreach ( $this->settings as $section => $section_data ) {
			$section_id = $this->settings_slug . '_section_' . $section;

			add_settings_section(
				$section_id,
				$section_data['title'],
				array( $this, 'plugin_settings_section_html' ),
				$this->settings_slug,
				array(
					'data' => array(
						'section' => $section,
					),
				)
			);

			foreach ( $section_data['fields'] as $field => $field_data ) {
				$setting_id   = $this->settings_slug . '_field_' . $field;
				$setting_name = $this->settings_slug . '_' . $field;

				$field_data['field'] = $field;

				add_settings_field(
					$setting_id,
					$field_data['label'],
					array( $this, 'plugin_settings_field_html' ),
					$this->settings_slug,
					$section_id,
					array(
						'label_for' => $setting_name,
						'class'     => isset( $field_data['class'] ) ? $field_data['class'] : '',
						'data'      => $field_data,
					)
				);

				register_setting(
					$this->settings_slug,
					$setting_name,
					array(
						'type'              => isset( $field_data['val_type'] ) ? sanitize_text_field( $field_data['val_type'] ) : 'string',
						'sanitize_callback' => $this->get_sanitize_callback( $field_data['type'] ),
					)
				);
			}
		}
	}

	/**
	 * Get sanitize callback based on the field type or a custom one.
	 *
	 * @param string $field_type - Field html type.
	 *
	 * @return string
	 */
	private function get_sanitize_callback( $field_type ) {
		if ( isset( $field_data['san_cb'] ) ) {
			return sanitize_text_field( $field_data['san_cb'] );
		}

		switch ( $field_type ) {
			case 'textarea':
				return 'sanitize_textarea_field';
			default:
				return 'sanitize_text_field';
		}
	}

	/**
	 * Render settings section.
	 *
	 * @return void
	 */
	public function plugin_settings_section_html() {
		if ( isset( $_GET['error'] ) ) { // phpcs:ignore -- Nonce verified by wp.
			xfive_sentry_integration_test_error_callback();
		}
	}

	/**
	 * Display settings fields html
	 *
	 * @param array $args Settings field arguments.
	 *
	 * @return void
	 */
	public function plugin_settings_field_html( $args ) {
		$field_type = $args['data']['type'];

		$field_value         = self::get_setting( $args['data']['field'], null );
		$field_default_value = isset( $args['data']['default'] ) ? sanitize_text_field( $args['data']['default'] ) : '';

		if ( $field_value === null && $field_default_value ) {
			$field_value = $field_default_value;
		}

		$args['value'] = $field_value;

		PluginHelpers::get_html_part( 'settings-fields', $field_type, $args );
	}

	/**
	 * Render settings page.
	 *
	 * @return void
	 */
	public function plugin_settings_page_html() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$button_label = __( 'Save Settings', 'xfive-sentry-integration' );
		?>
		<div class="wrap clb-settings-page">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<form action="options.php" method="POST">
				<?php
				settings_fields( $this->settings_slug );

				do_settings_sections( $this->settings_slug );

				submit_button( $button_label );
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Get setting value.
	 *
	 * @param string $setting_name Setting name.
	 * @param string $default_value Default value.
	 *
	 * @return string
	 */
	public static function get_setting( $setting_name, $default_value = '' ) {
		if ( isset( self::$settings_cache[ $setting_name ] ) ) {
			return self::$settings_cache[ $setting_name ];
		}

		$setting = get_option( self::get_instance()->settings_slug . '_' . $setting_name, $default_value );

		self::$settings_cache[ $setting_name ] = $setting;

		return $setting;
	}
}
