<?php

namespace XfiveSentry\WP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use XfiveSentry\Trait\Singleton;
use XfiveSentry\Service\Sentry;
use XfiveSentry\WP\Settings;
use XfiveSentry\Helper\PluginHelpers;

class Plugin {

	use Singleton;

	/**
	 * Sentry instance
	 *
	 * @var Sentry
	 */
	private $sentry = null;

	/**
	 * Plugin constructor.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'initialize_sentry' ) );
		add_action( 'plugins_loaded', array( $this, 'load_plugin_core' ) );
		add_action( 'init', array( $this, 'i18n' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'load_fe_scripts' ), 1 );
		add_action( 'admin_enqueue_scripts', array( $this, 'load_admin_scripts' ), 1 );
	}

	/**
	 * Load plugin core classes.
	 */
	public function load_plugin_core() {
		Settings::get_instance();
	}

	/**
	 * Initialize Sentry SDK.
	 */
	public function initialize_sentry() {
		$this->sentry = new Sentry();
		$this->sentry->init();
	}

	/**
	 * Load plugin translation strings
	 */
	public function i18n() {
		// Load user's custom translations from wp-content/languages/ folder.
		load_textdomain(
			'xfive-sentry-integration',
			sprintf(
				'%s/%s-%s.mo',
				WP_LANG_DIR,
				XFIVE_SENTRY_SLUG,
				get_locale(),
			)
		);
	}

	/**
	 * Load frontend scripts.
	 */
	public function load_fe_scripts() {
		$this->load_scripts();
	}

	/**
	 * Load admin scripts.
	 */
	public function load_admin_scripts() {
		if ( PluginHelpers::is_admin_area() && $this->sentry->get_errors_filtering( 'admin_js' ) === 'ignore' ) {
			return;
		}

		$this->load_scripts();
	}

	/**
	 * Load scripts.
	 */
	public function load_scripts() {
		$app_js = XFIVE_SENTRY_SLUG . '-app';

		wp_enqueue_script(
			$app_js,
			PluginHelpers::get_script_url( 'app', 'js' ),
			array(),
			XFIVE_SENTRY_VERSION,
			array(
				'in_footer' => true,
				'strategy'  => 'defer',
			)
		);
		wp_localize_script(
			$app_js,
			'xfiveSentryIntegrationData',
			array(
				'isLocalhost'   => PluginHelpers::is_localhost(),
				'config'        => $this->sentry->get_config( 'js' ),
				'ignorePlugins' => $this->sentry->get_errors_filtering( 'plugins_js' ) === 'ignore' ? 'yes' : 'no',
				'ignoreErrors'  => PluginHelpers::textarea_to_array( $this->sentry->get_errors_filtering( 'ignore_js' ) ),
				'allowUrls'     => PluginHelpers::textarea_to_array( $this->sentry->get_errors_filtering( 'allow_urls' ) ),
				'denyUrls'      => PluginHelpers::textarea_to_array( $this->sentry->get_errors_filtering( 'deny_urls' ) ),
			)
		);
	}
}
