<?php

namespace XfiveSentry\Service;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use Sentry\init;
use XfiveSentry\WP\Settings;
use XfiveSentry\Helper\PluginHelpers;

class Sentry {

	/**
	 * Sentry configuration - https://docs.sentry.io/platforms/php/configuration/.
	 *
	 * @var array
	 */
	private $config = array();

	/**
	 * Environment name.
	 *
	 * @var array
	 */
	private $environment = 'production';

	/**
	 * Is tracing enabled.
	 *
	 * @var string
	 */
	private $tracing_enabled = 'no';

	/**
	 * Traces sample rate for PHP.
	 *
	 * @var float
	 */
	private $traces_sample_rate_php = 0.2;

	/**
	 * Traces sample rate for JS.
	 *
	 * @var float
	 */
	private $traces_sample_rate_js = 0.2;

	/**
	 * Class constructor.
	 *
	 * @return void
	 */
	public function __construct() {
		require_once XFIVE_SENTRY_DIR . 'vendor/autoload.php';

		$this->set_env();
		$this->set_tracing();
		$this->set_config();
	}

	/**
	 * Set Sentry configuration.
	 *
	 * @return void
	 */
	private function set_config() {
		$base_config = array(
			'dsn'         => Settings::get_setting( 'dsn' ),
			'environment' => $this->get_env(),
		);

		$php_config = $this->get_php_config();
		$js_config  = $this->get_js_config();

		$this->config = array(
			'php' => array_merge( $base_config, $php_config ),
			'js'  => array_merge( $base_config, $js_config ),
		);
	}

	/**
	 * Get Sentry configuration.
	 *
	 * @param string $type Configuration type.
	 *
	 * @return array
	 */
	public function get_config( $type = 'php' ) {
		return $this->config[ $type ] ?? array();
	}

	/**
	 * Get Sentry configuration for php only.
	 *
	 * @return array
	 */
	public function get_php_config() {
		$config = array();

		if ( $this->tracing_enabled === 'yes' ) {
			$config['traces_sample_rate'] = $this->get_tracing( 'php' );
		}

		$plugins_errors = $this->get_errors_filtering( 'plugins_php' );
		$ignore_errors  = $this->get_errors_filtering( 'ignore_php' );

		$config['before_send'] = function ( \Sentry\Event $event ) use ( $plugins_errors, $ignore_errors ) {
			$should_stop       = false;
			$has_ignore_errors = sanitize_text_field( $ignore_errors ) !== '';

			if ( $plugins_errors === 'ignore' || $has_ignore_errors ) {
				$exceptions = $event->getExceptions();

				if ( $exceptions ) {
					foreach ( $exceptions as $exception ) {
						$trace   = $exception->getStacktrace();
						$frames  = $trace->getFrames();
						$message = $exception->getValue();

						if ( $plugins_errors === 'ignore' && $frames ) {
							foreach ( $frames as $frame ) {
								$file            = $frame->getFile();
								$normalized_file = str_replace( '\\', '/', $file );

								if ( strpos( $normalized_file, '/wp-content/plugins/' ) !== false && strpos( $normalized_file, '/wp-content/plugins/' . XFIVE_SENTRY_SLUG ) === false ) {
									$should_stop = true;
									break;
								}
							}
						}

						if ( $has_ignore_errors ) {
							$errors_to_ignore = PluginHelpers::textarea_to_array( $ignore_errors );

							if ( $errors_to_ignore ) {
								foreach ( $errors_to_ignore as $error_to_ignore ) {
									if ( strpos( $error_to_ignore, $message ) !== false ) {
										$should_stop = true;
										break;
									}
								}
							}
						}
					}
				}
			}

			if ( $should_stop ) {
				return null;
			}

			return $event;
		};

		return $config;
	}

	/**
	 * Get Sentry configuration for js only.
	 *
	 * @return array
	 */
	public function get_js_config() {
		$config = array();

		if ( $this->tracing_enabled === 'yes' ) {
			$config['tracesSampleRate'] = $this->get_tracing( 'js' );
		}

		return $config;
	}

	/**
	 * Set tracing values.
	 *
	 * @return void
	 */
	public function set_tracing() {
		$this->tracing_enabled        = Settings::get_setting( 'tracing' );
		$this->traces_sample_rate_php = Settings::get_setting( 'traces_sample_rate_php' );
		$this->traces_sample_rate_js  = Settings::get_setting( 'traces_sample_rate_js' );
	}

	/**
	 * Get tracing value.
	 *
	 * @param string $type Tracing type.
	 *
	 * @return float
	 */
	public function get_tracing( $type = 'php' ) {
		if ( $this->tracing_enabled !== 'yes' ) {
			return false;
		}

		if ( $type === 'php' ) {
			return (float) $this->traces_sample_rate_php;
		} elseif ( $type === 'js' ) {
			return (float) $this->traces_sample_rate_js;
		}

		return false;
	}

	/**
	 * Get errors filtering settings.
	 *
	 * @param string $key Setting key.
	 *
	 * @return string
	 */
	public function get_errors_filtering( $key ) {
		$settings = array(
			'admin_php'   => Settings::get_setting( 'admin_errors_php', 'ignore' ),
			'plugins_php' => Settings::get_setting( 'plugins_errors_php', 'ignore' ),
			'admin_js'    => Settings::get_setting( 'admin_errors_js', 'ignore' ),
			'plugins_js'  => Settings::get_setting( 'plugins_errors_js', 'ignore' ),
			'ignore_php'  => Settings::get_setting( 'ignore_errors_php', '' ),
			'ignore_js'   => Settings::get_setting( 'ignore_errors_js', '' ),
			'allow_urls'  => Settings::get_setting( 'allow_urls', '' ),
			'deny_urls'   => Settings::get_setting( 'deny_urls', '' ),
		);

		return isset( $settings[ $key ] ) ? $settings[ $key ] : 'ignore';
	}

	/**
	 * Set environment value.
	 *
	 * @return void
	 */
	public function set_env() {
		$setting = Settings::get_setting( 'env' );

		if ( $setting ) {
			$this->environment = $setting;
		} else {
			$this->environment = PluginHelpers::get_wp_env();
		}
	}

	/**
	 * Get environment value.
	 *
	 * @return string
	 */
	public function get_env() {
		return $this->environment;
	}

	/**
	 * Initialize Sentry.
	 *
	 * @return void
	 */
	public function init() {
		if ( PluginHelpers::is_admin_area() && $this->get_errors_filtering( 'admin_php' ) === 'ignore' ) {
			return;
		}

		if ( class_exists( '\Sentry\SentrySdk' ) && ! PluginHelpers::is_localhost() ) {
			\Sentry\init( $this->get_config( 'php' ) );
		}
	}
}
