<?php

namespace XfiveSentry\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Helper functions.
 */
class PluginHelpers {

	/**
	 * Get the IP address.
	 *
	 * @return string
	 */
	public static function get_ip() {
		if ( isset( $_SERVER['HTTP_CLIENT_IP'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_CLIENT_IP'] ) );
		} elseif ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
			$ip = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
		} else {
			$ip = isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : '127.0.0.1';
		}

		return $ip;
	}

	/**
	 * Check if the current IP is localhost.
	 *
	 * @return bool
	 */
	public static function is_localhost() {
		return in_array( self::get_ip(), array( '127.0.0.1', '::1' ), true );
	}

	/**
	 * Get the current environment type.
	 *
	 * @return string
	 */
	public static function get_wp_env() {
		return defined( 'WP_ENVIRONMENT_TYPE' ) ? WP_ENVIRONMENT_TYPE : 'production';
	}

	/**
	 * Get html part - include file with html content.
	 *
	 * @param string $folder Folder name.
	 * @param string $part File name.
	 * @param array  $args Arguments.
	 * @param bool   $echo_output Echo or return.
	 *
	 * @return string
	 */
	public static function get_html_part( $folder, $part, $args = array(), $echo_output = true ) {
		$file_name = XFIVE_SENTRY_DIR . $folder . '/' . $part . '.php';

		if ( ! file_exists( $file_name ) ) {
			return '';
		}

		ob_start();
		include $file_name;
		$output = ob_get_clean();

		if ( $echo_output ) {
			echo $output; // phpcs:ignore
		} else {
			return $output;
		}
	}

	/**
	 * This function will generate script url
	 *
	 * @param string $name File name.
	 * @param string $type File type.
	 *
	 * @return string
	 */
	public static function get_script_url( $name, $type ) {
		$file = XFIVE_SENTRY_DIR . 'build/mix-manifest.json';

		if ( ! file_exists( $file ) ) {
			return false;
		}

		$mix_file = json_decode( file_get_contents( $file ), true ); // phpcs:ignore

		$key = sprintf( '/%1$s/%2$s.%1$s', $type, $name );

		return isset( $mix_file[$key] ) ? XFIVE_SENTRY_URL . '/build' . $mix_file[$key] : false;
	}

	/**
	 * Check if we are in admin area except for the plugin settings page.
	 *
	 * @return bool
	 */
	public static function is_admin_area() {
		if ( is_admin() ) {
			$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : ''; // phpcs:ignore

			return $page !== XFIVE_SENTRY_SLUG;
		}

		return false;
	}

	/**
	 * Convert textarea new lines to array
	 *
	 * @param string $text - Textarea value.
	 *
	 * @return array
	 */
	public static function textarea_to_array( $text ) {
		if ( ! $text ) {
			return array();
		}

		return array_map( 'sanitize_text_field', explode( "\n", $text ) );
	}
}
