<?php
/**
 * Plugin Name: Credit Online Smart by BT Direct
 * Short Description: Credit SMART (BT Direct) payment gateway for WooCommerce.
 * Description: Adds BT Direct (Credit SMART) financing to WooCommerce checkout so customers can apply online and pay in installments. Includes callbacks, and order status updates.
 * Version: 1.0.22
 * Author: XFactorApp
 * Author URI: http://xfactorapp.com/
 * Requires Plugins: woocommerce
 * Requires at least: 5.0
 * Requires PHP: 7.2
 * WC requires at least: 5.0
 * WC tested up to: 10.3.0
 *
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 */

defined('ABSPATH') || exit;

define('KRUNITY_META_INSTANCE_ID', '_krunity_id');
define('KRUNITY_META_STATUS_LOG', '_krunity_status_log');

require_once 'src/Krunity.php';

use Automattic\WooCommerce\Enums\OrderStatus;
use Krunity\Krunity;

Krunity::addWCPluginLink('settings', 'Settings', 'checkout');

function krunity_add_gateway()
{
    static $pluginKrunity;

    if (!isset($pluginKrunity)) {
        class XFactor_Krunity
        {
            private static $instance;

            /**
             * Returns the *Singleton* instance of this class.
             *
             */
            public static function get_instance()
            {
                if (null === self::$instance) {
                    self::$instance = new self();
                }
                return self::$instance;
            }

            public function __construct()
            {
                add_action('rest_api_init', [$this, 'add_routes']);
                add_filter('woocommerce_payment_gateways', [$this, 'add_gateways']);
            }

            public function add_gateways($methods)
            {
                if (krunity_gateway_class_bootstrap()) {
                    $Krunity = new XFactor_Gateway_Krunity();
                    $Krunity::$instance = &$Krunity;

                    $methods['paymentplugins_krunity'] = XFactor_Gateway_Krunity::class;
                }

                return $methods;
            }


            function add_routes()
            {

                register_rest_route('krunity/v1', 'login', [
                    'methods' => WP_REST_Server::EDITABLE,
                    'callback' => [$this, 'routeLogin'],
                    'show_in_rest' => true,
                    'permission_callback' =>'__return_true',
                    'args' => [
                        'Username' => [
                            'required' => true,
                        ],
                        'Password' => [
                            'required' => true,
                        ],
                    ],

                ]);
                register_rest_route('krunity/v1', 'update', [
                    'methods' => WP_REST_Server::EDITABLE,
                    'callback' => [$this, 'routeUpdate'],
                    'show_in_rest' => true,
                    'permission_callback' => '__return_true', // Autorizat prin JWT
                    'args' => [
                        'IdShoppingCart' => [
                            'required' => true,
                        ], 'IdPaymentProcessInstance' => [
                            'required' => true,
                        ], 'MerchantStatusKey' => [
                            'required' => true,
                        ], 'MerchantStatusDescription' => [
                            'required' => false,
                        ], 'ApprovedAmount' => [
                            'required' => false,
                        ],
                    ],

                ]);
                register_rest_route('krunity/v1', 'test', [
                    'methods' => 'POST',
                    'callback' => [$this, 'test'],
                    'show_in_rest' => true,
                    'nonce'   => wp_create_nonce('wp_rest'),
                    'permission_callback' => array( $this, 'must_be_shop_manager_or_admin' ),
                    'args' => [],
                ]);
            }


            /**
             * @return void
             * http://localhost/wp-json/krunity/v1/login/
             */

            public function routeLogin($request)
            {
                krunity_gateway_class_bootstrap();
                $Krunity = XFactor_Gateway_Krunity::$instance;
                if (!$Krunity) {
                    Krunity::log('Gateway not available [routeLogin].', ['key' => 'routeLogin']);
                    wp_send_json_error('Gateway not available', 500);
                }

                $username = $request->get_param('Username');
                $password = $request->get_param('Password');

                if (!($Krunity->get_option('username') && $Krunity->get_option('username') === $username)) {
                    wp_send_json_error('Invalid username!');
                    return;
                }

                if (!($Krunity->get_option('password') && $Krunity->get_option('password') === $password)) {
                    wp_send_json_error('Invalid password!');
                    return;
                }

                static::respondData(['Token' => Krunity::getJwtToken([])]);
            }

            /**
             * @return void
             * http://localhost/wp-json/krunity/v1/test/
             */

            public function test($request)
            {
                krunity_gateway_class_bootstrap();
                $Krunity = XFactor_Gateway_Krunity::$instance;
                if (!$Krunity) {
                    Krunity::log('Gateway not available [test].', ['key' => 'test']);
                    wp_send_json_error('Gateway not available', 500);
                }

                $response = \Krunity\Krunity::postJson($Krunity->get_option('base_url') . '/token/', [
                    "Username" => $Krunity->get_option('username'),
                    "Password" => $Krunity->get_option('password'),
                    "IsExternalUser" => "true"
                ]);

                if (is_array($response) && isset($response['token']) && strlen((string)$response['token']) > 0) {
                    $response['token'] = substr($response['token'], 0, 50) . '......';
                    $response['refreshToken'] = substr($response['refreshToken'], 0, 30) . '......';
                }

                static::respondData($response);
            }

            public
            static function respondData($data = [], $status_code = null, $options = 0)
            {
                wp_send_json($data, $status_code, $options);
            }


            /**
             * @return void
             * http://localhost/wp-json/krunity/update/?Token=eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpYXQiOjE2NzgxMTQ4NDIsImV4cCI6MTY3ODExODQ0Mn0.MHWj7_2pkvr3wR_yJOXpZL66UYuhhCDV8c_zTybnTVM&IdShoppingCart
             */
            public
            function routeUpdate($request)
            {
                $order = null;
                $order_id = 0;

                try {
                    krunity_gateway_class_bootstrap();
                    $Krunity = XFactor_Gateway_Krunity::$instance;
                    if (!$Krunity) {
                        Krunity::log('Gateway not available [routeUpdate].', ['key' => 'routeUpdate']);
                        wp_send_json_error('Gateway not available', 500);
                    }

                    if ($Krunity->get_option('third_party_auth') === 'yes') {
                        $user = wp_get_current_user();
                        if (!$user->exists()) {
                            wp_send_json_error('Authentification failed', 403);
                        }
                    } else {
                        $token = Krunity::getBearerToken();
                        $tokenDecoded = (array)Krunity::decodeJWTToken($token);

                        if (!$tokenDecoded) {
                            wp_send_json_error('JWT Token is missing', 401);
                        }

                        if ($tokenDecoded['status'] === 'error') {
                            wp_send_json_error($tokenDecoded['error'], 401);
                        }
                    }

                    $order_id = absint($request->get_param('IdShoppingCart'));
                    if (!$order_id) {
                        wp_send_json_error('Invalid IdShoppingCart!', 400);
                    }

                    $order = wc_get_order($order_id);
                    if (!is_object($order)) {
                        wp_send_json_error('Invalid IdShoppingCart!', 404);
                    }

                    $krunityId = sanitize_text_field((string)$request->get_param('IdPaymentProcessInstance'));
                    if ($krunityId === '') {
                        wp_send_json_error('Missing IdPaymentProcessInstance!', 400);
                    }

                    $stored = (string) $order->get_meta(KRUNITY_META_INSTANCE_ID, true);
                    if ($stored === '' || !hash_equals($stored, $krunityId)) {
                        \Krunity\Krunity::krunity_log_order($order_id, 'Invalid IdPaymentProcessInstance!', ['status' => 'Invalid']);
                        wp_send_json_error('Invalid IdPaymentProcessInstance!', 400);
                    }

                    if ( $order->has_status( [ OrderStatus::CANCELLED ] ) ) {
                        \Krunity\Krunity::krunity_log_order( $order_id, 'Order is cancelled.', [ 'status' => 'Cancelled' ] );
                        wp_send_json_success( [ 'status' => 'Cancelled', 'msg' => 'Order is cancelled.' ] );
                    }

                    if ( $order->has_status( [ OrderStatus::COMPLETED ] ) ) {
                        \Krunity\Krunity::krunity_log_order( $order_id, 'Order is completed.', [ 'status' => 'Completed' ] );
                        wp_send_json_success( [ 'status' => 'Completed', 'msg' => 'Order is completed.' ] );
                    }

                    if ( $order->is_paid() ) {
                        \Krunity\Krunity::krunity_log_order( $order_id, 'Already paid, ignoring.', [ 'status' => 'Duplicate' ] );
                        wp_send_json_success( [ 'status' => 'AlreadyPaid', 'msg' => 'Order is already paid.' ] );
                    }

                    $statusKey = sanitize_text_field((string)$request->get_param('MerchantStatusKey'));
                    $statusDesc = sanitize_text_field((string)$request->get_param('MerchantStatusDescription'));

                    $data = [
                        'status' => $statusKey,
                    ];

                    $approvedRaw = $request->get_param('ApprovedAmount');
                    if ($approvedRaw !== null && $approvedRaw !== '') {
                        $data['amount'] = $approvedRaw;
                    }

                    \Krunity\Krunity::krunity_log_order($order_id, $statusDesc, $data);

                    if ($statusKey === 'FinalApproved') {
                        if ($approvedRaw === null || $approvedRaw === '') {
                            wp_send_json_error('Missing ApprovedAmount!', 400);
                        }

                        $approved = (float) wc_format_decimal((string)$approvedRaw, wc_get_price_decimals());
                        $total    = (float) wc_format_decimal((string)$order->get_total(), wc_get_price_decimals());

                        if ($approved < $total) {
                            wp_send_json_error('Invalid ApprovedAmount!', 400);
                        }

                        $order->payment_complete( $krunityId );

                        \Krunity\Krunity::krunity_log_order($order_id, 'Order completed!', ['status' => 'Success', 'transaction_id'  => $krunityId, 'approved_amount' => $approved]);
                    } /*else if (in_array($request->get_param('MerchantStatusKey'), ['OnlineFinished', 'Abandoned', 'RedirectedTLMK', 'Preapprove', 'PreapproveWithModifications', 'Resended', 'WaitingTLMK', 'AutomaticValidations', 'Processed', 'ProcessedOnHold', 'PreapprovePending', 'MovedToOtherDevice'])) {
                        $order->update_status(OrderStatus::ON_HOLD);
                        \Krunity\Krunity::krunity_log_order($order_id, 'Order payment on hold', ['status' => 'OnHold', 'transaction_id'  => $krunityId]);
                    }*/ else if (in_array($request->get_param('MerchantStatusKey'), ['Rejected', 'PreaproveExpired', 'PreaproveCanceled', 'Canceled', 'CanceledTLMK', 'SessionExpired'])) {
                        $order->update_status(OrderStatus::FAILED);
                        \Krunity\Krunity::krunity_log_order($order_id, 'Order payment failed!', ['status' => 'PaymentFailed', 'transaction_id'  => $krunityId]);
                    }

                    $order->save();

                    wp_send_json_success(['status' => 'ok']);
                } catch (\Throwable $e) {
                    $message = $e->getMessage();

                    if (is_object($order)) {
                        \Krunity\Krunity::krunity_log_order($order_id, 'Update status failed', ['status' => 'Error', 'msg' => $message]);
                    }

                    $trace = $e->getTrace();
                    $traceFiles = [$e->getFile() . ':' . $e->getLine()];
                    foreach ($trace as $v) {
                        $traceFiles[] = $v['file'] . ':' . $v['line'];
                    }
                    $message .= ' ' . implode(' -> ', $traceFiles);

                    if ( class_exists( 'WC_Logger' ) ) {
                        $logger = wc_get_logger();
                        $logger->error( $message, [ 'source' => 'xfactorapp-krunity-credit-smart-bt-direct' ] );
                    }

                    wp_send_json_error(['message' => $e->getMessage()], 400);
                }
            }


            /**
             * User must be either shop_manager or administrator.
             *
             * @return bool
             */
            public
            function must_be_shop_manager_or_admin() {
                // phpcs:ignore
                if ( ! current_user_can( 'administrator' ) ) {
                    return false;
                }
                return true;
            }

        }

        $pluginKrunity = XFactor_Krunity::get_instance();
    }

    return $pluginKrunity;
}

function krunity_gateway_class_bootstrap(): bool {
    if ( class_exists( 'XFactor_Gateway_Krunity', false ) ) {
        return true;
    } else {
        $file = __DIR__ . '/src/XFactor_Gateway_Krunity.php';
        if ( file_exists( $file ) ) {
            require_once $file;
            return true;
        } else {
            Krunity::log('Gateway not found!', ['file' => $file]);
        }
    }

    return false;
}


add_action('plugins_loaded', 'woocommerce_gateway_krunity_init');
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
                'custom_order_tables', // HPOS
                __FILE__,
                true // true = compatibil
        );
    }
} );

function woocommerce_gateway_krunity_init()
{
    krunity_add_gateway();
}

add_action( 'woocommerce_blocks_payment_method_type_registration', function( $registry ) {
    if ( class_exists( '\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
        require_once __DIR__ . '/src/Blocks/Class_Krunity_Blocks.php';
        $registry->register( new \XFactorApp\Krunity\Blocks\Krunity_Blocks() );
    }
} );

add_filter('all_plugins', function ($plugins) {
    $target = 'trunk/xfactorapp-krunity-credit-smart-bt-direct.php';
    $target2 = 'xfactorapp-krunity-credit-smart-bt-direct/xfactorapp-krunity-credit-smart-bt-direct.php';

    if (isset($plugins[$target])) {
        $plugins[$target]['Name'] = Krunity::PLUGIN_TITLE;
    }if (isset($plugins[$target2])) {
        $plugins[$target2]['Name'] = Krunity::PLUGIN_TITLE;
    }

    return $plugins;
});

add_action('admin_footer', function () {
    $screen = get_current_screen();
    if ( ! $screen ) return;

    // HPOS: id = 'woocommerce_page_wc-orders'; clasic: 'shop_order'
    $is_order_screen =
        $screen->id === 'woocommerce_page_wc-orders' ||         // HPOS order edit
        $screen->id === 'shop_order' ||                          // classic post type editor
        $screen->id === 'woocommerce_page_wc-orders--edit';      // (unele versiuni folosesc sufix)

    if ( ! $is_order_screen ) return;

    try {
        $order_id = absint($_GET['id'] ?? $_GET['order_id'] ?? $_GET['post'] ?? 0);
        if (!$order_id) return;

        $order = wc_get_order($order_id);
        if (!$order) return;

        $logs = $order->get_meta(KRUNITY_META_STATUS_LOG , false);
        if (!is_array($logs)) $logs = [];

        usort($logs, function ($a, $b) {
            $a = is_object($a) && method_exists($a, 'get_data') ? $a->get_data() : (array) $a;
            $b = is_object($b) && method_exists($b, 'get_data') ? $b->get_data() : (array) $b;

            $aDate = isset($a['value']['date']) ? strtotime($a['value']['date']) : 0;
            $bDate = isset($b['value']['date']) ? strtotime($b['value']['date']) : 0;

            return $bDate <=> $aDate; // DESC
        });

        // Instance ID
        $instanceId = $order->get_meta( KRUNITY_META_INSTANCE_ID, true );

    } catch (\Throwable $e) {
        $logs = [];
    }

    ?>
    <div id="krunity-order-logs-box" class="postbox" style="display:none;margin-top:16px">
        <h2 class="hndle"><span><?php echo esc_html__('Krunity – Order logs', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></span></h2>
        <div class="inside">
            <?php if ( empty($logs) ) : ?>
                <p><?php echo esc_html__('No logs for this order yet.', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></p>
            <?php else : ?>
                <table class="widefat fixed striped">
                    <?php if ($instanceId) { ?>
                        <caption><p style="text-align: left;"><strong><?= esc_html__( 'Instance ID', 'xfactorapp-krunity-credit-smart-bt-direct' ) . ':</strong> ' . esc_html( (string) $instanceId ) ?></p></caption>
                    <?php } ?>
                    <thead>
                    <tr>
                        <th style="width:180px"><?php echo esc_html__('Date', 'woocommerce'); ?></th>
                        <th style="min-width:200px"><?php echo esc_html__('Status', 'woocommerce'); ?></th>
                        <th><?php echo esc_html__('description', 'woocommerce'); ?></th>
                        <th style="width:auto;"><?php echo esc_html__('Content', 'woocommerce'); ?></th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach ( $logs as $row ) :
                        $entry = is_object( $row ) && method_exists( $row, 'get_data' ) ? $row->get_data() : (array) $row;
                        $value = isset( $entry['value'] ) ? (array) $entry['value'] : [];

                        $date        = isset( $value['date'] ) ? (string) $value['date'] : '';
                        $description = isset( $value['description'] ) ? (string) $value['description'] : '';
                        $amount_raw  = isset( $value['amount'] ) ? $value['amount'] : null;
                        $status      = isset( $value['status'] ) ? (string) $value['status'] : '';
                        $debug_raw   = isset( $value['debug'] ) ? $value['debug'] : '';

                        $amount_str = '';
                        if ( $amount_raw !== null && $amount_raw !== '' ) {
                            if ( is_numeric( $amount_raw ) ) {
                                $amount_str = ' [<strong>' . number_format_i18n( (float) $amount_raw, 2 ) . ' RON</strong>]';
                            }
                        }
                        ?>
                        <tr>
                            <td><?php echo esc_html($date); ?></td>
                            <td><?php echo esc_html($status) . $amount_str; ?></td>
                            <td><?php echo esc_html($description); ?></td>
                            <td>
                                <details>
                                    <summary><?php echo esc_html__('View', 'woocommerce'); ?></summary>
                                    <pre style="white-space:pre-wrap;word-break:break-word;margin:6px 0 0"><?php
                                        echo esc_html($debug_raw) ?>
                                    </pre>
                                </details>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>

    <script>
        (function(){
            function insertBox() {
                var box = document.getElementById('krunity-order-logs-box');
                if (!box) return;

                // Ținte posibile pentru ancorare (în ordinea preferinței)
                var target =
                    document.getElementById('postcustom') ||                       // meta box "Custom Fields" (clasic & HPOS)
                    document.querySelector('#woocommerce-order-downloads') ||       // Downloadable product permissions
                    document.querySelector('.wrap.woocommerce .metabox-holder');    // fallback container

                if (target && target.parentNode) {
                    target.parentNode.insertBefore(box, target); // inserează ÎNAINTE de Custom Fields
                    box.style.display = 'block';
                } else {
                    // dacă nu găsim ținta încă (UI încă se hidratează), mai încercăm puțin
                    setTimeout(insertBox, 300);
                }
            }
            // încarcă după ce DOM-ul e gata
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', insertBox);
            } else {
                insertBox();
            }
        })();
    </script>
    <?php
});


add_action('wp_enqueue_scripts', function () {
    if (is_product()) {
        wp_enqueue_script('bt-direct-widget', plugins_url('assets/js/bt-direct-widget.js', __FILE__), [], '1.0.22', true);

        wp_add_inline_script('bt-direct-widget',
            "document.querySelectorAll('bt-direct-widget').forEach(widget => {
                widget.addEventListener('action', () => {
                    const addToCartButton = document.querySelector('button.single_add_to_cart_button');
                    if (addToCartButton) {
                        addToCartButton.click();
                    } else {
                        const cartForm = document.querySelector('form.cart');
                        if (cartForm) {
                            cartForm.submit();
                        }
                    }
                });
            });"
        );
    }
});

add_action('woocommerce_after_add_to_cart_button', function () {
    $settings = get_option('woocommerce_krunity_settings', []);
    if (($settings['enabled'] ?? 'no') !== 'yes') {
        return;
    }

    global $product;
    if (!$product) {
        return;
    }

    krunity_gateway_class_bootstrap();
    if ( ! class_exists('XFactor_Gateway_Krunity', false) ) {
        Krunity::log('Gateway not available [cart_button].', ['key' => 'woocommerce_after_add_to_cart_button']);
        return;
    }

    if ( empty(XFactor_Gateway_Krunity::$instance) ) {
        WC()->payment_gateways();
    }

    $Krunity = XFactor_Gateway_Krunity::$instance;
    $price = $product->get_price();

    if ($Krunity) {
        if ($Krunity->get_option('widget_general_enable') === 'no') {
            return;
        }

        if ($price > (float)$Krunity->get_option('widget_promotional_min_amount')) {
            $rate = $Krunity->get_option('widget_promotional_rate') ?? 24.0;
            $minPeriod = $Krunity->get_option('widget_promotional_period_min') ?? 6;
            $maxPeriod = $Krunity->get_option('widget_promotional_period_max') ?? 60;
        } else {
            $rate = $Krunity->get_option('widget_standard_rate') ?? 34.92;
            $minPeriod = $Krunity->get_option('widget_standard_period_min') ?? 6;
            $maxPeriod = $Krunity->get_option('widget_standard_period_max') ?? 60;
        }

        $period = $Krunity->get_option('widget_general_period') ?? 36;

        ?>
        <div style="margin-top: 20px;">
            <bt-direct-widget
                    data-price="<?php echo esc_attr($price); ?>"
                    data-period="<?= $period ?>"
                    data-interest="<?= $rate ?>"
                    data-period-min="<?= $minPeriod ?>"
                    data-period-max="<?= $maxPeriod ?>"
                    data-action-text="Adaugă în coș"
                    data-action-button="true"
                    data-dialog="<?= $Krunity->get_option('widget_general_modal') !== 'no' ? 'true' : 'false' ?>"
            ></bt-direct-widget>
        </div>
        <?php
    }
});
