<?php
if ( ! defined( 'ABSPATH' ) ) exit;

require_once 'Krunity.php';

use Krunity\Krunity;

/**
 * Krunity.
 *
 * Provides a Krumity Gateway.
 *
 * @class       WC_Gateway_COD
 * @extends     WC_Payment_Gateway
 * @version     2.1.0
 * @package     WooCommerce\Classes\Payment
 */
class XFactor_Gateway_Krunity extends WC_Payment_Gateway
{
    const KEY = 'krunity';
    const DEFAULT_REDIRECT_MESSAGE = '<strong>In cateva secunde vei fi redirectionat catre platforma SMART by BT Direct pentru completarea cererii de credit.</strong>';

    public static $instance;
    public $enable_for_methods;
    public $enable_for_virtual;
    public $third_party_auth;

    public function __construct()
    {

        if (!Krunity::isKrunityActive()) {
            Krunity::showAdminHeaderErrorMessage('WooCommerce must be active');
        }
        $this->setup_properties();
        $this->init_form_fields();
        $this->init_settings();

        $this->title = $this->get_option('title');

        $this->enable_for_methods = $this->get_option('enable_for_methods', []);
        $this->enable_for_virtual = $this->get_option('enable_for_virtual', 'yes') === 'yes';
        $this->third_party_auth = $this->get_option('third_party_auth', 'yes') === 'yes';

        // Actions.
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
        add_action('woocommerce_thankyou_' . $this->id, [$this, 'thankyou_page']);
        add_filter('woocommerce_payment_complete_order_status', [$this, 'change_payment_complete_order_status'], 10, 3);

        add_action('admin_enqueue_scripts', function($hook) {
            if (strpos($hook, 'woocommerce_page_wc-settings') === false) return;

            $token_url = rest_url('krunity/v1/test');

            wp_add_inline_script('jquery-core', '
                document.addEventListener("DOMContentLoaded", function () {
                    const btn = document.getElementById("krunity-debug-button");
                    if (!btn) return;
                
                    if (!btn.hasAttribute("data-click-bound")) {
                        btn.addEventListener("click", function () {
                            const responseDiv = document.getElementById("krunity-debug-response");
                            responseDiv.innerHTML = "Se verifică...";
                
                            fetch(' . wp_json_encode($token_url) . ', {
                                method: "POST",
                                credentials: "include",
                                headers: {
                                    "X-WP-Nonce": ' . wp_json_encode(wp_create_nonce('wp_rest')) . ',
                                    "Content-Type": "application/json"
                                },
                                body: "{}"
                            })
                            .then(r => r.json().then(j => ({ ok: r.ok, status: r.status, json: j })))
                            .then(({ok, status, json}) => {
                                responseDiv.innerHTML = "<pre>" + JSON.stringify({ok, status, json}, null, 2) + "</pre>";
                            })
                            .catch(error => {
                                responseDiv.innerHTML = "<span style=\'color:red\'>Eroare: " + error.message + "</span>";
                            });
                        });
                        
                        btn.setAttribute("data-click-bound", "true");
                    }
                });'
            );
        });
    }


    public function thankyou_page( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        $print_debug = ( 'yes' === $this->get_option( 'print_debug' ) );

        // --- Build a nonce for the return leg and append it to the thank-you URL ---
        $nonce_action = 'krunity_thankyou_' . $order_id;
        $nonce_value  = wp_create_nonce( $nonce_action );

        // If no "redirected" flag, send customer to gateway
        $redirected = isset( $_GET['redirected'] ) ? sanitize_text_field( wp_unslash( $_GET['redirected'] ) ) : '';

        if ($redirected === '1' || strtolower($redirected) === 'true') {
            $nonce_param = isset($_GET['krunity_nonce']) ? sanitize_text_field( wp_unslash($_GET['krunity_nonce']) ) : '';
            if ( empty($nonce_param) || ! wp_verify_nonce($nonce_param, $nonce_action) ) {
                wp_die( esc_html__('Invalid request. Please refresh the page.', 'xfactorapp-krunity-credit-smart-bt-direct') );
            }

            return;
        }

        $existing = $order->get_meta( KRUNITY_META_INSTANCE_ID );
        if ( !empty($existing) ) {
            $wfInstanceId = $existing;
        } else {
            $auth = Krunity::postJson(
                $this->get_option('base_url') . '/token/',
                [
                    'Username'       => $this->get_option('username'),
                    'Password'       => $this->get_option('password'),
                    'IsExternalUser' => 'true',
                    'order_id'       => $order_id,
                ]
            );

            if ( ! $auth || ! isset( $auth['token'] ) || strlen( (string) $auth['token'] ) === 0 ) {
                Krunity::pre('Invalid response from Gateway [AUTH]');
                $commonData = [
                    'order_id' => $order_id,
                    'response' => $auth
                ];
                \Krunity\Krunity::krunity_log_order(
                    $order_id,
                    'Invalid token',
                    $print_debug ? array_merge($commonData, [
                        'Username' => $this->get_option('username'),
                        'Password' => $this->get_option('password'),
                        'IsExternalUser' => 'true',
                        'base_url' => $this->get_option('base_url'),
                    ]) : $commonData
                );
                return;
            }

            $token       = $auth['token'];
            $firstName   = $order->get_billing_first_name();
            $lastName    = $order->get_billing_last_name();
            $phoneNumber = $order->get_billing_phone();

            // default return is WooCommerce "order received" page
            $returnUrl = add_query_arg(
                [
                    'redirected'    => '1',
                    'krunity_nonce' => $nonce_value,
                ],
                $order->get_checkout_order_received_url()
            );

            if ( 'static_url' === $this->get_option('return_url_type') ) {
                $returnUrl = add_query_arg(
                    [
                        'redirected'    => '1',
                        'krunity_nonce' => $nonce_value,
                    ],
                    $this->get_option('return_url')
                );
            } elseif ( 'no_redirect' === $this->get_option('return_url_type') ) {
                $returnUrl = '';
            }

            $data = [
                'FlowKey'                 => 'shopping_cart',
                'Hostname'                => $this->get_option('hostname'),
                'Price'                   => $order->get_total(),
                'IdShoppingCart'          => $order->get_id(),
                'RedirectUrl'             => $returnUrl,
                'FromMerchantFirstName'   => $firstName,
                'FromMerchantLastName'    => $lastName,
                'FromMerchantPhoneNumber' => $phoneNumber,
            ];

            \Krunity\Krunity::krunity_log_order( $order_id, 'The request was sent to BTD.', $print_debug
                ? $data
                : [
                    'Price' => $order->get_total(),
                    'IdShoppingCart' => $order->get_id()
                ] );

            $flow = Krunity::postJson(
                $this->get_option('base_url') . '/Integration/CreateFlowInstance',
                $data,
                $token
            );

            if ( $print_debug ) {
                Krunity::pre('Data Sent');
                Krunity::pre($data);
                Krunity::pre('Data Received');
                Krunity::pre($flow);
            }

            $order->save();
            $wfInstanceId = $flow['WfInstanceId'];

            if ( empty( $wfInstanceId ) ) {
                Krunity::pre('Invalid response from Gateway [FLOW]');
                \Krunity\Krunity::krunity_log_order(
                    $order_id,
                    'Invalid WfInstanceId',
                    $print_debug ? array_merge( (array) $flow, $data ) : [
                        'Price' => $order->get_total(),
                        'wfInstanceId' => $wfInstanceId,
                        'IdShoppingCart' => $order->get_id()
                    ]
                );
                return;
            }

            wc_maybe_reduce_stock_levels( $order_id ); // rezerva stocul daca Hold stock > 0

            $order->add_meta_data( KRUNITY_META_INSTANCE_ID, $wfInstanceId, true );
            $order->save();
        }

        $out = '';
        if ( $redirected !== '1' && strtolower($redirected) !== 'true' ) {

            // Build submit URL safely
            $submit_url = $this->get_option('submit_url');
            $url = add_query_arg(
                [
                    'price'         => (string) $order->get_total(),
                    'period'        => '12',
                    'hostname'      => $this->get_option('hostname'),
                    'idShoppingCart'=> (string) $order_id,
                    'wfInstanceId'  => (string) $wfInstanceId,
                ],
                $submit_url
            );

            if ( $print_debug ) {
                $out .= esc_url( $url );
            }

            $iframe = ( $this->get_option('integration_type') === 'iframe' );

            if ( $iframe ) {
                $out .= sprintf(
                    '<iframe allow="camera" style="height:50rem; width:60rem" src="%s"></iframe>',
                    esc_url( $url )
                );
            } else {
                printf(
                    '<strong><center>%s</center></strong>',
                    esc_html( self::DEFAULT_REDIRECT_MESSAGE )
                );

                echo '<script>setTimeout(function(){window.location.replace(' . wp_json_encode($url) . ')},2000)</script>';

                wp_redirect( $url );
                exit;
            }

        } else {
            // We only trust the "return" leg if the nonce is valid.
            $nonce_param = isset( $_GET['krunity_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['krunity_nonce'] ) ) : '';

            if ( empty( $nonce_param ) || ! wp_verify_nonce( $nonce_param, $nonce_action ) ) {
                if ( $print_debug ) {
                    Krunity::pre( 'Invalid or missing nonce on return.' );
                }
                wp_die( esc_html__( 'Invalid request. Please refresh the page.', 'xfactorapp-krunity-credit-smart-bt-direct' ) );
            }

            if ( $print_debug ) {
                Krunity::pre( [
                    'redirected'   => true,
                    'order_id'     => $order_id,
                    'wfInstanceId' => $flow['WfInstanceId'],
                ] );
            }
        }

        echo wp_kses($out, array_merge(Krunity::ALLOWED_MARKUP_FOR_DEBUG, ['iframe' => ['allow' => true, 'style' => true]]));
    }



    public function admin_options()
    {
        ?>
        <h3><?php echo $this->method_title; ?></h3>
        <?php echo $this->method_description; ?>
        <br>
        <br>

        <nav class="nav-tab-wrapper krunity-nav-tab-wrapper" style="margin-bottom: 20px;">
            <a href="#general" class="nav-tab" data-tab="general"><?php esc_html_e('General Settings', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></a>
            <a href="#credentials" class="nav-tab nav-tab-active" data-tab="credentials"><?php esc_html_e('Credentials', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></a>
            <a href="#widget" class="nav-tab" data-tab="widget"><?php esc_html_e('Widget Settings', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></a>
        </nav>

        <div class="krunity-subtab-wrapper" style="margin-bottom: 20px; display: none;">
            <ul class="subsubsub" style="margin: 0; float: none;">
                <li><a href="#general" class="krunity-subtab current" data-subtab="general"><?php esc_html_e('General', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></a> | </li>
                <li><a href="#standard" class="krunity-subtab" data-subtab="standard"><?php esc_html_e('Standard', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></a> | </li>
                <li><a href="#promotional" class="krunity-subtab" data-subtab="promotional"><?php esc_html_e('Promotional', 'xfactorapp-krunity-credit-smart-bt-direct'); ?></a></li>
            </ul>
            <br class="clear">
        </div>

        <style>
            .krunity-tab-content-hidden { display: none !important; }
            .krunity-subtab.current { font-weight: bold; }
        </style>

        <script type="text/javascript">
            jQuery(document).ready(function($) {
                var $tabs = $('.krunity-nav-tab-wrapper .nav-tab');
                var $subtabs = $('.krunity-subtab');
                var $rows = $('.form-table tr');
                var currentTab = 'credentials';
                var currentSubTab = 'general';

                function updateVisibility() {
                    $rows.addClass('krunity-tab-content-hidden');
                    
                    if (currentTab === 'widget') {
                        $('.krunity-subtab-wrapper').show();
                        $('.krunity-subtab-' + currentSubTab).closest('tr').removeClass('krunity-tab-content-hidden');
                    } else {
                        $('.krunity-subtab-wrapper').hide();
                        $('.krunity-tab-' + currentTab).closest('tr').removeClass('krunity-tab-content-hidden');
                    }
                }

                function switchTab(tabId) {
                    currentTab = tabId;
                    $tabs.removeClass('nav-tab-active');
                    $('.krunity-nav-tab-wrapper .nav-tab[data-tab="' + tabId + '"]').addClass('nav-tab-active');
                    updateVisibility();
                }

                function switchSubTab(subTabId) {
                    currentSubTab = subTabId;
                    $subtabs.removeClass('current');
                    $('.krunity-subtab[data-subtab="' + subTabId + '"]').addClass('current');
                    updateVisibility();
                }

                $tabs.on('click', function(e) {
                    e.preventDefault();
                    var tabId = $(this).data('tab');
                    switchTab(tabId);
                    window.location.hash = tabId + (tabId === 'widget' ? '-' + currentSubTab : '');
                });

                $subtabs.on('click', function(e) {
                    e.preventDefault();
                    var subTabId = $(this).data('subtab');
                    switchSubTab(subTabId);
                    window.location.hash = currentTab + '-' + subTabId;
                });

                // Initial tab
                var hash = window.location.hash.replace('#', '');
                if (hash) {
                    var parts = hash.split('-');
                    if (parts[0] && $('.krunity-nav-tab-wrapper .nav-tab[data-tab="' + parts[0] + '"]').length) {
                        currentTab = parts[0];
                        if (parts[1] && $('.krunity-subtab[data-subtab="' + parts[1] + '"]').length) {
                            currentSubTab = parts[1];
                        }
                    }
                }
                
                switchTab(currentTab);
                if (currentTab === 'widget') {
                    switchSubTab(currentSubTab);
                }
            });
        </script>

        <table class="form-table">
            <?php $this->generate_settings_html(); ?>
        </table>
        <?php
    }

    public function init_settings()
    {
        parent::init_settings();
        if (isset($this->settings['widget']) && is_array($this->settings['widget'])) {
            foreach ($this->settings['widget'] as $group => $fields) {
                if (is_array($fields)) {
                    foreach ($fields as $key => $value) {
                        $this->settings["widget_{$group}_{$key}"] = $value;
                    }
                }
            }
        }
    }

    public function process_admin_options()
    {
        $this->init_settings();
        $post_data = $this->get_post_data();
        $new_settings = $this->settings;
        $widget_data = [];

        foreach ($this->get_form_fields() as $key => $field) {
            if ('title' !== $this->get_field_type($field)) {
                try {
                    $value = $this->get_field_value($key, $field, $post_data);
                    
                    if (strpos($key, 'widget_') === 0) {
                        $parts = explode('_', $key);
                        if (count($parts) >= 3) {
                            $group = $parts[1];
                            $field_name = implode('_', array_slice($parts, 2));
                            $widget_data[$group][$field_name] = $value;
                        }
                    } else {
                        $new_settings[$key] = $value;
                    }
                } catch (Exception $e) {
                    $this->add_error($e->getMessage());
                }
            }
        }

        if (!empty($widget_data)) {
            $new_settings['widget'] = $widget_data;
        }

        $this->settings = $new_settings;
        $option_key = $this->get_option_key();
        update_option($option_key, apply_filters('woocommerce_settings_api_sanitized_fields_' . $this->id, $this->settings), 'yes');
        
        $this->init_settings(); // Refresh flattened settings
        return true;
    }

    public function generate_raw_html($key, $data)
    {
        $field_key = $this->get_field_key($key);
        $defaults  = array(
            'title'             => '',
            'disabled'          => false,
            'class'             => '',
            'css'               => '',
            'placeholder'       => '',
            'type'              => 'text',
            'desc_tip'          => false,
            'description'       => '',
            'custom_attributes' => array(),
        );

        $data = wp_parse_args($data, $defaults);

        ob_start();
        ?>
        <tr valign="top">
            <th scope="row" class="titledesc">
                <label for="<?php echo esc_attr($field_key); ?>"><?php echo wp_kses_post($data['title']); ?> <?php echo $this->get_tooltip_html($data); ?></label>
            </th>
            <td class="forminp">
                <fieldset>
                    <legend class="screen-reader-text"><span><?php echo wp_kses_post($data['title']); ?></span></legend>
                    <div class="<?php echo esc_attr($data['class']); ?>">
                        <?php echo $data['description']; ?>
                    </div>
                </fieldset>
            </td>
        </tr>
        <?php
        return ob_get_clean();
    }

    public function init_form_fields()
    {

        $this->form_fields = [
            'enabled' => [
                'title' => 'Enable/Disable',
                'label' => 'Enable ' . Krunity::PLUGIN_TITLE,
                'type' => 'checkbox',
                'description' => '',
                'default' => 'no',
                'class' => 'krunity-tab-general',
            ],
            'integration_type' => [
                'title' => 'Integration Type',
                'type' => 'select',
                'description' => '',
                'default' => 'redirect',
                'options' => ['iframe' => 'Iframe', 'redirect' => 'Redirect'],
                'class' => 'krunity-tab-general',
            ],
            'return_url_type' => [
                'title' => esc_html__('ReturnUrl Type', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'select',
                'options' => [
                    'woocommerce_default' => 'WooCommerce Default Thank You Page',
                    'static_url' => 'Static Url. Please fill the next field',
                    'no_redirect' => 'No Redirect',
                ],
                'default' => 'woocommerce_default',
                'required' => true,
                'class' => 'krunity-tab-general',
            ],
            'return_url' => [
                'title' => esc_html__('ReturnUrl', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'description' => esc_html__('Fill in Static Url', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'text',
                'required' => false,
                'default' => '',
                'class' => 'krunity-tab-general',
            ],
            'min_amount' => [
                'title' => esc_html__('MinAmount', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 300,
                'class' => 'krunity-tab-general',
            ],
            'enable_for_virtual' => [
                'title' => esc_html__('Accept for virtual orders', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'checkbox',
                'default' => 'yes',
                'class' => 'krunity-tab-general',
            ],
            'enable_for_methods' => [
                'title' => esc_html__('Enable for shipping methods', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'multiselect',
                'class' => 'wc-enhanced-select krunity-tab-general',
                'css' => 'width: 400px;',
                'default' => '',
                'options' => $this->load_shipping_method_options(),
                'desc_tip' => true,
                'custom_attributes' => [
                    'data-placeholder' => esc_html__('Select shipping methods', 'xfactorapp-krunity-credit-smart-bt-direct'),
                ],
            ],
            'print_debug' => [
                'label' => esc_html__('Show dedug info in checkout process', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'title' => esc_html__('Print debug info', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'checkbox',
                'default' => 'no',
                'class' => 'krunity-tab-general',
            ],



            'username' => [
                'title' => esc_html__('Username', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'text',
                'required' => true,
                'class' => 'krunity-tab-credentials',
            ],
            'password' => [
                'title' => esc_html__('Password', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'password',
                'required' => true,
                'class' => 'krunity-tab-credentials',
            ],
            'base_url' => [
                'title' => esc_html__('BaseUrl', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'text',
                'required' => true,
                'default' => 'https://api-onlinecredit.btdirect.ro/api ',
                'class' => 'krunity-tab-credentials',
            ],
            'submit_url' => [
                'title' => esc_html__('SubmitUrl', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'text',
                'required' => true,
                'default' => 'https://onlinecredit.btdirect.ro/wf/shoppingcart?wfId=25',
                'class' => 'krunity-tab-credentials',
            ],
            'hostname' => [
                'title' => esc_html__('Hostname', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'Hostname',
                'required' => true,
                'class' => 'krunity-tab-credentials',
            ],
            'third_party_auth' => [
                'title' => esc_html__('Use third party JWT authentication', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'checkbox',
                'default' => 'no',
                'class' => 'krunity-tab-credentials',
            ],


            'widget_general_enable' => [
                'title' => esc_html__('Enable Widget', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'checkbox',
                'required' => true,
                'default' => 'yes',
                'description' => esc_html__('Enable widget calculator (default: yes)', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'class' => 'krunity-tab-widget krunity-subtab-general',
            ],
            'widget_general_modal' => [
                'title' => esc_html__('Widget Modal', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'checkbox',
                'required' => true,
                'default' => 'yes',
                'description' => esc_html__('Display calculator in modal (default: no)', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'class' => 'krunity-tab-widget krunity-subtab-general',
            ],
            'widget_general_period' => [
                'title' => esc_html__('Preselected period', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 36,
                'description' => esc_html__('default: 36', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'class' => 'krunity-tab-widget krunity-subtab-general',
                'custom_attributes' => [
                    'step' => 1,
                    'min'  => 1
                ]
            ],


            'widget_standard_rate' => [
                'title' => esc_html__('Rate', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 34.92,
                'class' => 'krunity-tab-widget krunity-subtab-standard',
                'subclass' => 'standard',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 0
                ]
            ],
            'widget_standard_min_amount' => [
                'title' => esc_html__('Minimum amount', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 300,
                'class' => 'krunity-tab-widget krunity-subtab-standard',
                'subclass' => 'standard',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 0
                ]
            ],
            'widget_standard_period_min' => [
                'title' => esc_html__('Minimum period', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 6,
                'class' => 'krunity-tab-widget krunity-subtab-standard',
                'subclass' => 'standard',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 1
                ]
            ],
            'widget_standard_period_max' => [
                'title' => esc_html__('Maximum period', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 60,
                'class' => 'krunity-tab-widget krunity-subtab-standard',
                'subclass' => 'standard',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 1
                ]
            ],

            'widget_promotional_rate' => [
                'title' => esc_html__('Rate', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 24.0,
                'class' => 'krunity-tab-widget krunity-subtab-promotional',
                'subclass' => 'promotional',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 0
                ]
            ],
            'widget_promotional_min_amount' => [
                'title' => esc_html__('Minimum amount', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 15000,
                'class' => 'krunity-tab-widget krunity-subtab-promotional',
                'subclass' => 'promotional',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 0
                ]
            ],
            'widget_promotional_period_min' => [
                'title' => esc_html__('Minimum period', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 100000,
                'class' => 'krunity-tab-widget krunity-subtab-promotional',
                'subclass' => 'promotional',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 1
                ]
            ],
            'widget_promotional_period_max' => [
                'title' => esc_html__('Maximum period', 'xfactorapp-krunity-credit-smart-bt-direct'),
                'type' => 'number',
                'required' => true,
                'default' => 60,
                'class' => 'krunity-tab-widget krunity-subtab-promotional',
                'subclass' => 'promotional',
                'custom_attributes' => [
                    'step' => 0.01,
                    'min'  => 1
                ]
            ],

        ];

        $this->form_fields['debug'] = [
            'title'       => 'Debug',
            'type'        => 'raw',
            'class'       => 'krunity-tab-general',
            'description' => '
                    <button type="button" class="button button-secondary" id="krunity-debug-button" style="margin-top: 10px;">Test Connection</button>
                    <div id="krunity-debug-response" style="margin-top:10px;"></div>
                ',
        ];

        $this->update_option('title', Krunity::PLUGIN_TITLE);
    }


    private function load_shipping_method_options()
    {
        // Since this is expensive, we only want to do it if we're actually on the settings page.

        $data_store = WC_Data_Store::load('shipping-zone');
        $raw_zones = $data_store->get_zones();

        foreach ($raw_zones as $raw_zone) {
            $zones[] = new WC_Shipping_Zone($raw_zone);
        }

        $zones[] = new WC_Shipping_Zone(0);

        $options = [];
        foreach (WC()->shipping()->load_shipping_methods() as $method) {

            $options[$method->get_method_title()] = [];

            // Translators: %1$s shipping method name.
            $options[$method->get_method_title()][$method->id] = sprintf(__('Any &quot;%1$s&quot; method', 'xfactorapp-krunity-credit-smart-bt-direct'), $method->get_method_title());

            foreach ($zones as $zone) {

                $shipping_method_instances = $zone->get_shipping_methods();

                foreach ($shipping_method_instances as $shipping_method_instance_id => $shipping_method_instance) {

                    if ($shipping_method_instance->id !== $method->id) {
                        continue;
                    }

                    $option_id = $shipping_method_instance->get_rate_id();

                    // Translators: %1$s shipping method title, %2$s shipping method id.
                    $option_instance_title = sprintf(__('%1$s (#%2$s)', 'xfactorapp-krunity-credit-smart-bt-direct'), $shipping_method_instance->get_title(), $shipping_method_instance_id);

                    // Translators: %1$s zone name, %2$s shipping method instance name.
                    $option_title = sprintf(__('%1$s &ndash; %2$s', 'xfactorapp-krunity-credit-smart-bt-direct'), $zone->get_id() ? $zone->get_zone_name() : esc_html__('Other locations', 'xfactorapp-krunity-credit-smart-bt-direct'), $option_instance_title);

                    $options[$method->get_method_title()][$option_id] = $option_title;
                }
            }
        }

        return $options;
    }

    /**
     * Check If The Gateway Is Available For Use.
     *
     * @return bool
     */
    public function is_available()
    {
        $is_rest        = defined('REST_REQUEST') && REST_REQUEST;
        $is_ajax        = function_exists('wp_doing_ajax') && wp_doing_ajax();
        $is_checkout    = function_exists('is_checkout') && is_checkout();
        $is_pay_for_order = ( is_page( wc_get_page_id('checkout') ) && absint( get_query_var('order-pay') ) > 0 );

        if ( ! $is_checkout && ! $is_pay_for_order && ! $is_ajax && ! $is_rest ) {
            return false;
        }

        $order = null;
        $print_debug = $this->get_option('print_debug') === 'yes' && is_checkout() && $is_ajax;
        $minAmount = (float)$this->get_option('min_amount');

        //check by amount
        if (WC()->cart) {
            if (WC()->cart->get_cart_contents_total() < $minAmount) {
                if ($print_debug) wc_add_notice(
                    esc_html__( '[BTD] Nu este indeplinit minAmount', 'xfactorapp-krunity-credit-smart-bt-direct' ),
                    'error'
                );
                return false;
            }
        }
        if (is_page(wc_get_page_id('checkout')) && 0 < get_query_var('order-pay')) {
            $order_id = absint(get_query_var('order-pay'));
            $order = wc_get_order($order_id);
            if ($order->get_total() < $minAmount) {
                if ($print_debug) wc_add_notice(
                    esc_html__( '[BTD] Nu este indeplinita conditie de minAmount', 'xfactorapp-krunity-credit-smart-bt-direct' ),
                    'error'
                );
                return false;
            }
        }


        $needs_shipping = false;

        // Test if shipping is needed first.
        if (WC()->cart && WC()->cart->needs_shipping()) {
            $needs_shipping = true;
        } elseif (is_page(wc_get_page_id('checkout')) && 0 < get_query_var('order-pay')) {
            $order_id = absint(get_query_var('order-pay'));
            $order = wc_get_order($order_id);

            // Test if order needs shipping.
            if ($order && 0 < count($order->get_items())) {
                foreach ($order->get_items() as $item) {
                    $_product = $item->get_product();
                    if ($_product && $_product->needs_shipping()) {
                        $needs_shipping = true;
                        break;
                    }
                }
            }
        }

        $needs_shipping = apply_filters('woocommerce_cart_needs_shipping', $needs_shipping);

        // Virtual order, with virtual disabled.
        if (!$this->enable_for_virtual && !$needs_shipping) {
            if ($print_debug) wc_add_notice(
                esc_html__( '[BTD] Dezactivat pentru comenzi virtuale', 'xfactorapp-krunity-credit-smart-bt-direct' ),
                'error'
            );
            return false;
        }

        // Only apply if all packages are being shipped via chosen method, or order is virtual.
        if (!empty($this->enable_for_methods) && $needs_shipping) {
            $order_shipping_items = is_object($order) ? $order->get_shipping_methods() : false;
            $chosen_shipping_methods_session = WC()->session->get('chosen_shipping_methods');

            if ($order_shipping_items) {
                $canonical_rate_ids = $this->get_canonical_order_shipping_item_rate_ids($order_shipping_items);
            } else {
                $canonical_rate_ids = $this->get_canonical_package_rate_ids($chosen_shipping_methods_session);
            }

            if (!count($this->get_matching_rates($canonical_rate_ids))) {
                if ($print_debug) wc_add_notice(
                    esc_html__( '[BTD] Metoda de livrare aleasă nu permite această metodă de plată.', 'xfactorapp-krunity-credit-smart-bt-direct' ),
                    'error'
                );
                return false;
            }
        }

        return parent::is_available();
    }

    /**
     * Converts the chosen rate IDs generated by Shipping Methods to a canonical 'method_id:instance_id' format.
     *
     * @param array $order_shipping_items Array of WC_Order_Item_Shipping objects.
     * @return array $canonical_rate_ids    Rate IDs in a canonical format.
     * @since  3.4.0
     *
     */
    private function get_canonical_order_shipping_item_rate_ids($order_shipping_items)
    {

        $canonical_rate_ids = [];

        foreach ($order_shipping_items as $order_shipping_item) {
            $canonical_rate_ids[] = $order_shipping_item->get_method_id() . ':' . $order_shipping_item->get_instance_id();
        }

        return $canonical_rate_ids;
    }

    /**
     * Converts the chosen rate IDs generated by Shipping Methods to a canonical 'method_id:instance_id' format.
     *
     * @param array $chosen_package_rate_ids Rate IDs as generated by shipping methods. Can be anything if a shipping method doesn't honor WC conventions.
     * @return array $canonical_rate_ids  Rate IDs in a canonical format.
     * @since  3.4.0
     *
     */
    private function get_canonical_package_rate_ids($chosen_package_rate_ids)
    {

        $shipping_packages = WC()->shipping()->get_packages();
        $canonical_rate_ids = [];

        if (!empty($chosen_package_rate_ids) && is_array($chosen_package_rate_ids)) {
            foreach ($chosen_package_rate_ids as $package_key => $chosen_package_rate_id) {
                if (!empty($shipping_packages[$package_key]['rates'][$chosen_package_rate_id])) {
                    $chosen_rate = $shipping_packages[$package_key]['rates'][$chosen_package_rate_id];
                    $canonical_rate_ids[] = $chosen_rate->get_method_id() . ':' . $chosen_rate->get_instance_id();
                }
            }
        }

        return $canonical_rate_ids;
    }

    /**
     * Indicates whether a rate exists in an array of canonically-formatted rate IDs that activates this gateway.
     *
     * @param array $rate_ids Rate ids to check.
     * @return boolean
     * @since  3.4.0
     *
     */
    private function get_matching_rates($rate_ids)
    {
        // First, match entries in 'method_id:instance_id' format. Then, match entries in 'method_id' format by stripping off the instance ID from the candidates.
        return array_unique(array_merge(array_intersect($this->enable_for_methods, $rate_ids), array_intersect($this->enable_for_methods, array_unique(array_map('wc_get_string_before_colon', $rate_ids)))));
    }

    /**
     * Process the payment and return the result.
     *
     * @param int $order_id Order ID.
     * @return array
     */
    public function process_payment($order_id)
    {
        $order = wc_get_order($order_id);

        if ($order->get_total() > 0) {
            // Mark as processing or on-hold (payment won't be taken until delivery).
            #$order->update_status(apply_filters('woocommerce_cod_process_payment_order_status', $order->has_downloadable_item() ? 'on-hold' : 'processing', $order), esc_html__('Payment to be made upon delivery.', 'xfactorapp-krunity-credit-smart-bt-direct'));
            $order->update_status( \Automattic\WooCommerce\Enums\OrderStatus::PENDING, esc_html__('Awaiting payment.', 'xfactorapp-krunity-credit-smart-bt-direct') );
        } else {
            wc_maybe_reduce_stock_levels( $order_id ); // inofensiv daca a rulat deja
            $order->payment_complete();
        }

        // Remove cart.
        WC()->cart->empty_cart();

        // Return thankyou redirect.
        return [
            'result' => 'success',
            'redirect' => $this->get_return_url($order),
        ];
    }

    /**
     * Change payment complete order status to completed for COD orders.
     *
     * @param string $status Current order status.
     * @param int $order_id Order ID.
     * @param WC_Order|false $order Order object.
     * @return string
     * @since  3.1.0
     */
    public function change_payment_complete_order_status($status, $order_id = 0, $order = false)
    {
        if ($order && 'cod' === $order->get_payment_method()) {
            $status = 'completed';
        }
        return $status;
    }

    public function get_icon() {
        $src = $this->logo;
        return sprintf(
            '<img src="%s" alt="%s" style="height:30px;position:absolute;right:auto;margin-left:15px;" />',
            esc_url($src),
            esc_attr($this->get_title())
        );
    }


    protected function setup_properties()
    {
        #$logo = plugins_url(basename(plugin_dir_path(__DIR__)), basename(__DIR__)) . '/assets/img/logo_bt_1_h40.png';
        $icon = plugins_url(basename(plugin_dir_path(__DIR__)), basename(__DIR__)) . '/assets/img/logo.svg';
        $logo = plugins_url(basename(plugin_dir_path(__DIR__)), basename(__DIR__)) . '/assets/img/logo.png';

        $this->id = self::KEY;
        $this->icon = $icon;
        $this->logo = $logo;
//        $this->description = sprintf(
//            '<br>
//            <span>%s</span>
//            <br>
//            <img src="%s" alt="%s" style="max-width:150px;height:auto;" />',
//            esc_html(Krunity::PLUGIN_DESCRIPTION),
//            esc_url($logo),
//            esc_attr($this->get_title())
//        );
        $this->method_title = Krunity::PLUGIN_TITLE;
        $url = get_rest_url(null, 'krunity/v1');
        $this->method_description = Krunity::PLUGIN_DESCRIPTION . ' <br>Endpoint: <a target="_blank" href="' . $url . '">' . $url . '</a>';
        $this->has_fields = false;
        $this->supports = [ 'products', 'refunds' ];
    }

}
