<?php

namespace Krunity;

use WC_Order;

if ( ! defined( 'ABSPATH' ) ) exit;

class Krunity
{
    const PLUGIN_KEY = 'xfactorapp-krunity-credit-smart-bt-direct';
    const WC_PLUGIN_SLUG = 'wc-settings';

    const PLUGIN_TITLE='Credit Online Smart by BT Direct';
    const PLUGIN_DESCRIPTION='Plata 100% online prin Credit SMART (BT Direct)';

    public const SECURE_AUTH_KEY = 'krunity';
    public const ALLOWED_MARKUP_FOR_DEBUG = [
        'strong' => [],
        'center' => [],
        'p' => [],
        'br' => [],
        'pre' => [],
        'a' => ['href' => true, 'target' => true, 'rel' => true],
    ];

    public static function pre($obj, $return = false)
    {
        #$out = '<pre>' . wp_json_encode($obj, JSON_PRETTY_PRINT) . '</pre>';

        $out = '';
        if (is_array($obj)) {
            foreach ($obj as $k => $v) {
                if (is_scalar($v) || $v === null) {
                    $out .= '[' . $k . '] => ' . var_export($v, true) . '</br>';
                } else {
                    $out .= '[' . $k . '] => ' . wp_json_encode($v, JSON_PRETTY_PRINT) . '</br>';
                }
            }
        } else {
            $out = wp_json_encode($obj, JSON_PRETTY_PRINT);
        }

        //$out = '<pre>' . $out . '</pre>';

        if ($return) {
            return $out;
        }

        echo wp_kses($out, Krunity::ALLOWED_MARKUP_FOR_DEBUG);
    }

    public static function showAdminHeaderErrorMessage(string $message)
    {
        static::showAdminHeaderMessage($message, 'error');
    }

    public static function showAdminHeaderMessage(string $message, $type = 'error')
    {
        $type = is_string($type) ? strtolower($type) : 'error';
        switch ($type) {
            case 'success': $class = 'notice notice-success'; break;
            case 'warning': $class = 'notice notice-warning'; break;
            case 'info':    $class = 'notice notice-info';    break;
            case 'error':
            default:        $class = 'notice notice-error';   break;
        }

        add_action('admin_notices', function () use ($message, $class) {
            printf(
                '<div class="%1$s"><p>%2$s</p></div>',
                esc_attr($class),
                esc_html($message)
            );
        });
    }

    public static function isWooCommerceActive(): bool
    {
        return class_exists('woocommerce');
    }

    public static function isKrunityActive(): bool
    {
        return class_exists('Krunity\Krunity');
    }

    public static function addWPPluginLink($key, $title, $link)
    {
        add_filter('plugin_action_links', function ($plugin_actions, $plugin_file) use ($key, $title, $link) {
            $new_actions = [];
            // translators: 1: URL for the link, 2: Link text (title)
            $new_actions[$key] = sprintf(__('<a href="%1$s">%2$s</a>', 'xfactorapp-krunity-credit-smart-bt-direct'), $link, esc_html($title));
            return array_merge($new_actions, $plugin_actions);
        }, 10, 2);
    }

    public static function addWCPluginLink($key, $title, $tab = 'integration')
    {
        add_filter('plugin_action_links', function ($links) use ($key, $title, $tab) {
            $links[] = '<a href="' . menu_page_url(self::WC_PLUGIN_SLUG, false) . '&tab=' . $tab . '">' . $title . '</a>';
            return $links;
        });
    }

    public static function log($description, $extra = []) {
        $data = [
            'date' => current_time('mysql'),
            'description' => $description,
        ];

        $log = json_encode($data);
        if ($extra) {
            $debug = is_array($extra) && count($extra) > 0 ? json_encode($extra) : (is_string($extra) && strlen($extra) > 0 ? $extra : '');
            $log .= PHP_EOL . $debug;
        }

        if ( class_exists( 'WC_Logger' ) ) {
            $logger = wc_get_logger();
            $logger->debug( $log, [ 'source' => 'xfactorapp-krunity-credit-smart-bt-direct' ] );
        } else {
            error_log($log);
        }
    }

    public static function krunity_log_order($orderOrId, $description, $extra = []) {
        $order = $orderOrId;
        if (!$order instanceof WC_Order) {
            $order = wc_get_order($orderOrId);
        }

        $orderId = ($orderOrId instanceof WC_Order ? $orderOrId->get_id() : $orderOrId);

        if (!$order || !$order instanceof WC_Order) {
            if ( class_exists( 'WC_Logger' ) ) {
                $logger = wc_get_logger();
                $logger->debug( "Invalid order ID: $orderId", [ 'source' => 'xfactorapp-krunity-credit-smart-bt-direct' ] );
            }
            return;
        }

//        $extraMsg = '';
//        if ($extra && count($extra) > 0) {
//            $extraMsg .= '<br>';
//            foreach ($extra as $key => $value) {
//                $extraMsg .= ucfirst((string)$key) . ': ' . $value . '<br>';
//            }
//        }

        #$order->add_order_note('[Krunity] ' . $description . $extraMsg, false);

        $data = [
            'date' => current_time('mysql'),
            'description' => $description,
        ];
        if (isset($extra['status'])) {
            $data['status'] = $extra['status'];
        }
        if (isset($extra['amount'])) {
            $data['amount'] = $extra['amount'];
        }

        $order->add_meta_data(KRUNITY_META_STATUS_LOG, array_merge($data, [
            'debug' => is_array($extra) && count($extra) > 0 ? json_encode($extra) : (is_string($extra) && strlen($extra) > 0 ? $extra : ''),
        ]), false);

        $order->save();
        if ( class_exists( 'WC_Logger' ) ) {
            $logger = wc_get_logger();
            $debug = is_array($extra) && count($extra) > 0 ? json_encode($extra) : (is_string($extra) && strlen($extra) > 0 ? $extra : '');
            $logger->debug( "Saved log for order $orderId:" . json_encode($data) . PHP_EOL . $debug, [ 'source' => 'xfactorapp-krunity-credit-smart-bt-direct' ] );
        }
    }

    public static function postJson(string $href, array $data, ?string $token = null): ?array
    {
        $headers = [
            'Content-Type'  => 'application/json',
        ];

        if ( $token ) {
            $headers['Authorization'] = 'Bearer ' . $token;
        }

        $args = [
            'headers'   => $headers,
            'body'      => wp_json_encode( $data ),
            'timeout'   => 30,
            'sslverify' => true, // ⚠️ Only disable in dev — not safe in production
        ];

        $response = wp_remote_post( $href, $args );

        if ( is_wp_error( $response ) ) {
            $error_message = $response->get_error_message();
            $result = [
                'status'     => 'error',
                'message'    => $error_message,
                'errorCode'  => $response->get_error_code(),
                'content'    => '',
            ];

            if ( isset( $data['order_id'] ) ) {
                self::krunity_log_order( $data['order_id'], $href, [
                    'errorMessage' => $error_message,
                    'errorCode'    => $response->get_error_code(),
                    'content'      => '',
                    'contentType'  => 'none',
                ] );
            }

            return $result;
        }

        $content = wp_remote_retrieve_body( $response );

//        if (is_string($content)) {
//            return [
//                'status'     => 'error',
//                'content'    => $content,
//            ];
//        }

        try {
            $result = json_decode( $content, true, 512, JSON_THROW_ON_ERROR );
        } catch ( \Throwable $e ) {
            $result = [
                'status'     => 'error',
                'errorCode'  => 'json_decode_error',
                'content'    => $content,
            ];
        }

        return $result;

        /*
        $ch = curl_init($href);
        $headers = [
            "Content-Type: application/json"
        ];

        if ($token) {
            $headers[] = "Authorization: Bearer $token";
        }
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_URL, $href);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $content = curl_exec($ch);
        $curl_error = curl_errno( $ch );

        if ($curl_error) {
            $errorMessage = curl_error($ch);
            $result = ['status' => 'error', 'message' => $errorMessage, 'errorCode' => $curl_error, 'content' => $content];
            if (isset($data['order_id'])) {
                self::krunity_log_order($data['order_id'], $href, ['errorMessage' => curl_error($ch), 'errorCode' => $curl_error, 'content' => $content, 'contentType' => getType($content)]);
            }
        } else {
            try {
                $result = json_decode($content, true, 512, JSON_THROW_ON_ERROR);
            } catch (\Throwable $e) {
                $result = ['status' => 'error', 'errorCode' => $curl_error, 'content' => $content];
            }
        }

        curl_close($ch);
        return $result;*/
    }

    public static function initJWT()
    {
        // If some other plugin or a global autoloader already loaded firebase/php-jwt,
        // don’t try to load our bundled copy again.
        if (class_exists('\XFirebase\JWT\JWT')) {
            return;
        }

        // Use absolute paths so we’re not relying on the current working directory.
        $base = __DIR__ . '/vendor/firebase/php-jwt/src/';

        require_once $base . 'BeforeValidException.php';
        require_once $base . 'SignatureInvalidException.php';
        require_once $base . 'ExpiredException.php';
        require_once $base . 'Key.php';
        require_once $base . 'CachedKeySet.php';
        require_once $base . 'JWT.php';
    }

    public static function getJWTToken($token)
    {
        static::initJWT();

        $project = get_bloginfo('name');
        $issuer = $project; // this can be the servername
        $audience = $project;
        $issuedAt = time(); // issued at
        $notBefore = $issuedAt - 1; //not before in seconds
        $expireTime = 60 * 60 * 1; //o ora

        $expire = $issuedAt + $expireTime; // expire time in seconds
        $token["iat"] = $issuedAt;
        $token["exp"] = $expire;

        return \XFirebase\JWT\JWT::encode($token, Krunity::SECURE_AUTH_KEY, 'HS256');

    }

    public static function decodeJWTToken($token)
    {
        static::initJWT();
        try {
            str_replace('.', '', (string)$token, $count);
            if (!($count === 2 && strlen((string)$token) > 10)) {
                return null;
            }

            $decode = (array)\XFirebase\JWT\JWT::decode($token, new \XFirebase\JWT\Key(Krunity::SECURE_AUTH_KEY, 'HS256'));
            $decode['status'] = 'ok';
        } catch (\Exception $e) {
            $decode['status'] = 'error';
            $decode['error'] = $e->getMessage();

        }
        return $decode;
    }

    /**
     * Return a validated Bearer token or null.
     *
     * @param bool $allow_post_fallback If true, accept POST 'token' when header missing.
     * @param bool $allow_get_fallback  Strongly discouraged; only for legacy endpoints over HTTPS.
     */
    public static function getBearerToken(): ?string {
        $header = self::getAuthorizationHeader();
        if ( $header ) {
            // Reject header with CR/LF to prevent header-injection weirdness
            if ( preg_match('/[\r\n]/', $header) ) {
                return null;
            }
            if ( preg_match('/^Bearer\s+(.+)$/i', $header, $m) ) {
                $token = trim($m[1]);
                if ( self::is_valid_bearer_token( $token ) ) {
                    return $token;
                }
                return null;
            }
        }

        return null;
    }

    /**
     * Pull the Authorization header from server vars in a server-agnostic way.
     */
    public static function getAuthorizationHeader(): ?string {
        $header = null;

        // 1) Încearcă anteturile „normale”, fără superglobale
        if ( function_exists('getallheaders') ) {
            $headers = getallheaders();
            foreach ( $headers as $k => $v ) {
                if ( strtolower($k) === 'authorization' && ! empty($v) ) {
                    $header = $v;
                    break;
                }
            }
        }

        if ( null === $header && function_exists('apache_request_headers') ) {
            $headers = apache_request_headers();
            foreach ( $headers as $k => $v ) {
                if ( strtolower($k) === 'authorization' && ! empty($v) ) {
                    $header = $v;
                    break;
                }
            }
        }

        // 2) Fallback pe $_SERVER — UNSLASH + SANITIZE ca să placă PHPCS
        if ( null === $header && ! empty($_SERVER['HTTP_AUTHORIZATION']) ) {
            $header = sanitize_text_field( wp_unslash( (string) $_SERVER['HTTP_AUTHORIZATION'] ) );
        }

        if ( null === $header && ! empty($_SERVER['REDIRECT_HTTP_AUTHORIZATION']) ) {
            $header = sanitize_text_field( wp_unslash( (string) $_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ) );
        }

        if ( null === $header ) {
            return null;
        }

        // 3) Protecție suplimentară: respinge CR/LF și alte control chars
        if ( preg_match('/[\x00-\x1F\x7F]/', $header) ) {
            return null;
        }

        return trim( $header );
    }

    /**
     * Strict validation for common bearer formats without mutating the token.
     * - JWT: three base64url segments separated by dots.
     * - Opaque/Random: 20–512 alnum or URL-safe chars.
     * - Length cap to avoid resource abuse.
     */
    private static function is_valid_bearer_token( string $token ): bool {
        // Hard caps
        $len = strlen($token);
        if ( $len < 20 || $len > 4096 ) {
            return false;
        }

        // No whitespace or control chars
        if ( preg_match('/\s/', $token) ) {
            return false;
        }

        // Accept JWTs (base64url segments)
        if ( preg_match('/^[A-Za-z0-9\-_]+\.[A-Za-z0-9\-_]+\.[A-Za-z0-9\-_]+$/', $token) ) {
            return true;
        }

        // Accept opaque tokens (URL-safe, incl. dot, underscore, dash, tilde, plus, slash, equals for legacy base64)
        if ( preg_match('/^[A-Za-z0-9\.\_\-\~\+\/=]{20,}$/', $token) ) {
            return true;
        }

        return false;
    }

    /**
     * Mask secrets for logs (never log full tokens).
     */
    private static function mask_secret( string $secret ): string {
        $len = strlen($secret);
        if ( $len <= 8 ) {
            return str_repeat('•', $len);
        }
        return substr($secret, 0, 4) . '…' . substr($secret, -4);
    }


}