<?php
/*
Plugin Name: WTMLab-Meta
Plugin URI: https://wtmlab.com/wordpress/wtmlab-meta-plugin/
Description: WTMLab-Meta is a universal custom meta tags plugin by WTMLAB. It removes the default <title>, canonical link, and robots meta tags, and lets you set custom meta data for posts, pages, taxonomy terms, author pages, and archives. On any page type, if a custom meta title is provided it overrides the default title—and that title is used for the Open Graph and Twitter title tags; otherwise, the default title is used. Other fields (meta description, OG/Twitter descriptions, images, canonical URL, robots, and custom head code) fall back to safe defaults.
Version: 1.0
Author: WTMLAB
Author URI: https://wtmlab.com/
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: wtmlab-meta
Domain Path: /languages
Requires PHP: 7.0
*/

/* =============================================================================
   1. Remove Default Tags
============================================================================= */
/**
 * Remove WordPress’s default <title>, canonical link, and robots meta tag output.
 */
function wtmlab_meta_remove_default_tags() {
    remove_action( 'wp_head', '_wp_render_title_tag' );
    remove_action( 'wp_head', 'rel_canonical' );
    if ( function_exists( 'wp_robots' ) ) {
        remove_action( 'wp_head', 'wp_robots', 1 );
    }
}
add_action( 'after_setup_theme', 'wtmlab_meta_remove_default_tags', 100 );

/* =============================================================================
   2. Dynamic Title Filter for Singular Pages
============================================================================= */
/**
 * Override the document title on singular posts/pages.
 *
 * If a custom meta title (stored in _wtmlab_meta_title) is provided, it is used;
 * otherwise, the default post/page title is returned.
 */
function wtmlab_meta_pre_get_document_title( $title ) {
    if ( is_singular() ) {
        global $post;
        $custom_title = get_post_meta( $post->ID, '_wtmlab_meta_title', true );
        return ! empty( $custom_title ) ? $custom_title : get_the_title( $post->ID );
    }
    return $title;
}
add_filter( 'pre_get_document_title', 'wtmlab_meta_pre_get_document_title' );

/* =============================================================================
   3. Meta Box for Singular Posts/Pages
============================================================================= */
/**
 * Add a meta box for posts and pages.
 */
function wtmlab_meta_add_meta_box() {
    $post_types = array( 'post', 'page' );
    foreach ( $post_types as $pt ) {
        add_meta_box(
            'wtmlab_meta_box',
            esc_html__( 'WTMLab-Meta: Custom Meta Tags', 'wtmlab-meta' ),
            'wtmlab_meta_meta_box_html',
            $pt,
            'normal',
            'high'
        );
    }
}
add_action( 'add_meta_boxes', 'wtmlab_meta_add_meta_box' );

/**
 * Render the meta box HTML for posts/pages.
 */
function wtmlab_meta_meta_box_html( $post ) {
    wp_nonce_field( 'wtmlab_meta_save_meta_box', 'wtmlab_meta_box_nonce' );
    
    // Retrieve saved meta values.
    $meta_title          = get_post_meta( $post->ID, '_wtmlab_meta_title', true );
    $meta_description    = get_post_meta( $post->ID, '_wtmlab_meta_description', true );
    $og_description      = get_post_meta( $post->ID, '_wtmlab_meta_og_description', true );
    $twitter_description = get_post_meta( $post->ID, '_wtmlab_meta_twitter_description', true );
    $og_image            = get_post_meta( $post->ID, '_wtmlab_meta_og_image', true );
    $twitter_image       = get_post_meta( $post->ID, '_wtmlab_meta_twitter_image', true );
    $canonical_url       = get_post_meta( $post->ID, '_wtmlab_meta_canonical_url', true );
    $robots              = get_post_meta( $post->ID, '_wtmlab_meta_robots', true );
    $custom_code         = get_post_meta( $post->ID, '_wtmlab_meta_custom_code', true );
    ?>
    <p>
        <label for="wtmlab_meta_title"><strong><?php echo esc_html__( 'Custom Meta Title:', 'wtmlab-meta' ); ?></strong></label><br>
        <input type="text" name="wtmlab_meta_title" id="wtmlab_meta_title" value="<?php echo esc_attr( $meta_title ); ?>" style="width:100%;" placeholder="<?php echo esc_attr__( 'Overrides default title if provided', 'wtmlab-meta' ); ?>" />
    </p>
    <p>
        <label for="wtmlab_meta_description"><strong><?php echo esc_html__( 'Meta Description:', 'wtmlab-meta' ); ?></strong></label><br>
        <textarea name="wtmlab_meta_description" id="wtmlab_meta_description" rows="3" style="width:100%;" placeholder="<?php echo esc_attr__( 'Optional meta description (if empty, post content will be used)', 'wtmlab-meta' ); ?>"><?php echo esc_textarea( $meta_description ); ?></textarea>
    </p>
    <p>
        <label for="wtmlab_meta_og_description"><strong><?php echo esc_html__( 'OG Description:', 'wtmlab-meta' ); ?></strong></label><br>
        <textarea name="wtmlab_meta_og_description" id="wtmlab_meta_og_description" rows="3" style="width:100%;" placeholder="<?php echo esc_attr__( 'Optional OG description (if empty, will use meta description)', 'wtmlab-meta' ); ?>"><?php echo esc_textarea( $og_description ); ?></textarea>
    </p>
    <p>
        <label for="wtmlab_meta_twitter_description"><strong><?php echo esc_html__( 'Twitter Description:', 'wtmlab-meta' ); ?></strong></label><br>
        <textarea name="wtmlab_meta_twitter_description" id="wtmlab_meta_twitter_description" rows="3" style="width:100%;" placeholder="<?php echo esc_attr__( 'Optional Twitter description (if empty, will use meta description)', 'wtmlab-meta' ); ?>"><?php echo esc_textarea( $twitter_description ); ?></textarea>
    </p>
    <p>
        <label for="wtmlab_meta_og_image"><strong><?php echo esc_html__( 'OG Image URL:', 'wtmlab-meta' ); ?></strong></label><br>
        <input type="text" name="wtmlab_meta_og_image" id="wtmlab_meta_og_image" value="<?php echo esc_url( $og_image ); ?>" style="width:100%;" placeholder="<?php echo esc_attr__( 'Optional OG image URL (if empty, featured image will be used)', 'wtmlab-meta' ); ?>" />
    </p>
    <p>
        <label for="wtmlab_meta_twitter_image"><strong><?php echo esc_html__( 'Twitter Image URL:', 'wtmlab-meta' ); ?></strong></label><br>
        <input type="text" name="wtmlab_meta_twitter_image" id="wtmlab_meta_twitter_image" value="<?php echo esc_url( $twitter_image ); ?>" style="width:100%;" placeholder="<?php echo esc_attr__( 'Optional Twitter image URL (if empty, OG image will be used)', 'wtmlab-meta' ); ?>" />
    </p>
    <p>
        <label for="wtmlab_meta_canonical_url"><strong><?php echo esc_html__( 'Canonical URL:', 'wtmlab-meta' ); ?></strong></label><br>
        <input type="text" name="wtmlab_meta_canonical_url" id="wtmlab_meta_canonical_url" value="<?php echo esc_url( $canonical_url ); ?>" style="width:100%;" placeholder="<?php echo esc_attr__( 'Optional canonical URL (if empty, defaults to the post permalink)', 'wtmlab-meta' ); ?>" />
    </p>
    <p>
        <label><strong><?php echo esc_html__( 'Robots:', 'wtmlab-meta' ); ?></strong></label><br>
        <input type="checkbox" name="wtmlab_meta_robots_noindex" value="noindex" <?php checked( strpos( $robots, 'noindex' ) !== false ); ?> /> <?php echo esc_html__( 'No Index', 'wtmlab-meta' ); ?>&nbsp;
        <input type="checkbox" name="wtmlab_meta_robots_nofollow" value="nofollow" <?php checked( strpos( $robots, 'nofollow' ) !== false ); ?> /> <?php echo esc_html__( 'No Follow', 'wtmlab-meta' ); ?>
    </p>
    <p>
        <label for="wtmlab_meta_custom_code"><strong><?php echo esc_html__( 'Custom Head Code:', 'wtmlab-meta' ); ?></strong></label><br>
        <textarea name="wtmlab_meta_custom_code" id="wtmlab_meta_custom_code" rows="5" style="width:100%;" placeholder="<?php echo esc_attr__( 'Paste any custom code (e.g., Google Analytics, meta verification)', 'wtmlab-meta' ); ?>"><?php echo esc_textarea( $custom_code ); ?></textarea>
    </p>
    <?php
}

/* =============================================================================
   4. Save Meta Box Data for Singular Posts/Pages
============================================================================= */
function wtmlab_meta_save_meta_box_data( $post_id ) {
    if ( ! isset( $_POST['wtmlab_meta_box_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wtmlab_meta_box_nonce'] ) ), 'wtmlab_meta_save_meta_box' ) ) {
        return;
    }
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
        return;
    }
    if ( ! current_user_can( 'edit_post', $post_id ) ) {
        return;
    }
    if ( isset( $_POST['wtmlab_meta_title'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_title', sanitize_text_field( wp_unslash( $_POST['wtmlab_meta_title'] ) ) );
    }
    if ( isset( $_POST['wtmlab_meta_description'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_description', sanitize_textarea_field( wp_unslash( $_POST['wtmlab_meta_description'] ) ) );
    }
    if ( isset( $_POST['wtmlab_meta_og_description'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_og_description', sanitize_textarea_field( wp_unslash( $_POST['wtmlab_meta_og_description'] ) ) );
    }
    if ( isset( $_POST['wtmlab_meta_twitter_description'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_twitter_description', sanitize_textarea_field( wp_unslash( $_POST['wtmlab_meta_twitter_description'] ) ) );
    }
    if ( isset( $_POST['wtmlab_meta_og_image'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_og_image', esc_url_raw( wp_unslash( $_POST['wtmlab_meta_og_image'] ) ) );
    }
    if ( isset( $_POST['wtmlab_meta_twitter_image'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_twitter_image', esc_url_raw( wp_unslash( $_POST['wtmlab_meta_twitter_image'] ) ) );
    }
    if ( isset( $_POST['wtmlab_meta_canonical_url'] ) ) {
        update_post_meta( $post_id, '_wtmlab_meta_canonical_url', esc_url_raw( wp_unslash( $_POST['wtmlab_meta_canonical_url'] ) ) );
    }
    $robots = array();
    if ( isset( $_POST['wtmlab_meta_robots_noindex'] ) && 'noindex' === sanitize_text_field( wp_unslash( $_POST['wtmlab_meta_robots_noindex'] ) ) ) {
        $robots[] = 'noindex';
    }
    if ( isset( $_POST['wtmlab_meta_robots_nofollow'] ) && 'nofollow' === sanitize_text_field( wp_unslash( $_POST['wtmlab_meta_robots_nofollow'] ) ) ) {
        $robots[] = 'nofollow';
    }
    update_post_meta( $post_id, '_wtmlab_meta_robots', implode( ', ', $robots ) );
    
    if ( isset( $_POST['wtmlab_meta_custom_code'] ) ) {
        if ( current_user_can( 'unfiltered_html' ) ) {
            $custom_code = wp_unslash( $_POST['wtmlab_meta_custom_code'] );
        } else {
            $custom_code = wp_kses_post( wp_unslash( $_POST['wtmlab_meta_custom_code'] ) );
        }
        update_post_meta( $post_id, '_wtmlab_meta_custom_code', $custom_code );
    }
}
add_action( 'save_post', 'wtmlab_meta_save_meta_box_data' );

/* =============================================================================
   5. Output Meta Tags for Singular Posts/Pages (Priority 1)
============================================================================= */
function wtmlab_meta_output_singular_meta_tags() {
    if ( is_singular() ) {
        global $post;
        if ( ! $post ) return;
        
        $custom_title = get_post_meta( $post->ID, '_wtmlab_meta_title', true );
        $final_title  = ! empty( $custom_title ) ? $custom_title : get_the_title( $post->ID );
        
        $meta_description = get_post_meta( $post->ID, '_wtmlab_meta_description', true );
        if ( empty( $meta_description ) ) {
            $meta_description = $post->post_excerpt ? $post->post_excerpt : wp_trim_words( $post->post_content, 55, '...' );
            $meta_description = wp_strip_all_tags( $meta_description );
        }
        
        $og_description = get_post_meta( $post->ID, '_wtmlab_meta_og_description', true );
        if ( empty( $og_description ) ) {
            $og_description = $meta_description;
        }
        $twitter_description = get_post_meta( $post->ID, '_wtmlab_meta_twitter_description', true );
        if ( empty( $twitter_description ) ) {
            $twitter_description = $meta_description;
        }
        
        $og_image = get_post_meta( $post->ID, '_wtmlab_meta_og_image', true );
        if ( empty( $og_image ) && has_post_thumbnail( $post->ID ) ) {
            $og_image = get_the_post_thumbnail_url( $post->ID, 'full' );
        }
        $twitter_image = get_post_meta( $post->ID, '_wtmlab_meta_twitter_image', true );
        if ( empty( $twitter_image ) ) {
            $twitter_image = $og_image;
        }
        
        $canonical_url = get_post_meta( $post->ID, '_wtmlab_meta_canonical_url', true );
        if ( empty( $canonical_url ) ) {
            $canonical_url = get_permalink( $post->ID );
        }
        
        $robots = get_post_meta( $post->ID, '_wtmlab_meta_robots', true );
        
        $custom_code = get_post_meta( $post->ID, '_wtmlab_meta_custom_code', true );
        
        echo "\n<!-- WTMLab-Meta Output (Singular) Start -->\n";
        echo '<meta name="description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        if ( $keywords = get_post_meta( $post->ID, '_wtmlab_meta_keywords', true ) ) {
            echo '<meta name="keywords" content="' . esc_attr( $keywords ) . '">' . "\n";
        }
        if ( ! empty( $robots ) ) {
            echo '<meta name="robots" content="' . esc_attr( $robots ) . '">' . "\n";
        }
        echo '<meta property="og:title" content="' . esc_attr( $final_title ) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr( $og_description ) . '">' . "\n";
        if ( ! empty( $og_image ) ) {
            echo '<meta property="og:image" content="' . esc_url( $og_image ) . '">' . "\n";
        }
        echo '<meta property="og:url" content="' . esc_url( get_permalink( $post->ID ) ) . '">' . "\n";
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr( $final_title ) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr( $twitter_description ) . '">' . "\n";
        if ( ! empty( $twitter_image ) ) {
            echo '<meta name="twitter:image" content="' . esc_url( $twitter_image ) . '">' . "\n";
        }
        echo '<link rel="canonical" href="' . esc_url( $canonical_url ) . '">' . "\n";
        if ( ! empty( $custom_code ) ) {
            echo wp_kses_post( $custom_code ) . "\n";
        }
        echo "<!-- WTMLab-Meta Output (Singular) End -->\n";
    }
}
add_action( 'wp_head', 'wtmlab_meta_output_singular_meta_tags', 1 );

/* ============================================================================
   6. Output Meta Tags for Taxonomy Term Pages (Categories, Tags, etc.)
============================================================================= */
function wtmlab_meta_output_term_meta_tags() {
    if ( is_category() || is_tag() || is_tax() ) {
        $term = get_queried_object();
        if ( ! $term ) return;
        
        $meta_title = get_term_meta( $term->term_id, 'wtmlab_meta_title', true );
        if ( empty( $meta_title ) ) {
            $meta_title = $term->name;
        }
        $meta_description = get_term_meta( $term->term_id, 'wtmlab_meta_description', true );
        if ( empty( $meta_description ) ) {
            $meta_description = term_description( $term );
        }
        $og_description = get_term_meta( $term->term_id, 'wtmlab_meta_og_description', true );
        if ( empty( $og_description ) ) {
            $og_description = $meta_description;
        }
        $twitter_description = get_term_meta( $term->term_id, 'wtmlab_meta_twitter_description', true );
        if ( empty( $twitter_description ) ) {
            $twitter_description = $meta_description;
        }
        $og_image = get_term_meta( $term->term_id, 'wtmlab_meta_og_image', true );
        $twitter_image = get_term_meta( $term->term_id, 'wtmlab_meta_twitter_image', true );
        if ( empty( $twitter_image ) ) {
            $twitter_image = $og_image;
        }
        $canonical_url = get_term_meta( $term->term_id, 'wtmlab_meta_canonical_url', true );
        if ( empty( $canonical_url ) ) {
            $canonical_url = get_term_link( $term );
        }
        $robots = get_term_meta( $term->term_id, 'wtmlab_meta_robots', true );
        $custom_code = get_term_meta( $term->term_id, 'wtmlab_meta_custom_code', true );
        
        echo "\n<!-- WTMLab-Meta Output (Term) Start -->\n";
        echo '<title>' . esc_html( $meta_title ) . '</title>' . "\n";
        echo '<meta name="description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        if ( $keywords = get_term_meta( $term->term_id, 'wtmlab_meta_keywords', true ) ) {
            echo '<meta name="keywords" content="' . esc_attr( $keywords ) . '">' . "\n";
        }
        if ( ! empty( $robots ) ) {
            echo '<meta name="robots" content="' . esc_attr( $robots ) . '">' . "\n";
        }
        echo '<meta property="og:title" content="' . esc_attr( $meta_title ) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr( $og_description ) . '">' . "\n";
        if ( ! empty( $og_image ) ) {
            echo '<meta property="og:image" content="' . esc_url( $og_image ) . '">' . "\n";
        }
        echo '<meta property="og:url" content="' . esc_url( $canonical_url ) . '">' . "\n";
        // Add Twitter Card meta tags for taxonomy.
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr( $meta_title ) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr( $twitter_description ) . '">' . "\n";
        if ( ! empty( $twitter_image ) ) {
            echo '<meta name="twitter:image" content="' . esc_url( $twitter_image ) . '">' . "\n";
        }
        echo '<link rel="canonical" href="' . esc_url( $canonical_url ) . '">' . "\n";
        if ( ! empty( $custom_code ) ) {
            echo wp_kses_post( $custom_code ) . "\n";
        }
        echo "<!-- WTMLab-Meta Output (Term) End -->\n";
    }
}
add_action( 'wp_head', 'wtmlab_meta_output_term_meta_tags', 1 );

/* ============================================================================
   7. Output Meta Tags for Author Pages
============================================================================= */
function wtmlab_meta_output_author_meta_tags() {
    if ( is_author() ) {
        $author = get_queried_object();
        if ( ! $author ) return;
        
        $meta_title = get_user_meta( $author->ID, 'wtmlab_meta_title', true );
        if ( empty( $meta_title ) ) {
            $meta_title = $author->display_name;
        }
        $meta_description = get_user_meta( $author->ID, 'wtmlab_meta_description', true );
        if ( empty( $meta_description ) ) {
            $meta_description = get_the_author_meta( 'description', $author->ID );
        }
        $custom_code = get_user_meta( $author->ID, 'wtmlab_meta_custom_code', true );
        $canonical_url = get_author_posts_url( $author->ID );
        $robots = get_user_meta( $author->ID, 'wtmlab_meta_robots', true );
        
        echo "\n<!-- WTMLab-Meta Output (Author) Start -->\n";
        echo '<title>' . esc_html( $meta_title ) . '</title>' . "\n";
        echo '<meta name="description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        if ( $keywords = get_user_meta( $author->ID, 'wtmlab_meta_keywords', true ) ) {
            echo '<meta name="keywords" content="' . esc_attr( $keywords ) . '">' . "\n";
        }
        if ( ! empty( $robots ) ) {
            echo '<meta name="robots" content="' . esc_attr( $robots ) . '">' . "\n";
        }
        echo '<meta property="og:title" content="' . esc_attr( $meta_title ) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url( $canonical_url ) . '">' . "\n";
        // Twitter card for author.
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr( $meta_title ) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        echo '<link rel="canonical" href="' . esc_url( $canonical_url ) . '">' . "\n";
        if ( ! empty( $custom_code ) ) {
            echo wp_kses_post( $custom_code ) . "\n";
        }
        echo "<!-- WTMLab-Meta Output (Author) End -->\n";
    }
}
add_action( 'wp_head', 'wtmlab_meta_output_author_meta_tags', 1 );

/* ============================================================================
   8. Archive Meta Options Page
============================================================================= */
function wtmlab_meta_archive_meta_menu() {
    add_options_page( esc_html__( 'Archive Meta Tags', 'wtmlab-meta' ), esc_html__( 'Archive Meta Tags', 'wtmlab-meta' ), 'manage_options', 'wtmlab-meta-archive-meta', 'wtmlab_meta_archive_meta_options_page' );
}
add_action( 'admin_menu', 'wtmlab_meta_archive_meta_menu' );

function wtmlab_meta_archive_meta_settings_init() {
    register_setting(
        'wtmlab_meta_archive_options_group',
        'wtmlab_meta_archive_options',
        array(
            'type'              => 'array',
            'sanitize_callback' => 'wtmlab_meta_archive_options_sanitize',
        )
    );
    add_settings_section(
        'wtmlab_meta_archive_settings_section',
        esc_html__( 'Archive Meta Tags Settings', 'wtmlab-meta' ),
        'wtmlab_meta_archive_settings_section_callback',
        'wtmlab-meta-archive-meta'
    );
    add_settings_field(
        'wtmlab_meta_archive_meta_title',
        esc_html__( 'Meta Title', 'wtmlab-meta' ),
        'wtmlab_meta_archive_meta_title_render',
        'wtmlab-meta-archive-meta',
        'wtmlab_meta_archive_settings_section'
    );
    add_settings_field(
        'wtmlab_meta_archive_meta_description',
        esc_html__( 'Meta Description', 'wtmlab-meta' ),
        'wtmlab_meta_archive_meta_description_render',
        'wtmlab-meta-archive-meta',
        'wtmlab_meta_archive_settings_section'
    );
    add_settings_field(
        'wtmlab_meta_archive_custom_code',
        esc_html__( 'Custom Head Code', 'wtmlab-meta' ),
        'wtmlab_meta_archive_custom_code_render',
        'wtmlab-meta-archive-meta',
        'wtmlab_meta_archive_settings_section'
    );
}
add_action( 'admin_init', 'wtmlab_meta_archive_meta_settings_init' );

function wtmlab_meta_archive_settings_section_callback() {
    echo esc_html__( 'Set default meta tags for archive pages.', 'wtmlab-meta' );
}

function wtmlab_meta_archive_meta_title_render() {
    $options = get_option( 'wtmlab_meta_archive_options' );
    ?>
    <input type="text" name="wtmlab_meta_archive_options[meta_title]" value="<?php echo isset( $options['meta_title'] ) ? esc_attr( $options['meta_title'] ) : ''; ?>" style="width:50%;" />
    <?php
}

function wtmlab_meta_archive_meta_description_render() {
    $options = get_option( 'wtmlab_meta_archive_options' );
    ?>
    <textarea name="wtmlab_meta_archive_options[meta_description]" rows="3" cols="50"><?php echo isset( $options['meta_description'] ) ? esc_textarea( $options['meta_description'] ) : ''; ?></textarea>
    <?php
}

function wtmlab_meta_archive_custom_code_render() {
    $options = get_option( 'wtmlab_meta_archive_options' );
    ?>
    <textarea name="wtmlab_meta_archive_options[custom_code]" rows="5" cols="50" style="width:50%;"><?php echo isset( $options['custom_code'] ) ? esc_textarea( $options['custom_code'] ) : ''; ?></textarea>
    <?php
}

function wtmlab_meta_archive_meta_options_page() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html__( 'Archive Meta Tags', 'wtmlab-meta' ); ?></h1>
        <form action="options.php" method="post">
            <?php
            settings_fields( 'wtmlab_meta_archive_options_group' );
            do_settings_sections( 'wtmlab-meta-archive-meta' );
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

function wtmlab_meta_archive_options_sanitize( $input ) {
    $sanitized = array();
    if ( isset( $input['meta_title'] ) ) {
        $sanitized['meta_title'] = sanitize_text_field( wp_unslash( $input['meta_title'] ) );
    }
    if ( isset( $input['meta_description'] ) ) {
        $sanitized['meta_description'] = sanitize_textarea_field( wp_unslash( $input['meta_description'] ) );
    }
    if ( isset( $input['custom_code'] ) ) {
        if ( current_user_can( 'unfiltered_html' ) ) {
            $sanitized['custom_code'] = wp_unslash( $input['custom_code'] );
        } else {
            $sanitized['custom_code'] = wp_kses_post( wp_unslash( $input['custom_code'] ) );
        }
    }
    return $sanitized;
}

/* ============================================================================
   8. Output Meta Tags for Archive Pages (Priority 1)
============================================================================= */
function wtmlab_meta_output_archive_meta_tags() {
    if ( is_archive() && ! ( is_category() || is_tag() || is_tax() || is_author() ) ) {
        $options = get_option( 'wtmlab_meta_archive_options' );
        $meta_title = isset( $options['meta_title'] ) && ! empty( $options['meta_title'] ) ? $options['meta_title'] : get_the_archive_title();
        $meta_description = isset( $options['meta_description'] ) && ! empty( $options['meta_description'] ) ? $options['meta_description'] : get_bloginfo( 'description' );
        $custom_code = isset( $options['custom_code'] ) ? $options['custom_code'] : '';
        $canonical_url = isset( $options['canonical_url'] ) && ! empty( $options['canonical_url'] ) ? $options['canonical_url'] : get_permalink( get_option( 'page_for_posts' ) );
        $robots = isset( $options['robots'] ) ? $options['robots'] : '';
        
        echo "\n<!-- WTMLab-Meta Output (Archive) Start -->\n";
        echo '<title>' . esc_html( $meta_title ) . '</title>' . "\n";
        echo '<meta name="description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        if ( isset( $options['meta_keywords'] ) && ! empty( $options['meta_keywords'] ) ) {
            echo '<meta name="keywords" content="' . esc_attr( $options['meta_keywords'] ) . '">' . "\n";
        }
        if ( ! empty( $robots ) ) {
            echo '<meta name="robots" content="' . esc_attr( $robots ) . '">' . "\n";
        }
        echo '<meta property="og:title" content="' . esc_attr( $meta_title ) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url( $canonical_url ) . '">' . "\n";
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr( $meta_title ) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr( $meta_description ) . '">' . "\n";
        echo '<link rel="canonical" href="' . esc_url( $canonical_url ) . '">' . "\n";
        if ( ! empty( $custom_code ) ) {
            echo wp_kses_post( $custom_code ) . "\n";
        }
        echo "<!-- WTMLab-Meta Output (Archive) End -->\n";
    }
}
add_action( 'wp_head', 'wtmlab_meta_output_archive_meta_tags', 1 );
?>
