<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
  exit;
}

// Function to update user data via API
function writerx_update_user($request) {
  global $wpdb;

  $parameters = $request->get_json_params();

  // Validate required parameters
  if (!isset($parameters['user_id'])) {
    return new WP_Error(
      'missing_parameters',
      'Missing required parameter: user_id.',
      array('status' => 400)
    );
  }

  $user_id = intval($parameters['user_id']);
  $user = get_user_by('id', $user_id);

  if (!$user) {
    return new WP_Error(
      'user_not_found',
      'User not found.',
      array('status' => 404)
    );
  }

  // Update username if provided
  if (isset($parameters['username']) && $parameters['username'] !== $user->user_login) {
    $username = sanitize_user($parameters['username']);
    if (username_exists($username)) {
      return new WP_Error(
        'username_exists',
        'Username already exists.',
        array('status' => 409)
      );
    }

    // Direct SQL query to update user_login
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $update_result = $wpdb->update(
      $wpdb->users,
      array('user_login' => $username),
      array('ID' => $user_id)
    );

    if ($update_result === false) {
      return new WP_Error(
        'db_update_failed',
        'Failed to update username in the database.',
        array('status' => 500)
      );
    }

    // Update nickname and display name
    update_user_meta($user_id, 'nickname', $username);
    wp_update_user(array(
      'ID' => $user_id,
      'display_name' => $username,
    ));
  }

  // Update email if provided
  if (isset($parameters['email']) && $parameters['email'] !== $user->user_email) {
    $email = sanitize_email($parameters['email']);
    if (email_exists($email)) {
      return new WP_Error(
        'email_exists',
        'Email already exists.',
        array('status' => 409)
      );
    }
    wp_update_user(array('ID' => $user_id, 'user_email' => $email));
  }

  // Update bio if provided
  if (isset($parameters['bio'])) {
    $bio = sanitize_textarea_field($parameters['bio']);
    update_user_meta($user_id, 'description', $bio);
  }

  // Handle avatar update if provided
  if (isset($parameters['avatar_url'])) {
    $avatar_url = esc_url_raw($parameters['avatar_url']);
    $avatar_id = writerx_upload_avatar($avatar_url);
    if (!is_wp_error($avatar_id)) {
      update_user_meta($user_id, 'custom_avatar', $avatar_id);
    } else {
      return new WP_Error(
        'avatar_upload_failed',
        'Failed to upload avatar: ' . $avatar_id->get_error_message(),
        array('status' => 400)
      );
    }
  }

  // Return success status
  return array(
    'status' => 'success',
  );
}
