<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
  exit;
}

// Function to create a new user via API
function writerx_create_new_user($request) {
  $parameters = $request->get_json_params();

  // Validate required parameters
  if (!isset($parameters['username']) || !isset($parameters['email'])) {
    return new WP_Error(
      'missing_parameters',
      'Missing required parameters: username and email.',
      array('status' => 400)
    );
  }

  $username = sanitize_user($parameters['username']);
  $email = sanitize_email($parameters['email']);
  $password = wp_generate_password(20, true, true); // Generate a strong password
  $bio = isset($parameters['bio']) ? sanitize_textarea_field($parameters['bio']) : '';
  $avatar_url = isset($parameters['avatar_url']) ? esc_url_raw($parameters['avatar_url']) : '';

  // Check if username or email already exists
  if (username_exists($username) || email_exists($email)) {
    return new WP_Error(
      'user_exists',
      'Username or email already exists.',
      array('status' => 409)
    );
  }

  // Create new user
  $user_id = wp_create_user($username, $password, $email);

  if (is_wp_error($user_id)) {
    return new WP_Error(
      'user_creation_failed',
      'Error creating user: ' . $user_id->get_error_message(),
      array('status' => 500)
    );
  }

  // Set user role to "author"
  $user = new WP_User($user_id);
  $user->set_role('author');

  // Update bio if provided
  if ($bio) {
    update_user_meta($user_id, 'description', $bio);
  }

  // Handle avatar upload if provided
  if ($avatar_url) {
    $avatar_id = writerx_upload_avatar($avatar_url);
    if (!is_wp_error($avatar_id)) {
      update_user_meta($user_id, 'custom_avatar', $avatar_id);
    } else {
      return new WP_Error(
        'avatar_upload_failed',
        'Failed to upload avatar: ' . $avatar_id->get_error_message(),
        array('status' => 400)
      );
    }
  }

  return array(
    'user_id' => $user_id,
  );
}

// Function to upload an avatar from URL
function writerx_upload_avatar($url) {
  // Validate URL
  if (!filter_var($url, FILTER_VALIDATE_URL)) {
    return new WP_Error('invalid_url', 'Invalid avatar URL.');
  }

  // Check if the file is accessible
  $response = wp_remote_head($url);
  if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
    return new WP_Error('unreachable_url', 'Avatar URL is not accessible.');
  }

  // Download file to temp directory
  $tmp_file = download_url($url);

  if (is_wp_error($tmp_file)) {
    return new WP_Error('download_failed', 'Failed to download avatar.');
  }

  // Get file name and ensure it's an image
  $file_array = array();
  $file_array['name'] = basename($url);
  $file_array['tmp_name'] = $tmp_file;

  $file_type = wp_check_filetype($file_array['name']);
  if (!in_array($file_type['type'], array('image/jpeg', 'image/png', 'image/gif'))) {
    @wp_delete_file($tmp_file); // Delete temp file if invalid
    return new WP_Error('invalid_file_type', 'Uploaded file is not a valid image.');
  }

  // Upload the file
  $attachment_id = media_handle_sideload($file_array, 0);

  // If there was an error uploading the file, delete temp file and return error
  if (is_wp_error($attachment_id)) {
    @wp_delete_file($file_array['tmp_name']);
    return $attachment_id;
  }

  // Get the image editor
  $image_path = get_attached_file($attachment_id);
  $editor = wp_get_image_editor($image_path);

  if (is_wp_error($editor)) {
    @wp_delete_file($file_array['tmp_name']);
    return new WP_Error('image_editor_error', 'Failed to create image editor.');
  }

  // Crop the image to square (150x150)
  $editor->resize(150, 150, true); // Crop to exact size
  $editor->save($image_path); // Save the image

  return $attachment_id;
}

// Filter to override the avatar URL with a custom avatar
add_filter('get_avatar_url', 'writerx_custom_avatar_url', 10, 3);
function writerx_custom_avatar_url($url, $id_or_email, $args) {
  $user = false;

  if (is_numeric($id_or_email)) {
    $user = get_user_by('id', $id_or_email);
  } elseif (is_object($id_or_email) && !empty($id_or_email->user_id)) {
    $user = get_user_by('id', $id_or_email->user_id);
  } elseif (is_string($id_or_email)) {
    $user = get_user_by('email', $id_or_email);
  }

  if ($user) {
    $avatar_id = get_user_meta($user->ID, 'custom_avatar', true);
    if ($avatar_id) {
      $custom_avatar_url = wp_get_attachment_url($avatar_id);
      if ($custom_avatar_url) {
        return $custom_avatar_url;
      }
    }
  }

  return $url;
}
