<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://wptrivo.com/
 * @since      1.0.0
 *
 * @package    Wptrivo_Gift_Cards_Lite
 * @subpackage Wptrivo_Gift_Cards_Lite/public
 */


/**
 * Exit if accessed directly
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Wptrivo_Gift_Cards_Lite
 * @subpackage Wptrivo_Gift_Cards_Lite/public
 * @author     wptrivo <dev@wptrivo.com>
 */
class Wptgcl_Wptrivo_Gift_Cards_Lite_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

		add_shortcode( 'wptgcl_gift_card_products', array($this,'wptgcl_gift_cards_prdct_list_cb') );

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Wptgcl_Wptrivo_Gift_Cards_Lite_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Wptgcl_Wptrivo_Gift_Cards_Lite_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( 'jquery-ui-theme', WPTGCL_PLUGIN_DIR_URL . 'vendor/wptrivo-gift-cards-lite-jquery-ui.css', array(), $this->version );
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/wptrivo-gift-cards-lite-public.css', array('jquery-ui-theme'), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Wptgcl_Wptrivo_Gift_Cards_Lite_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Wptgcl_Wptrivo_Gift_Cards_Lite_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class. 
		 */

		wp_enqueue_script('jquery-ui-datepicker');
		wp_enqueue_script( 'wc-select2', WPTGCL_PLUGIN_DIR_URL . 'vendor/wptrivo-gift-cards-lite-select2.js', array(), $this->version, false );
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/wptrivo-gift-cards-lite-public.js', array( 'jquery','jquery-ui-datepicker', 'wp-i18n','wc-select2' ), $this->version, false );

		wp_localize_script(
				$this->plugin_name,
				'wptgcl_public_ajax_obj',
				array(
					'ajax_url'             => admin_url( 'admin-ajax.php' ),
					'nonce'                => wp_create_nonce( 'ajax-nonce' ),
				)
		);
		wp_set_script_translations( $this->plugin_name, 'wptrivo-gift-cards-lite' );

	}

	/**
	 * Show gift card data if product is gift card.
	 * @since    1.0.0
	 * 
	 */
	public function wptgcl_display_gift_card_data_in_cart($item_data, $cart_item) {
		if (isset($cart_item['wptgcl_from_name'])) {
			$item_data[] = array(
				'name' => __('From', 'wptrivo-gift-cards-lite'),
				'value' => $cart_item['wptgcl_from_name']
			);
		}
		
		if (isset($cart_item['wptgcl_to_email'])) {
			$item_data[] = array(
				'name' => __('To', 'wptrivo-gift-cards-lite'),
				'value' => $cart_item['wptgcl_to_email']
			);
		}
		
		if (isset($cart_item['wptgcl_message']) && !empty($cart_item['wptgcl_message'])) {
			$item_data[] = array(
				'name' => __('Message', 'wptrivo-gift-cards-lite'),
				'value' => $cart_item['wptgcl_message']
			);
		}
		
		if (isset($cart_item['wptgcl_send_date']) && !empty($cart_item['wptgcl_send_date'])) {
			$item_data[] = array(
				'name' => __('Delivery Date', 'wptrivo-gift-cards-lite'),
				'value' => wp_date('F j, Y', strtotime($cart_item['wptgcl_send_date']))
			);
		}
		
		return $item_data;
	}

		/**
	 * Enable gift card product purchasable.
	 * @since    1.0.0
	 * 
	 */
	public function wptgcl_set_gift_card_prdct_purchasable( $purchasable, $product){

		if ( $product->get_type() === 'wptgcl_gift_card' ) {
			$purchasable = true;
		}
    	return $purchasable;
	}

	/**
	 * Set price to the gift card products.
	 * @since    1.0.0
	 * 
	 */
	public function wptgcl_set_gift_card_prdct_price( $price, $product ) {
		if ( $product->get_type() === 'wptgcl_gift_card' ) {
			$pricing_type = $product->get_meta('_wptgcl_pricing_type');

			switch ( $pricing_type ) {
				case 'fixed':
					$price = wc_price( $product->get_meta('_wptgcl_fixed_price') );
					break;

				case 'range':
					$min = $product->get_meta('_wptgcl_min_price');
					$max = $product->get_meta('_wptgcl_max_price');
					$price = wc_price( $min ) . ' - ' . wc_price( $max );
					break;

				case 'variable':
					$values = array_map( 'trim', explode(',', $product->get_meta('_wptgcl_variable_prices') ) );
					$price = __( 'From ', 'wptrivo-gift-cards-lite' ) . wc_price( min($values) );
					break;

				case 'default':
					$price = wc_price( $product->get_meta('_wptgcl_gift_card_value_default') );
					break;
				case 'custom':
					$price = __( 'Set your own amount', 'wptrivo-gift-cards-lite' );
					break;
			}
		}
    	return $price;
	}

	/**
	 * Add to Cart support for the gift card products.
	 * @since    1.0.0
	 * 
	 */
	public function gift_card_product_supports($supports, $feature, $product) {
		if ($product && $product->get_type() == 'wptgcl_gift_card') {
			switch ($feature) {
				case 'ajax_add_to_cart':
					$supports = true;
					break;
			}
		}
		return $supports;
	}

	/**
	 * Add template redirection for the gift card products.
	 * @since    1.0.0
	 * 
	 */
	public function wptgcl_gift_card_add_to_cart_template() {
		// Use the simple product template as base.
		wc_get_template('single-product/add-to-cart/simple.php');
	}

	/**
	 * Add Gift card form before the add to card button.
	 * @since    1.0.0
	 * 
	 */
	public function wptgcl_add_gift_card_frontend_fields() {
		global $product;
		$gnrl_opt_settings = get_option('wptgcl_general_settings_opt', array());
		if( isset( $gnrl_opt_settings['enable_gift_crds_settng'] ) && empty( $gnrl_opt_settings['enable_gift_crds_settng'] ) ){
			return;

		}
		
		if ($product && $product->get_type() == 'wptgcl_gift_card') {
			$pricing_type = get_post_meta($product->get_id(), '_wptgcl_pricing_type', true);
			$min_price = get_post_meta($product->get_id(), '_wptgcl_min_price', true);
			$max_price = get_post_meta($product->get_id(), '_wptgcl_max_price', true);
			$variable_prices = get_post_meta($product->get_id(), '_wptgcl_variable_prices', true);
			$default_value = get_post_meta($product->get_id(), '_wptgcl_gift_card_value_default', true);
			$wptgcl_read_only = '';
			if( $pricing_type == 'default' ){
				$wptgcl_read_only = 'readonly';
			}
			
			?>
			<div class="wptgcl_gift_card_wrapper" id="wptgcl_gift_card_fields">
				<?php wp_nonce_field( 'wptgcl_add_gift_card_to_cart_action', 'wptgcl_gift_card_nonce' ); ?>
				<input type="hidden" id="wptgcl_price_type_frm" value="<?php echo esc_attr($pricing_type)?>">
				<?php if ($pricing_type == 'range' || $pricing_type == 'custom' || $pricing_type == 'default') : ?>
				<p class="wptgcl_section">
					<label for="wptgcl_price"><?php esc_html_e('Enter Price', 'wptrivo-gift-cards-lite'); ?>
						<?php if ($pricing_type == 'range') : ?>
							(<?php echo wp_kses_post( wc_price($min_price) . ' - ' . wc_price($max_price) ); ?>):
						<?php else : ?>:
						<?php endif; ?>
					</label>
					<input type="number" 
						class="input-text wptgcl_price" 
						id="wptgcl_price" 
						name="wptgcl_price" 
						value="<?php echo esc_attr($pricing_type == 'range' ? $min_price : $default_value); ?>"
						<?php if ($pricing_type == 'range') : ?>
						min="<?php echo esc_attr($min_price); ?>" 
						max="<?php echo esc_attr($max_price); ?>"
						<?php endif; ?>
						step="1" 
						required <?php echo esc_attr( $wptgcl_read_only ) ?>>
						<span class="wptgcl_frm_error"></span>
				</p>
				<?php endif; ?>
				
				<?php if ($pricing_type == 'variable' && !empty($variable_prices)) : ?>
				<p class="wptgcl_section">
					<label for="wptgcl_variable_price"><?php esc_html_e('Select Price:', 'wptrivo-gift-cards-lite'); ?></label>
					<select class="wptgcl_variable_price" id="wptgcl_variable_price" name="wptgcl_price" required>
						<option value=""><?php esc_html_e('Choose an amount', 'wptrivo-gift-cards-lite'); ?></option>
						<?php 
						$prices = explode(',', $variable_prices);
						foreach ($prices as $price) {
							$price = trim($price);
							if (!empty($price)) {
								echo '<option value="' . esc_attr($price) . '">' . wp_kses_post( wc_price( $price ) ) . '</option>';
							}
						}
						?>
					</select>
					<span class="wptgcl_frm_error"></span>
				</p>
				<?php endif; ?>
				
				<!-- Delivery Date -->
				<p class="wptgcl_section wptgcl_date_style">
					<label for="wptgcl_send_date"><?php esc_html_e('Select Delivery Date:', 'wptrivo-gift-cards-lite'); ?></label>
					<input type="text" 
						name="wptgcl_send_date" 
						id="wptgcl_send_date" 
						class="wptgcl_send_date" 
						placeholder="<?php esc_html_e('Select date', 'wptrivo-gift-cards-lite'); ?>" 
						readonly>
					<span class="wptgcl_info"><?php esc_html_e('(Recipient will receive the gift card on selected date)', 'wptrivo-gift-cards-lite'); ?></span>
				</p>
				
				<!-- Sender Name -->
				<p class="wptgcl_section">
					<label class="wptgcl_from_label" for="wptgcl_from_name"><?php esc_html_e('From:', 'wptrivo-gift-cards-lite'); ?></label>
					<input type="text" 
						name="wptgcl_from_name" 
						id="wptgcl_from_name" 
						class="wptgcl_from_name" 
						placeholder="<?php esc_html_e('Enter the sender name', 'wptrivo-gift-cards-lite'); ?>" 
						required>
						<span class="wptgcl_frm_error"></span>
				</p>
				
				<!-- Gift Message -->
				<p class="wptgcl_section wptgcl_msg_field">
					<label for="wptgcl_message"><?php esc_html_e('Gift Message:', 'wptrivo-gift-cards-lite'); ?></label>
					<textarea name="wptgcl_message" 
							id="wptgcl_message" 
							class="wptgcl_message" 
							maxlength="100" 
							placeholder="<?php esc_html_e('Enter your gift message', 'wptrivo-gift-cards-lite'); ?>"></textarea>
					<span class="wptgcl_char_count"><?php esc_html_e('Characters:', 'wptrivo-gift-cards-lite'); ?> (<span id="wptgcl_char">0</span>/<?php echo isset($email_settings['cstm_msg_mail_len']) ? intval($email_settings['cstm_msg_mail_len']) : 100; ?>)</span>
				</p>
				
				<!-- Delivery Method -->
				<div class="wptgcl_section">
					<label class="wptgcl_delivery_method_label"><?php esc_html_e('Delivery Method:', 'wptrivo-gift-cards-lite'); ?></label>
					
					<div class="wptgcl_delivery_method">
						<input type="radio" 
							name="wptgcl_send_giftcard" 
							value="Mail to recipient" 
							class="wptgcl_send_giftcard" 
							checked="checked" 
							id="wptgcl_to_mail">
						<label for="wptgcl_to_mail" class="wptgcl_method"><?php esc_html_e('Email To Recipient', 'wptrivo-gift-cards-lite'); ?></label>
						
						<div class="wptgcl_delivery_via_email">
							<input type="email" 
								name="wptgcl_to_email" 
								id="wptgcl_to_email" 
								class="wptgcl_to_email" 
								placeholder="<?php esc_html_e('Enter the Recipient Email (Required)', 'wptrivo-gift-cards-lite'); ?>" 
								required>
							<input type="text" 
								name="wptgcl_to_name_optional" 
								id="wptgcl_to_name_optional" 
								class="wptgcl_to_name" 
								placeholder="<?php esc_html_e('Enter the Recipient Name (Optional)', 'wptrivo-gift-cards-lite'); ?>">
							<span class="wptgcl_msg_info"><?php esc_html_e('We will send it to recipient email address.', 'wptrivo-gift-cards-lite'); ?></span>
						</div>
						
						<!-- Optional: Add "Email to me" option -->
						<input type="radio" 
							name="wptgcl_send_giftcard" 
							value="Mail to me" 
							class="wptgcl_send_giftcard" 
							id="wptgcl_to_me">
						<label for="wptgcl_to_me" class="wptgcl_method"><?php esc_html_e('Email To Me', 'wptrivo-gift-cards-lite'); ?></label>
					</div>
				</div>
				
				<!-- Gift Card Templates -->
				<div class="wptgcl_section">
					<label class="wptgcl_template_label"><?php esc_html_e('Select Template:', 'wptrivo-gift-cards-lite'); ?></label>
					<div class="wptgcl_template_selection" id="wptgcl_template_selection">
						<?php echo wp_kses_post( wptgcl_get_gift_card_templates_for_product($product->get_id()) ); ?>
					</div>
					<input name="wptgcl_selected_temp" id="wptgcl_selected_temp" value="" type="hidden">
				</div>
				
				<!-- Preview Link -->
				<p class="wptgcl_preview_section">
					<a id="wptgcl_preview_gift_card" href="javascript:void(0);" class="wptgcl_preview_link">
						<?php esc_html_e('Preview Gift Card', 'wptrivo-gift-cards-lite'); ?>
					</a>
				</p>
			</div>
			
			<?php
		}
	}

	/**
	 * Add gift card form data to the cart.
	 * @since    1.0.0
	 * 
	 */
	public function wptgcl_add_gift_card_data_to_cart($cart_item_data, $product_id, $variation_id) {
		$product = wc_get_product($product_id);
		if ($product && $product->get_type() == 'wptgcl_gift_card') {
			if ( ! isset( $_POST['wptgcl_gift_card_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['wptgcl_gift_card_nonce'] ) ), 'wptgcl_add_gift_card_to_cart_action' )) {
				wc_add_notice( __( 'Security verification failed. Please try again.', 'wptrivo-gift-cards-lite' ), 'error' );
				return false;
			}
			
			$max_message_length =  100;
			
			if (isset($_POST['wptgcl_price'])) {
				$cart_item_data['wptgcl_price'] = sanitize_text_field( wp_unslash( $_POST['wptgcl_price'] ) );
			}
			if (isset($_POST['wptgcl_send_date'])) {
				$cart_item_data['wptgcl_send_date'] = sanitize_text_field( wp_unslash( $_POST['wptgcl_send_date'] ) );
			}
			if (isset($_POST['wptgcl_from_name'])) {
				$cart_item_data['wptgcl_from_name'] = sanitize_text_field( wp_unslash( $_POST['wptgcl_from_name'] ));
			}
			if (isset($_POST['wptgcl_message'])) {
				$message = sanitize_textarea_field( wp_unslash( $_POST['wptgcl_message'] ) );
				// Trim message if it exceeds max length dev-email@wpengine.local
				if (strlen($message) > $max_message_length) {
					$message = substr($message, 0, $max_message_length);
				}
				$cart_item_data['wptgcl_message'] = $message;
			}
			if (isset($_POST['wptgcl_send_giftcard'])) {
				$cart_item_data['wptgcl_send_giftcard'] = sanitize_text_field( wp_unslash( $_POST['wptgcl_send_giftcard'] ) );
			}
			$user = wp_get_current_user();
			if( $_POST['wptgcl_send_giftcard'] == 'Mail to me'){
				$recipient_eamil = $user->user_email;
			}else{
				$recipient_eamil = isset( $_POST['wptgcl_to_email'] ) ? sanitize_email( wp_unslash( $_POST['wptgcl_to_email'] ) ) : '';
			}
			if (isset($_POST['wptgcl_to_email'])) {
				$cart_item_data['wptgcl_to_email'] = $recipient_eamil;
			}
			if (isset($_POST['wptgcl_to_name_optional'])) {
				$cart_item_data['wptgcl_to_name_optional'] = sanitize_text_field( wp_unslash( $_POST['wptgcl_to_name_optional'] ) );
			}
			if (isset($_POST['wptgcl_selected_temp'])) {
				$cart_item_data['wptgcl_selected_temp'] = sanitize_text_field( wp_unslash( $_POST['wptgcl_selected_temp'] ) );
			}
			
			// Make each gift card unique in cart
			$cart_item_data['wptgcl_unique_key'] = md5(microtime().wp_rand());
		}
		
		return $cart_item_data;
	}

	/**
	 * Set gift card price before the total on the cart page.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_update_gift_card_price_cart($cart) {
		if (is_admin() && !defined('DOING_AJAX')) return;
		
		foreach ($cart->get_cart() as $cart_item) {
			if (isset($cart_item['wptgcl_price']) && !empty($cart_item['wptgcl_price'])) {
				$cart_item['data']->set_price($cart_item['wptgcl_price']);
			}
		}
	}

	/**
	 * Show preview of the gift card template.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_handle_preview_gift_card() {
		if ( ! isset( $_POST['nonce'] ) || !wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'ajax-nonce') ) {
			wp_send_json_success(array('status'=> false, 'msg'=> 'Security check failed'));
		}
		
		$template_id = isset($_POST['template_id'] ) ? intval($_POST['template_id']): '';
		$message = isset($_POST['message'] ) ? sanitize_textarea_field( wp_unslash( $_POST['message'] ) ): '';
		$from_name = isset($_POST['from_name'] ) ? sanitize_text_field( wp_unslash( $_POST['from_name'] ) ):'';
		$to_name = isset($_POST['to_name'] ) ? sanitize_text_field( wp_unslash( $_POST['to_name'] ) ):'';
		$price = isset($_POST['price'] ) ? floatval($_POST['price'] ): 0.00;
		
		// Generate preview HTML
		$preview_html = wptgcl_generate_gift_card_preview($template_id, $message, $from_name, $to_name, $price);
		
		wp_send_json_success(array('html' => $preview_html));
	}

	/**
	 * Add gift card data to the checkout page.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_save_gift_card_data_to_order($item, $cart_item_key, $values, $order) {

		if (isset($values['wptgcl_price'])) {
			$item->add_meta_data('_wptgcl_price', $values['wptgcl_price']);
		}
		if (isset($values['wptgcl_send_date'])) {
			$item->add_meta_data('_wptgcl_send_date', $values['wptgcl_send_date']);
		}
		if (isset($values['wptgcl_from_name'])) {
			$item->add_meta_data('_wptgcl_from_name', $values['wptgcl_from_name']);
		}
		if (isset($values['wptgcl_message'])) {
			$item->add_meta_data('_wptgcl_message', $values['wptgcl_message']);
		}
		if (isset($values['wptgcl_send_giftcard'])) {
			$item->add_meta_data('_wptgcl_send_giftcard', $values['wptgcl_send_giftcard']);
		}
		if (isset($values['wptgcl_to_email'])) {
			$item->add_meta_data('_wptgcl_to_email', $values['wptgcl_to_email']);
		}
		if (isset($values['wptgcl_to_name_optional'])) {
			$item->add_meta_data('_wptgcl_to_name_optional', $values['wptgcl_to_name_optional']);
		}
		if (isset($values['wptgcl_selected_temp'])) {
			$item->add_meta_data('_wptgcl_selected_temp', $values['wptgcl_selected_temp']);
		}
	}

	/**
	 * Generate coupon code on order processed to completed.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_create_gift_card_on_order_complete($order_id) {
		$order = wc_get_order($order_id);
    	if (!$order) return;
    
		foreach ($order->get_items() as $item_id => $item) {
				$product = $item->get_product();
	
			if ($product && $product->get_type() == 'wptgcl_gift_card') {
				// Check if gift card already created for this item
				$existing_code = wc_get_order_item_meta($item_id, '_wptgcl_gift_card_code', true);
				if ($existing_code) {
					continue; // Skip if already created
				}
				
				// Generate gift card code
				$gift_card_code = wptgcl_generate_gift_card_code();
				
				// Get gift card data from order item meta
				$gift_card_amount = wc_get_order_item_meta($item_id, '_wptgcl_price', true);
				if (!$gift_card_amount) {
					$gift_card_amount = $item->get_total();
				}
				
				$from_name       = wc_get_order_item_meta($item_id, '_wptgcl_from_name', true);
				$to_email        = wc_get_order_item_meta($item_id, '_wptgcl_to_email', true);
				$to_name         = wc_get_order_item_meta($item_id, '_wptgcl_to_name_optional', true);
				$message         = wc_get_order_item_meta($item_id, '_wptgcl_message', true);
				$send_date       = wc_get_order_item_meta($item_id, '_wptgcl_send_date', true);
				$template_id     = wc_get_order_item_meta($item_id, '_wptgcl_selected_temp', true);
				$delivery_method = wc_get_order_item_meta($item_id, '_wptgcl_send_giftcard', true);

				// Create WooCommerce coupon for the gift card
				$coupon_data = array(
					'post_title'   => $gift_card_code,
					'post_content' => '',
					'post_status'  => 'publish',
					'post_author'  => 1,
					'post_type'    => 'shop_coupon'
				);
				
				$coupon_id = wp_insert_post($coupon_data);
				
				if ($coupon_id) {
					// Base coupon meta
					update_post_meta($coupon_id, 'coupon_amount', $gift_card_amount);
					
					$general_settings = get_option('wptgcl_general_settings_opt', array());
					$discount_type = 'fixed_cart';
					update_post_meta($coupon_id, 'discount_type', $discount_type);
					// Usage Limit
					$usage_limit = !empty($general_settings['gift_card_usage_limit']) ? intval($general_settings['gift_card_usage_limit']) : 1;
					update_post_meta($coupon_id, 'usage_limit', $usage_limit);

					// Usage Limit Per User
					$usage_limit_per_user = !empty($general_settings['gift_card_usage_limit_per_user']) ? intval($general_settings['gift_card_usage_limit_per_user']) : 1;
					update_post_meta($coupon_id, 'usage_limit_per_user', $usage_limit_per_user);

					// Limit usage to X items
					$limit_usage_to_x_items = isset($general_settings['gift_card_limit_usage_x_items']) ? intval($general_settings['gift_card_limit_usage_x_items']) : '';
					update_post_meta($coupon_id, 'limit_usage_to_x_items', $limit_usage_to_x_items);
					// Free Shipping
					$free_shipping = !empty($general_settings['gift_card_free_shipping']) ? 'yes' : 'no';
					update_post_meta($coupon_id, 'free_shipping', $free_shipping);

					// Enable Individual Use
					if (!empty($general_settings['enable_gift_crds_indivdul'])) {
						update_post_meta($coupon_id, 'individual_use', 'yes');
					} else {
						update_post_meta($coupon_id, 'individual_use', 'no');
					}

					// Shipping and Tax
					if (!empty($general_settings['apply_gft_crd_shppng_tx'])) {
						update_post_meta($coupon_id, 'apply_to_shipping', 'yes'); // no user of the 
					} else {
						update_post_meta($coupon_id, 'apply_to_shipping', 'no');
					}

					// Min & Max Spend
					if (!empty($general_settings['gift_card_min_spnd'])) {
						update_post_meta($coupon_id, 'minimum_amount', floatval($general_settings['gift_card_min_spnd']));
					}
					if (!empty($general_settings['gift_card_max_spnd'])) {
						update_post_meta($coupon_id, 'maximum_amount', floatval($general_settings['gift_card_max_spnd']));
					}

					// Enable Tax Calculation (custom flag for later use)
					if (!empty($general_settings['enable_tax_cal_settng'])) {
						update_post_meta($coupon_id, '_wptgcl_enable_tax_calculation', 'yes');
					} else {
						update_post_meta($coupon_id, '_wptgcl_enable_tax_calculation', 'no');
					}

					// Expiry date based on settings
					$expiry_days = isset($general_settings['wptgcl_gift_crds_exp_dys']) ? intval($general_settings['wptgcl_gift_crds_exp_dys']) : 0;
					if ($expiry_days > 0) {
						$expiry_date = wp_date('Y-m-d', strtotime("+{$expiry_days} days"));
						update_post_meta($coupon_id, 'date_expires', strtotime($expiry_date));
					}

					// Handle product/category exclusions based on settings
					$product_settings = get_option('wptgcl_product_settings_opt', array());

					if (isset($product_settings['exclude_sale_itms']) && $product_settings['exclude_sale_itms'] == 1) {
						update_post_meta($coupon_id, 'exclude_sale_items', 'yes');
					}

					if ( ! empty( $product_settings['wptgcl_excld_products'] ) ) {
						$excluded_products = array_map( 'intval', (array) $product_settings['wptgcl_excld_products'] );
						update_post_meta( $coupon_id, 'exclude_product_ids', $excluded_products );
					}

					if ( ! empty( $product_settings['wptgcl_excld_products_cat'] ) ) {
						$excluded_categories = array_map( 'intval', (array) $product_settings['wptgcl_excld_products_cat'] );
						update_post_meta( $coupon_id, 'exclude_product_categories', $excluded_categories );
					}

					// Save gift card specific meta
					update_post_meta($coupon_id, '_wptgcl_is_gift_card', 'yes');
					update_post_meta($coupon_id, '_wptgcl_coupon_code', $gift_card_code);
					update_post_meta($coupon_id, '_wptgcl_gift_card_amount', $gift_card_amount);
					update_post_meta($coupon_id, '_wptgcl_from_name', $from_name);
					update_post_meta($coupon_id, '_wptgcl_to_email', $to_email);
					update_post_meta($coupon_id, '_wptgcl_to_name', $to_name);
					update_post_meta($coupon_id, '_wptgcl_message', $message);
					update_post_meta($coupon_id, '_wptgcl_send_date', $send_date);
					update_post_meta($coupon_id, '_wptgcl_template_id', $template_id);
					update_post_meta($coupon_id, '_wptgcl_delivery_method', $delivery_method);
					update_post_meta($coupon_id, '_wptgcl_order_id', $order_id);
					update_post_meta($coupon_id, '_wptgcl_created_date', current_time('mysql'));
					
					wc_update_order_item_meta($item_id, '_wptgcl_gift_card_code', $gift_card_code);
					wc_update_order_item_meta($item_id, '_wptgcl_coupon_id', $coupon_id);

					$timestamp = 0;
					if (!empty($send_date)) {
						$datetime = new DateTime($send_date, wp_timezone());
						$datetime->setTime(9, 0); 
						$timestamp = $datetime->getTimestamp();
					}
					if ($timestamp > current_time('timestamp')) {
						if (!wp_next_scheduled('wptgcl_send_gift_card_email', array($coupon_id))) {
							wp_schedule_single_event($timestamp, 'wptgcl_send_gift_card_email', array($coupon_id));
						}
					} else {
						wptgcl_send_gift_card_email($coupon_id);
					}
				}
			}
		}
	}

	/**
	 * Send mail with the selected template after order status change to completed.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_send_scheduled_gift_card_emails_cb( $coupon_id ) {
		wptgcl_send_gift_card_email( $coupon_id );
		
	}

	/**
	 * Track gift card's coupon usage.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_track_gift_card_usage( $coupon_code ){

		$coupon = new WC_Coupon( $coupon_code );
		$coupon_id = $coupon->get_id();
		
		// Check if it's a gift card
		if (get_post_meta($coupon_id, '_wptgcl_is_gift_card', true) != 'yes') {
			return;
		}
		
		// Get current cart total that will be discounted
		$cart_total = WC()->cart->get_subtotal();
		$original_amount = get_post_meta($coupon_id, '_wptgcl_gift_card_amount', true);
		$discount_amount = min($original_amount, $cart_total);
		$remaining_amount = $original_amount - $discount_amount;
		
		// Store usage info (will be processed after order completion)
		WC()->session->set('wptgcl_used_gift_card_' . $coupon_id, array(
			'coupon_id' => $coupon_id,
			'used_amount' => $discount_amount,
			'remaining_amount' => $remaining_amount,
			'original_amount' => $original_amount
		));
	}

	/**
	 * Track and update coupon usages.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_process_gift_card_usage( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) return;

		foreach ( $order->get_coupon_codes() as $coupon_code ) {
			$coupon = new WC_Coupon( $coupon_code );
			$coupon_id = $coupon->get_id();

			// Only process gift cards
			if ( get_post_meta( $coupon_id, '_wptgcl_is_gift_card', true ) !== 'yes' ) {
				continue;
			}

			// Current gift card info
			$original_amount    = (float) get_post_meta( $coupon_id, '_wptgcl_gift_card_amount', true );
			$used_amount_so_far = (float) get_post_meta( $coupon_id, '_wptgcl_used_amount', true );
			$remaining_amount   = (float) get_post_meta( $coupon_id, '_wptgcl_remaining_amount', true );

			// Calculate discount applied for THIS coupon only
			$current_discount = 0;
			foreach ( $order->get_items( 'coupon' ) as $coupon_item ) {
				if ( strtolower( $coupon_item->get_code() ) === strtolower( $coupon_code ) ) {
					$current_discount += (float) $coupon_item->get_discount();
					$current_discount += (float) $coupon_item->get_discount_tax(); // include tax if coupon applied to tax
				}
			}

			//Update usage
			$new_used_amount      = $used_amount_so_far + $current_discount;
			$new_remaining_amount = max( 0, $original_amount - $new_used_amount );

			update_post_meta( $coupon_id, '_wptgcl_used_amount', $new_used_amount );
			update_post_meta( $coupon_id, '_wptgcl_remaining_amount', $new_remaining_amount );
			update_post_meta( $coupon_id, '_wptgcl_last_used_order', $order_id );
			update_post_meta( $coupon_id, '_wptgcl_last_used_date', current_time( 'mysql' ) );

			//Handle remaining balance
			if ( $new_remaining_amount > 0 ) {
				update_post_meta( $coupon_id, 'coupon_amount', $new_remaining_amount );

			} else {
				// Fully used → disable coupon
				update_post_meta( $coupon_id, 'usage_limit', 0 );
				wp_update_post( array(
					'ID'          => $coupon_id,
					'post_status' => 'draft',
				) );
			}
		}
	}

	/**
	 * shortcode callback for the listing fo the gift card products.
	 * @since    1.0.0
	 * 
	*/
	public function wptgcl_gift_cards_prdct_list_cb( $atts ) {
		$atts = shortcode_atts( array(
			'limit'   => 12,        
			'columns' => 4,         
			'orderby' => 'date',    
			'order'   => 'DESC',    
			'img_size' => 'medium', 
		), $atts, 'wptgc_gift_cards' );

		$args = array(
			'post_type'      => 'product',
			'posts_per_page' => intval( $atts['limit'] ),
			'orderby'        => sanitize_text_field( $atts['orderby'] ),
			'order'          => sanitize_text_field( $atts['order'] ),
			'tax_query'      => array( // phpcs:ignore
				array(
					'taxonomy' => 'product_type',
					'field'    => 'slug',
					'terms'    => array( 'wptgcl_gift_card' ),
				),
			),
		);

		$products = new WP_Query( $args );

		ob_start();

		if ( $products->have_posts() ) {
			echo '<ul class="products columns-' . intval( $atts['columns'] ) . '" id="wptgcl_prdct_shrtcode">';

			while ( $products->have_posts() ) {
				$products->the_post();
				global $product;

				echo '<li class="product">';

				echo '<a href="' . esc_url( get_the_permalink() ) . '">';
				echo wp_kses_post( woocommerce_get_product_thumbnail( $atts['img_size'] ) );
				echo '</a>';

				// Product Title
				echo '<h2 class="woocommerce-loop-product__title">' . esc_html( get_the_title() ) . '</h2>';

				// Product Price
				echo '<span class="price">' . wp_kses_post( $product->get_price_html() ) . '</span>';
				echo '</li>';
			}

			echo '</ul>';
		} else {
			echo '<p>' . esc_html__( 'No gift cards found.', 'wptrivo-gift-cards-lite' ) . '</p>';
		}

		wp_reset_postdata();

		return ob_get_clean();
	}

}
