<?php

/**
 * Fired during plugin activation
 *
 * @link       http://wptrivo.com/
 * @since      1.0.0
 *
 * @package    Wptrivo_Gift_Cards_Lite
 * @subpackage Wptrivo_Gift_Cards_Lite/includes
 */

/**
 * Exit if accessed directly
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Create default templates on plugin activation
 *
 * @since    1.0.0
 */
function wptgcl_create_default_templates() {
    $default_templates = array(
        array(
            'title' => 'Birthday Template',
            'type' => 'birthday',
            'is_default' => true
        ),
        array(
            'title' => 'General Template',
            'type' => 'general',
            'is_default' => false
        ),
    );

    foreach ($default_templates as $template_data) {
        $template_defaults = wptgcl_get_default_template_by_type($template_data['type']);
        
        $post_data = array(
            'post_title' => $template_data['title'],
            'post_type' => 'wptgcl_template',
            'post_status' => 'publish',
            'post_content' => $template_defaults['content'],
            'post_excerpt' => 'Default ' . $template_data['type'] . ' gift card template'
        );

        $post_id = wp_insert_post($post_data);

        if ($post_id) {
            switch ($template_data['type']) {
                case 'birthday':
                    $default_image_path = WPTGCL_PLUGIN_DIR_PATH . 'admin/assets/images/birth_day.jpg';
                    break;
                case 'general':
                    $default_image_path = WPTGCL_PLUGIN_DIR_PATH. 'admin/assets/images/bst_wishes.jpg';
                    break;
            }
            
            $attach_id = wptgcl_get_or_upload_default_image( $default_image_path );

            if ( $attach_id ) {
                set_post_thumbnail( $post_id, $attach_id );
            }
            update_post_meta($post_id, '_wptgcl_template_type', $template_data['type']);
            update_post_meta($post_id, '_wptgcl_is_active', 'yes');
            update_post_meta($post_id, '_wptgcl_is_default', $template_data['is_default'] ? 'yes' : 'no');
            update_post_meta($post_id, '_wptgcl_template_html', $template_defaults['content']);
        }
    }
}


/**
 * Get default template by type
 *
 * @since    1.0.0
 */
function wptgcl_get_default_template_by_type($type) {
    $templates = array(
        'birthday' => array(
            'content' => '<style>
                @page { margin: 20px }
            </style>

            <div style="max-width: 600px; margin: auto; font-family: "Helvetica Neue", Arial, sans-serif;">

                <!-- Header -->
                <table style="margin: auto; width: 100%; background: linear-gradient(135deg, #ff8a00, #e52e71);" border="0" cellspacing="0" cellpadding="0" align="center">
                    <tr>
                        <td style="text-align: center; padding: 30px 15px;">
                            <h1 style="margin: 0; font-size: 28px; color: #fff; font-weight: bold;">🎉 Happy Birthday! 🎂</h1>
                            <p style="margin: 5px 0 0; font-size: 16px; color: #fff;">A special gift just for you</p>
                        </td>
                    </tr>
                </table>

                <!-- Hero Image -->
                <table style="width: 100%; background: #ffffff;" border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <td style="text-align: center;">
                            <img style="width: 100%; max-width: 600px;" src="{featured_image}" alt="Birthday Celebration" />
                        </td>
                    </tr>
                </table>

                <!-- Gift Card Box -->
                <table style="margin: 20px auto; width: 90%; border-radius: 12px; background: #fff0f6; border: 2px dashed #e52e71;" border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <td style="padding: 20px; text-align: center;">
                            <p style="margin: 0; font-size: 18px; font-weight: bold; color: #e52e71;">Your Gift Card Code</p>
                            <p style="margin: 10px 0; font-size: 28px; font-weight: bold; color: #333;">{coupon_code}</p>
                            <p style="margin: 0; font-size: 14px; color: #666;">Valid until: {exp_date}</p>
                        </td>
                    </tr>
                </table>

                <!-- Message & Details -->
                <table style="width: 100%; background: #ffffff; margin: 0 auto;" border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <td style="padding: 20px; text-align: center;">
                            <p style="font-size: 18px; color: #444; margin: 0 0 20px;">{message}</p>
                            <p style="margin: 0; font-size: 16px; color: #000;"><strong>From:</strong> {from_name}</p>
                            <p style="margin: 5px 0 20px; font-size: 16px; color: #000;"><strong>To:</strong> {to_name}</p>
                            <p style="margin: 0; font-size: 26px; font-weight: bold; color: #ff8a00;">{amount}</p>
                        </td>
                    </tr>
                </table>

                <!-- Decorative Divider -->
                <table style="width: 100%; margin: 20px 0;" border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <td style="border-top: 2px dotted #ff8a00;"></td>
                    </tr>
                </table>

                <!-- Footer / Disclaimer -->
                <table style="width: 100%; background: #e52e71;" border="0" cellspacing="0" cellpadding="0">
                    <tr>
                        <td style="text-align: center; padding: 20px; font-size: 14px; color: #fff;">
                            <br><br>
                            <em>Enjoy your day with love, laughter, and surprises! 💝</em>
                        </td>
                    </tr>
                </table>

            </div>',
        ),
        'general' => array(
            'content' => '<style>
                    @page { margin: 20px }
                </style>

                <div style="max-width: 600px; margin: auto; font-family: "Helvetica Neue", Arial, sans-serif;">

                    <!-- Header -->
                    <table style="margin: auto; width: 100%; background: #2c3e50;" border="0" cellspacing="0" cellpadding="0" align="center">
                        <tr>
                            <td style="text-align: center; padding: 25px 15px;">
                                <h1 style="margin: 0; font-size: 26px; color: #ecf0f1; font-weight: bold;">🎁 A Special Gift for You</h1>
                                <p style="margin: 5px 0 0; font-size: 15px; color: #bdc3c7;">Because you deserve something wonderful</p>
                            </td>
                        </tr>
                    </table>

                    <!-- Hero Image -->
                    <table style="width: 100%; background: #ffffff;" border="0" cellspacing="0" cellpadding="0">
                        <tr>
                            <td style="text-align: center;">
                                <img style="width: 100%; max-width: 600px;" src="{featured_image}" alt="Gift Card" />
                            </td>
                        </tr>
                    </table>

                    <!-- Gift Card Box -->
                    <table style="margin: 20px auto; width: 90%; border-radius: 10px; background: #f9f9f9; border: 2px dashed #2c3e50;" border="0" cellspacing="0" cellpadding="0">
                        <tr>
                            <td style="padding: 20px; text-align: center;">
                                <p style="margin: 0; font-size: 18px; font-weight: bold; color: #2c3e50;">Your Gift Card Code</p>
                                <p style="margin: 10px 0; font-size: 28px; font-weight: bold; color: #16a085;">{coupon_code}</p>
                                <p style="margin: 0; font-size: 14px; color: #666;">Valid until: {exp_date}</p>
                            </td>
                        </tr>
                    </table>

                    <!-- Message & Details -->
                    <table style="width: 100%; background: #ffffff; margin: 0 auto;" border="0" cellspacing="0" cellpadding="0">
                        <tr>
                            <td style="padding: 20px; text-align: center;">
                                <p style="font-size: 17px; color: #444; margin: 0 0 20px;">{message}</p>
                                <p style="margin: 0; font-size: 15px; color: #000;"><strong>From:</strong> {from_name}</p>
                                <p style="margin: 5px 0 20px; font-size: 15px; color: #000;"><strong>To:</strong> {to_name}</p>
                                <p style="margin: 0; font-size: 24px; font-weight: bold; color: #2c3e50;">{amount}</p>
                            </td>
                        </tr>
                    </table>

                    <!-- Divider -->
                    <table style="width: 100%; margin: 20px 0;" border="0" cellspacing="0" cellpadding="0">
                        <tr>
                            <td style="border-top: 2px solid #ecf0f1;"></td>
                        </tr>
                    </table>

                    <!-- Footer / Disclaimer -->
                    <table style="width: 100%; background: #2c3e50;" border="0" cellspacing="0" cellpadding="0">
                        <tr>
                            <td style="text-align: center; padding: 15px; font-size: 13px; color: #ecf0f1;">
                                <br><br>
                                <em>This gift card can be redeemed anytime during the validity period.</em>
                            </td>
                        </tr>
                    </table>

                </div>',
        ),
    );

    return $templates[$type] ?? $templates['general'];
}

/**
 * Upload the default image to media library (if not uploaded before)
 * 
 * @since    1.0.0
 */
function wptgcl_get_or_upload_default_image( $file_path ) {
    $file_name = basename( $file_path );

    // Use WP_Query to check if the file is already in the Media Library
    $query = new WP_Query( array(
        'post_type'      => 'attachment',
        'title'          => sanitize_file_name( $file_name ),
        'post_status'    => 'inherit',
        'posts_per_page' => 1,
        'fields'         => 'ids',
    ) );

    if ( $query->have_posts() ) {
        return $query->posts[0]; // Return existing attachment ID
    }

    // Otherwise, upload new file
    if ( file_exists( $file_path ) ) {
        $upload_file = wp_upload_bits( $file_name, null, file_get_contents( $file_path ) );

        if ( ! $upload_file['error'] ) {
            $wp_filetype = wp_check_filetype( $file_name, null );
            $attachment  = array(
                'post_mime_type' => $wp_filetype['type'],
                'post_title'     => sanitize_file_name( $file_name ),
                'post_content'   => '',
                'post_status'    => 'inherit'
            );
            $attach_id = wp_insert_attachment( $attachment, $upload_file['file'] );

            require_once( ABSPATH . 'wp-admin/includes/image.php' );
            $attach_data = wp_generate_attachment_metadata( $attach_id, $upload_file['file'] );
            wp_update_attachment_metadata( $attach_id, $attach_data );

            return $attach_id;
        }
    } 

    return false;
}

/**
 * Callback of ajax to show the preview of the templets.
 *
 * @since    1.0.0
 */
function wptgcl_generate_gift_card_preview($template_id, $message, $from_name, $to_name, $price) {
    // Get template content from CPT
    $template_post = get_post($template_id);

    if (!$template_post || $template_post->post_type !== 'wptgcl_template') {
        return '<p style="color:red;">Invalid gift card template selected.</p>';
    }

   $wptgcl_disclaimer = '';
    $content = $template_post->post_content;

    // Define placeholders → values
    $replacements = array(
        '{message}'    => esc_html($message),
        '{from_name}'  => esc_html($from_name),
        '{to_name}'    => esc_html($to_name),
        '{amount}'     => wc_price($price),
        '{coupon_code}'=> 'XXXX-XXXX', // default if not generated yet
        '{LOGO}'       => '<img src="' . WPTGCL_PLUGIN_DIR_URL. 'admin/assets/images/logo.png" alt="Logo" style="max-width:150px;">',
        '{featured_image}' => get_the_post_thumbnail_url($template_id, 'large') ?: plugin_dir_url(__FILE__) . 'assets/images/default-featured.jpg',
        '{exp_date}' => date_i18n(get_option('date_format'), strtotime('+1 year')), // example expiry
        '{disclamer}' => $wptgcl_disclaimer,
    );

    // Replace placeholders
    $final_html = strtr($content, $replacements);

    return $final_html;
}

/**
 * Send emails on the date selection.
 *
 * @since    1.0.0
 */
function wptgcl_send_gift_card_email($coupon_id) {
    $general_settings = get_option('wptgcl_general_settings_opt', array());
    
    // Check if gift cards are enabled
    if (!isset($general_settings['enable_gift_crds_settng']) || $general_settings['enable_gift_crds_settng'] != 1) {
        return;
    }
    
    // Get gift card data
    $gift_card_code = get_post_meta($coupon_id, '_wptgcl_coupon_code', true);
    $amount = get_post_meta($coupon_id, '_wptgcl_gift_card_amount', true);
    $from_name = get_post_meta($coupon_id, '_wptgcl_from_name', true);
    $to_email = get_post_meta($coupon_id, '_wptgcl_to_email', true);
    $to_name = get_post_meta($coupon_id, '_wptgcl_to_name', true);
    $message = get_post_meta($coupon_id, '_wptgcl_message', true);
    $delivery_method = get_post_meta($coupon_id, '_wptgcl_delivery_method', true);
    $template_id = get_post_meta($coupon_id, '_wptgcl_template_id', true);
    $order_id = get_post_meta($coupon_id, '_wptgcl_order_id', true);

    $to_name = $to_name ?: 'Valued Customer';
    
    // Get order for customer email if needed
    $order = wc_get_order($order_id);
    $customer_email = $order ? $order->get_billing_email() : '';
    
    // Determine recipient email
    $recipient_email = '';
    if ($delivery_method == 'Mail to recipient' && !empty( $to_email )) {
        $recipient_email = $to_email;
    } elseif (!empty($customer_email)) {
        $recipient_email = $customer_email;
    }
    
    if (empty($recipient_email)) {
        return; // No email to send to
    }
    
    // Email subject
    $subject = isset($general_settings['gift_crd_email_sub']) && !empty($general_settings['gift_crd_email_sub']) 
        ? $general_settings['gift_crd_email_sub'] 
        : 'Your Gift Card is Ready!';
    
    $body_data = [
        'amount' => $amount,
        'message' => $message,
        'coupon_code' => $gift_card_code,
        'from_name' => $from_name,
        'to_name' => $to_name,
    ];
    
    // Complete email HTML
    $email_html = wptgcl_render_giftcard_template( $template_id, $body_data );
    // Send email
    $headers = array('Content-Type: text/html; charset=UTF-8');
    $sent = wp_mail($recipient_email, $subject, $email_html, $headers);
    
    // Log email sending
    if ($sent) {
        update_post_meta($coupon_id, '_wptgcl_email_sent', 'yes');
        update_post_meta($coupon_id, '_wptgcl_email_sent_date', current_time('mysql'));
    }
    
    return $sent;
}

/**
 * This method used to replace the template's shortcodes.
 *
 * @since    1.0.0
 */
function wptgcl_render_giftcard_template( $template_id, $data = [] ) {
    $post = get_post( $template_id );
    if ( ! $post || $post->post_type !== 'wptgcl_template' ) {
        return '';
    }
    $thumbnail = get_the_post_thumbnail_url($template_id, 'thumbnail');
    if (!$thumbnail) {
            $thumbnail = WPTGCL_PLUGIN_DIR_URL . 'admin/assets/images/default-template.png';
    }

    $wptgc_disclaimer =  '';
   
    $coupon = new WC_Coupon( $data['coupon_code'] );
    $coupon_id = $coupon->get_id();
    
    $expiry_timestamp = get_post_meta($coupon_id, 'date_expires', true) ; 
    if ( $expiry_timestamp ) {
        $expiry_date = date_i18n( get_option('date_format'), $expiry_timestamp );
    } else {
        $expiry_date = __( 'Infinite', 'wptrivo-gift-cards-lite' );
    }
    // Get template content
    $content = $post->post_content;
    // Default placeholders
    $placeholders = [
        '{to_name}' => isset( $data['to_name'] ) ? esc_html( $data['to_name'] ) : '',
        '{amount}'         => isset( $data['amount'] ) ? wc_price( $data['amount'] ) : '',
        '{from_name}'    => isset( $data['from_name'] ) ? esc_html( $data['from_name'] ) : '',
        '{message}'        => isset( $data['message'] ) ? esc_html( $data['message'] ) : '',
        '{coupon_code}'        => isset( $data['coupon_code'] ) ? esc_html( $data['coupon_code'] ) : '',
        '{featured_image}' => $thumbnail,
        '{exp_date}' => $expiry_date ,
        '{disclamer}' => $wptgc_disclaimer,
    ];

    $content = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $content );

    // Apply WP content filters (shortcodes, formatting, etc.)
    return apply_filters( 'the_content', $content );
}

/**
 * Update template selection function to respect product settings
 *
 * @since    1.0.0
 */
function wptgcl_get_gift_card_templates_for_product($product_id) {
    $use_all_templates = get_post_meta($product_id, '_wptgcl_use_all_templates', true) == 'yes';
    
    if ($use_all_templates) {
        // Use all active templates
        return wptgcl_get_gift_card_templates($product_id);
    } else {
        // Use only selected templates
        $allowed_templates = get_post_meta($product_id, '_wptgcl_allowed_templates', true);
        
        if (empty($allowed_templates) || !is_array($allowed_templates)) {
            // Fallback to all templates if none selected
            return wptgcl_get_gift_card_templates($product_id);
        }
        
        $templates = get_posts(array(
            'post_type' => 'wptgcl_template',
            'posts_per_page' => -1,
            'post__in' => $allowed_templates,
            'post_status' => 'publish',
            'meta_query' => array( // phpcs:ignore
                array(
                    'key' => '_wptgcl_is_active',
                    'value' => 'yes'
                )
            ),
            'orderby' => 'menu_order',
            'order' => 'ASC'
        ));

        $html = '';
        $default_selected = false;
        
        foreach ($templates as $template) {
            $thumbnail = get_the_post_thumbnail_url($template->ID, 'thumbnail');
            if (!$thumbnail) {
                $thumbnail = plugin_dir_url(__FILE__) . 'assets/default-template.png';
            }
            
            $is_default = get_post_meta($template->ID, '_wptgcl_is_default', true) == 'yes';
            $selected_class = ($is_default && !$default_selected) ? ' wptgcl_selected' : '';
            if ($is_default && !$default_selected) {
                $default_selected = true;
            }
            
            $html .= sprintf(
                '<img class="wptgcl_template_img%s" data-template-id="%d" src="%s" alt="%s" title="%s">',
                $selected_class,
                $template->ID,
                esc_url($thumbnail),
                esc_attr($template->post_title),
                esc_attr($template->post_title)
            );
        }

        return $html;
    }
}

/**
 * Update the frontend template selection function
 *
 * @since    1.0.0
 */
function wptgcl_get_gift_card_templates($product_id = null) {
    // Get active templates
    $templates = get_posts(array(
        'post_type' => 'wptgcl_template',
        'posts_per_page' => -1,
        'meta_query' => array( // phpcs:ignore
            array(
                'key' => '_wptgcl_is_active',
                'value' => 'yes'
            )
        ),
        'orderby' => 'menu_order',
        'order' => 'ASC'
    ));

    if (empty($templates)) {
        // Create default templates if none exist
        wptgcl_create_default_templates();
        $templates = get_posts(array(
            'post_type' => 'wptgcl_template',
            'posts_per_page' => -1,
            'meta_query' => array( // phpcs:ignore
                array(
                    'key' => '_wptgcl_is_active',
                    'value' => 'yes'
                )
            )
        ));
    }

    $html = '';
    $default_selected = false;
    
    foreach ($templates as $template) {
        $thumbnail = get_the_post_thumbnail_url($template->ID, 'thumbnail');
        if (!$thumbnail) {
            $thumbnail = plugin_dir_url(__FILE__) . 'assets/default-template.png';
        }
        
        $is_default = get_post_meta($template->ID, '_wptgcl_is_default', true) == 'yes';
        $selected_class = ($is_default && !$default_selected) ? ' wptgcl_selected' : '';
        if ($is_default && !$default_selected) {
            $default_selected = true;
        }
        
        $html .= sprintf(
            '<img class="wptgcl_template_img%s" data-template-id="%d" src="%s" alt="%s" title="%s">',
            $selected_class,
            $template->ID,
            esc_url($thumbnail),
            esc_attr($template->post_title),
            esc_attr($template->post_title)
        );
    }

    return $html;
}

/**
 * Generate gift card code based on settings
 *
 * @since    1.0.0
 */
function wptgcl_generate_gift_card_code() {
    $general_settings = get_option('wptgcl_general_settings_opt', array());
    
    // Get settings with defaults
    $code_length = isset($general_settings['gift_card_code_len']) ? intval($general_settings['gift_card_code_len']) : 7;
    $prefix = isset($general_settings['gift_card_pre']) ? $general_settings['gift_card_pre'] : 'wptgcl';
    
    // Generate random code
    $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    $code = '';
    
    for ($i = 0; $i < $code_length; $i++) {
        $code .= $characters[wp_rand(0, strlen($characters) - 1)];
    }
    
    $full_code = strtoupper($prefix . '-' . $code);
    
    global $wpdb;
    $existing = $wpdb->get_var($wpdb->prepare( // phpcs:ignore
        "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_wptgcl_coupon_code' AND meta_value = %s",
        $full_code
    ));
    
    // If code exists, generate a new one.
    if ($existing > 0) {
        return wptgcl_generate_gift_card_code();
    }
    
    return $full_code;
}
