<?php
/**
 * Plugin Name: WPKJ Payment Gateway for FluentCart with Wechat
 * Plugin URI: https://www.wpdaxue.com/wpkj-payment-gateway-for-fluentcart-with-wechat.html
 * Description: WeChat Pay payment gateway integration for FluentCart - Support Native, H5, JSAPI payments
 * Version: 1.0.6
 * Requires at least: 6.5
 * Requires PHP: 8.2
 * Author: WPDAXUE.COM
 * Author URI: https://www.wpdaxue.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wpkj-payment-gateway-for-fluentcart-with-wechat
 * Domain Path: /languages
 */

defined('ABSPATH') || exit;

define('WPKJ_FC_WECHAT_VERSION', '1.0.6');
define('WPKJ_FC_WECHAT_FILE', __FILE__);
define('WPKJ_FC_WECHAT_PATH', plugin_dir_path(__FILE__));
define('WPKJ_FC_WECHAT_URL', plugin_dir_url(__FILE__));

/**
 * Check plugin dependencies
 */
function wpkj_fc_wechat_check_dependencies() {
    // Check if FluentCart is installed
    if (!class_exists('FluentCart\App\Modules\PaymentMethods\Core\GatewayManager')) {
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-error">
                <p><?php echo esc_html__('WPKJ Payment Gateway for FluentCart with Wechat requires FluentCart to be installed and activated.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'); ?></p>
            </div>
            <?php
        });
        return false;
    }
    
    // Check FluentCart version
    if (defined('FLUENTCART_VERSION') && version_compare(FLUENTCART_VERSION, '1.2.0', '<')) {
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-error">
                <p><?php echo esc_html__('WPKJ Payment Gateway for FluentCart with Wechat requires FluentCart version 1.2.0 or higher.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'); ?></p>
            </div>
            <?php
        });
        return false;
    }
    
    return true;
}

/**
 * Initialize autoloader
 */
spl_autoload_register(function($class) {
    $prefix = 'WPKJFluentCart\\Wechat\\';
    $base_dir = WPKJ_FC_WECHAT_PATH . 'src/';
    
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    
    $relative_class = substr($class, $len);
    $file = $base_dir . str_replace('\\', '/', $relative_class) . '.php';
    
    if (file_exists($file)) {
        require $file;
    }
});

/**
 * Bootstrap the plugin
 */
function wpkj_fc_wechat_bootstrap() {
    if (!wpkj_fc_wechat_check_dependencies()) {
        return;
    }
    
    add_action('init', function() {
        // Register payment gateway
        add_action('fluent_cart/register_payment_methods', function($args) {
            $gatewayManager = $args['gatewayManager'];
            
            $wechatGateway = new \WPKJFluentCart\Wechat\Gateway\WechatGateway();
            
            $gatewayManager->register('wechat', $wechatGateway);
            
        }, 10, 1);
        
        // Register refund processor
        $refundProcessor = new \WPKJFluentCart\Wechat\Processor\RefundProcessor();
        $refundProcessor->register();
        
        // Register payment status checker
        $statusChecker = new \WPKJFluentCart\Wechat\Processor\PaymentStatusChecker();
        $statusChecker->register();
        
        // Register Native payment page handler
        $nativePageHandler = new \WPKJFluentCart\Wechat\Processor\NativePageHandler();
        $nativePageHandler->register();
        
        // Register H5 waiting page handler
        $h5WaitingHandler = new \WPKJFluentCart\Wechat\Processor\H5WaitingPageHandler();
        $h5WaitingHandler->register();
        
        // Register order cancel listener (for subscription cancellation sync)
        $orderCancelListener = new \WPKJFluentCart\Wechat\Listeners\OrderCancelListener();
        $orderCancelListener->register();
        
        // Register Custom Payment API
        $customPaymentAPI = new \WPKJFluentCart\Wechat\API\CustomPaymentAPI();
        $customPaymentAPI->register();
        
    }, 9);
}
add_action('plugins_loaded', 'wpkj_fc_wechat_bootstrap', 20);

/**
 * Plugin activation hook
 */
function wpkj_fc_wechat_activate() {
    if (!wpkj_fc_wechat_check_dependencies()) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('WPKJ Payment Gateway for FluentCart with Wechat requires FluentCart to be installed and activated.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            esc_html__('Plugin Activation Error', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            array('back_link' => true)
        );
    }
}
register_activation_hook(__FILE__, 'wpkj_fc_wechat_activate');
