<?php

namespace WPKJFluentCart\Wechat\Webhook;

use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use FluentCart\App\Helpers\Status;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Processor\PaymentProcessor;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * WeChat Pay Return Handler
 * 
 * Handles return from WeChat Pay (redirect back to site)
 * Actively queries payment status instead of relying on async notification
 */
class ReturnHandler
{
    /**
     * Settings instance
     * 
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * API instance
     * 
     * @var WechatAPI
     */
    private $api;

    /**
     * Processor instance
     * 
     * @var PaymentProcessor
     */
    private $processor;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->settings = new WechatSettingsBase();
        
        // Get payment mode from FluentCart global settings
        $paymentMode = fluent_cart_get_option('order_mode', 'live');
        
        // Build API config
        $config = [
            'appid' => $this->settings->getAppId(),
            'mch_id' => $this->settings->getMchId(),
            'api_key' => $this->settings->getApiKey(),
            'sign_type' => $this->settings->getSignType(),
            'mode' => $paymentMode,
        ];
        
        $this->api = new WechatAPI($config);
        $this->processor = new PaymentProcessor($this->settings);
    }

    /**
     * Handle return from WeChat Pay
     * 
     * This is triggered when user returns from WeChat Pay after payment
     * We actively query the payment status instead of waiting for notification
     * 
     * @return void
     */
    public function handleReturn()
    {
        try {
            // Get parameters
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WeChat Pay return URL verification handled by signature
            $trxHash = isset($_GET['trx_hash']) ? sanitize_text_field(wp_unslash($_GET['trx_hash'])) : '';
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WeChat Pay return URL verification handled by signature
            $redirect = isset($_GET['fct_redirect']) ? sanitize_text_field(wp_unslash($_GET['fct_redirect'])) : '';
            
            // Check if this is actually a WeChat Pay return (not Alipay)
            // Both plugins use trx_hash, so we need to check payment method
            if ($redirect !== 'yes' || empty($trxHash)) {
                Logger::info('Skipping WeChat return handler - invalid parameters', [
                    'trx_hash' => $trxHash,
                    'fct_redirect' => $redirect
                ]);
                return;
            }

            Logger::info('Return URL Accessed', [
                'trx_hash' => $trxHash
            ]);

            // Find transaction
            $transaction = OrderTransaction::query()
                ->where('uuid', $trxHash)
                ->where('transaction_type', Status::TRANSACTION_TYPE_CHARGE)
                ->first();

            if (!$transaction) {
                Logger::error('Transaction Not Found in Return', [
                    'trx_hash' => $trxHash
                ]);
                $this->redirectToError(__('Transaction not found', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
                return;
            }
            
            // ✅ Check if this transaction belongs to WeChat Pay
            if ($transaction->payment_method !== 'wechat') {
                Logger::info('Skipping WeChat return handler - different payment method', [
                    'trx_hash' => $trxHash,
                    'payment_method' => $transaction->payment_method ?? 'unknown'
                ]);
                return;
            }

            $order = $transaction->order;
            if (!$order) {
                Logger::error('Order Not Found in Return', [
                    'transaction_id' => $transaction->id
                ]);
                $this->redirectToError(__('Order not found', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
                return;
            }

            // If not already succeeded, query payment status from WeChat Pay
            if ($transaction->status !== Status::TRANSACTION_SUCCEEDED) {
                Logger::info('Transaction not completed, querying payment status', [
                    'trx_hash' => $trxHash,
                    'current_status' => $transaction->status
                ]);
                $this->queryAndUpdatePaymentStatus($transaction);
                
                // Reload transaction to get updated status
                $transaction = $transaction->fresh();
            } else {
                Logger::info('Transaction already completed', ['trx_hash' => $trxHash]);
            }

            // Check final status after query
            if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
                // Payment successful - redirect to receipt page
                $receiptPageId = fluent_cart_get_option('receipt_page_id');
                
                if ($receiptPageId) {
                    $receiptUrl = get_permalink($receiptPageId);
                } else {
                    $receiptUrl = site_url('/receipt/');
                }
                
                $receiptUrl = add_query_arg([
                    'order_hash' => $order->uuid
                ], $receiptUrl);
                
                Logger::info('Payment completed, redirecting to Receipt', [
                    'order_uuid' => $order->uuid,
                    'transaction_status' => $transaction->status,
                    'receipt_page_id' => $receiptPageId ?? 'none'
                ]);

                wp_safe_redirect($receiptUrl);
                exit;
            } else {
                // Payment not completed yet - redirect to payment waiting page
                // This allows continued polling via AJAX
                $tradeType = $transaction->meta['trade_type'] ?? 'MWEB';
                
                if ($tradeType === 'MWEB') {
                    // For H5 payment, show waiting page with status polling
                    $waitingUrl = add_query_arg([
                        'fluent-cart' => 'wechat_payment_waiting',
                        'order_hash' => $order->uuid,
                        'trx_uuid' => $transaction->uuid
                    ], home_url('/'));
                    
                    Logger::info('Payment pending, redirecting to waiting page', [
                        'order_uuid' => $order->uuid,
                        'transaction_status' => $transaction->status,
                        'trade_type' => $tradeType
                    ]);
                    
                    wp_safe_redirect($waitingUrl);
                    exit;
                } else {
                    // For other payment types, redirect to custom payment page
                    $paymentUrl = add_query_arg([
                        'order_hash' => $order->uuid
                    ], \FluentCart\App\Services\Payments\PaymentHelper::getCustomPaymentLink($order->uuid));
                    
                    Logger::info('Payment pending, redirecting to payment page', [
                        'order_uuid' => $order->uuid,
                        'transaction_status' => $transaction->status
                    ]);
                    
                    wp_safe_redirect($paymentUrl);
                    exit;
                }
            }

        } catch (\Exception $e) {
            Logger::error('Return Handler Exception', [
                'error' => $e->getMessage()
            ]);
            $this->redirectToError($e->getMessage());
        }
    }

    /**
     * Query payment status from WeChat Pay and update order
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @return void
     */
    private function queryAndUpdatePaymentStatus($transaction)
    {
        try {
            // CRITICAL: Retrieve out_trade_no from transaction meta
            // DO NOT regenerate because it contains creation timestamp
            $outTradeNo = $transaction->meta['out_trade_no'] ?? null;
            
            // Fallback for old transactions without stored out_trade_no
            if (empty($outTradeNo)) {
                Logger::warning('Missing out_trade_no in transaction meta, using fallback', [
                    'transaction_uuid' => $transaction->uuid
                ]);
                // Use old format (without timestamp) for backward compatibility
                $outTradeNo = str_replace('-', '', $transaction->uuid);
            }

            // Query trade status
            $result = $this->api->queryPayment($outTradeNo);

            if (is_wp_error($result)) {
                Logger::error('Query payment API error', [
                    'transaction_uuid' => $transaction->uuid,
                    'error_code' => $result->get_error_code(),
                    'error_message' => $result->get_error_message()
                ]);
                return;
            }

            $tradeState = $result['trade_state'] ?? '';

            // Handle based on trade state
            switch ($tradeState) {
                case 'SUCCESS':
                    Logger::info('Payment successful', [
                        'transaction_uuid' => $transaction->uuid,
                        'trade_state' => $tradeState,
                        'transaction_id' => $result['transaction_id'] ?? ''
                    ]);
                    
                    // Payment successful, update order
                    $this->processor->confirmPaymentSuccess($transaction, $result);
                    break;

                case 'NOTPAY':
                case 'USERPAYING':
                    Logger::info('Payment pending', [
                        'transaction_uuid' => $transaction->uuid,
                        'trade_state' => $tradeState
                    ]);
                    break;

                case 'CLOSED':
                case 'REVOKED':
                case 'PAYERROR':
                    Logger::info('Payment failed or closed', [
                        'transaction_uuid' => $transaction->uuid,
                        'trade_state' => $tradeState
                    ]);
                    
                    // Payment failed or cancelled
                    $this->processor->processFailedPayment($transaction, [
                        'reason' => $result['trade_state_desc'] ?? 'Trade closed'
                    ]);
                    break;

                default:
                    Logger::warning('Unknown trade state', [
                        'transaction_uuid' => $transaction->uuid,
                        'trade_state' => $tradeState
                    ]);
                    break;
            }

        } catch (\Exception $e) {
            Logger::error('Query payment status exception', [
                'transaction_uuid' => $transaction->uuid,
                'exception_message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Redirect to error page
     * 
     * @param string $message Error message
     * @return void
     */
    private function redirectToError(string $message)
    {
        $errorUrl = add_query_arg([
            'payment_error' => 'yes',
            'error_message' => urlencode($message)
        ], home_url('/'));

        wp_safe_redirect($errorUrl);
        exit;
    }
}
