<?php

namespace WPKJFluentCart\Wechat\Webhook;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Helpers\StatusHelper;
use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use FluentCart\App\Services\Payments\PaymentHelper;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Config\WechatConfig;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Services\OrderService;
use WPKJFluentCart\Wechat\Services\SubscriptionService;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * WeChat Pay Notify Handler
 * 
 * Handles async payment notifications from WeChat Pay
 */
class NotifyHandler
{
    /**
     * Settings instance
     * 
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * API instance
     * 
     * @var WechatAPI
     */
    private $api;

    /**
     * Constructor
     * 
     * @param WechatSettingsBase $settings Settings instance
     */
    public function __construct(WechatSettingsBase $settings)
    {
        $this->settings = $settings;
        
        // Do not initialize API client here to avoid errors when settings are not configured
        // API client will be initialized lazily when needed
    }

    /**
     * Initialize API client lazily
     * 
     * @return void
     */
    private function initializeApi()
    {
        if ($this->api !== null) {
            return;
        }

        // Build API config
        $config = [
            'appid' => $this->settings->getAppId(),
            'mch_id' => $this->settings->getMchId(),
            'api_key' => $this->settings->getApiKey(),
            'sign_type' => $this->settings->getSignType(),
        ];
        
        $this->api = new WechatAPI($config);
    }

    /**
     * Process WeChat Pay notification
     * 
     * @return void
     */
    public function processNotify()
    {
        try {
            // Initialize API client
            $this->initializeApi();
            // Get raw POST data (XML)
            $xmlData = file_get_contents('php://input');
            
            if (empty($xmlData)) {
                Logger::error('Empty Notification Data');
                $this->returnFailure('Empty notification data');
                return;
            }

            Logger::info('Notification Received', [
                'data_length' => strlen($xmlData),
                // Sanitize XML preview for safe logging
                'data_preview' => sanitize_text_field(substr($xmlData, 0, 200))
            ]);

            // Parse XML to array
            $data = $this->xmlToArray($xmlData);

            if (!is_array($data)) {
                Logger::error('Invalid XML Format');
                $this->returnFailure('Invalid XML format');
                return;
            }

            // Check return_code
            if (!isset($data['return_code']) || $data['return_code'] !== 'SUCCESS') {
                $returnMsg = $data['return_msg'] ?? 'Unknown error';
                Logger::error('Notification Return Code Failed', [
                    'return_code' => $data['return_code'] ?? 'missing',
                    'return_msg' => $returnMsg
                ]);
                $this->returnFailure($returnMsg);
                return;
            }

            // Verify signature
            if (!$this->api->verifySign($data)) {
                Logger::error('Signature Verification Failed', [
                    'out_trade_no' => $data['out_trade_no'] ?? 'unknown'
                ]);
                $this->returnFailure('Signature verification failed');
                return;
            }

            // Check result_code
            if (!isset($data['result_code']) || $data['result_code'] !== 'SUCCESS') {
                $errCode = $data['err_code'] ?? 'unknown';
                $errCodeDes = $data['err_code_des'] ?? 'Unknown error';
                Logger::error('Notification Result Code Failed', [
                    'err_code' => $errCode,
                    'err_code_des' => $errCodeDes
                ]);
                $this->returnFailure($errCodeDes);
                return;
            }

            // Process payment success
            $this->handlePaymentSuccess($data);

        } catch (\Exception $e) {
            Logger::error('Notification Processing Exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            $this->returnFailure($e->getMessage());
        }
    }

    /**
     * Handle payment success
     * 
     * @param array $data Notification data
     * @return void
     */
    private function handlePaymentSuccess(array $data)
    {
        $outTradeNo = $data['out_trade_no'] ?? '';
        $transactionId = $data['transaction_id'] ?? '';
        $totalFee = $data['total_fee'] ?? 0;

        if (empty($outTradeNo)) {
            Logger::error('Missing out_trade_no');
            $this->returnFailure('Missing out_trade_no');
            return;
        }

        // Parse out_trade_no to get transaction UUID
        // Format: {uuid_without_dashes}_{timestamp} or just {uuid_without_dashes}
        $transactionUuid = $this->parseOutTradeNo($outTradeNo);
        
        // Find transaction by UUID
        $transaction = OrderTransaction::query()
            ->where('uuid', $transactionUuid)
            ->where('transaction_type', Status::TRANSACTION_TYPE_CHARGE)
            ->first();

        if (!$transaction) {
            Logger::warning('Transaction Not Found', [
                'out_trade_no' => $outTradeNo,
                'transaction_uuid' => $transactionUuid
            ]);
            
            // Still return success to WeChat Pay to prevent retries
            $this->returnSuccess();
            return;
        }
        
        // ✅ Check if this transaction belongs to WeChat Pay
        if ($transaction->payment_method !== 'wechat') {
            Logger::warning('Skipping WeChat notify handler - different payment method', [
                'transaction_uuid' => $transaction->uuid,
                'payment_method' => $transaction->payment_method ?? 'unknown',
                'out_trade_no' => $outTradeNo
            ]);
            // Return success to prevent retries
            $this->returnSuccess();
            return;
        }

        // Check if already processed
        if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
            Logger::info('Transaction Already Processed', [
                'transaction_uuid' => $transaction->uuid,
                'out_trade_no' => $outTradeNo
            ]);
            $this->returnSuccess();
            return;
        }

        $order = $transaction->order;
        if (!$order) {
            Logger::error('Order Not Found', [
                'transaction_id' => $transaction->id
            ]);
            $this->returnFailure('Order not found');
            return;
        }

        // Update transaction
        $transaction->status = Status::TRANSACTION_SUCCEEDED;
        $transaction->vendor_charge_id = $transactionId;
        $transaction->meta = array_merge($transaction->meta ?? [], [
            'wechat_transaction_id' => $transactionId,
            'wechat_bank_type' => $data['bank_type'] ?? '',
            'wechat_time_end' => $data['time_end'] ?? '',
            'wechat_cash_fee' => $data['cash_fee'] ?? $totalFee,
        ]);
        $transaction->save();

        // Check if this is a subscription payment and handle it
        if (SubscriptionService::isSubscriptionTransaction($transaction)) {
            SubscriptionService::handleSubscriptionPaymentSuccess($transaction, $data, 'webhook');
        }

        // Add log to order activity
        fluent_cart_add_log(
            __('WeChat Pay Payment Confirmation', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            sprintf(
                /* translators: %s: WeChat Pay transaction ID */
                __('Payment confirmed from WeChat Pay. Transaction ID: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $transactionId
            ),
            'info',
            [
                'module_name' => 'order',
                'module_id' => $order->id,
            ]
        );

        // Sync order statuses
        (new StatusHelper($order))->syncOrderStatuses($transaction);

        // Clear cart order association
        OrderService::clearCartOrderAssociation($order, 'wechat_notify');

        Logger::info('Payment Success Processed', [
            'order_uuid' => $order->uuid,
            'transaction_uuid' => $transaction->uuid,
            'out_trade_no' => $outTradeNo,
            'transaction_id' => $transactionId,
            'total_fee' => $totalFee
        ]);

        $this->returnSuccess();
    }

    /**
     * Parse out_trade_no to get transaction UUID
     * 
     * Supports two formats:
     * 1. New format (with timestamp): {uuid_without_dashes}_{timestamp}
     *    Example: 6a3f5c2e7b1d4a9e8f0c1d2e3f4a5b6c1761050245
     * 2. Old format (without timestamp): {uuid_without_dashes}
     *    Example: 6a3f5c2e7b1d4a9e8f0c1d2e3f4a5b6c
     * 
     * @param string $outTradeNo Out trade number
     * @return string Transaction UUID with dashes
     */
    private function parseOutTradeNo($outTradeNo)
    {
        // WeChat out_trade_no format: first 22 chars of UUID (without dashes) + 10-digit timestamp
        // Total length: 32 chars
        
        if (strlen($outTradeNo) === 32) {
            // Extract first 22 chars as UUID part
            $uuidPart = substr($outTradeNo, 0, 22);
            
            // Find transaction by partial UUID match
            // Try to find transaction where UUID starts with this pattern
            $pattern = substr($uuidPart, 0, 8) . '-' . 
                      substr($uuidPart, 8, 4) . '-' . 
                      substr($uuidPart, 12, 4) . '-' . 
                      substr($uuidPart, 16, 4) . '-' . 
                      substr($uuidPart, 20, 2) . '%';
                      
            // Try exact UUID reconstruction first (old format)
            $exactUuid = substr($outTradeNo, 0, 8) . '-' . 
                        substr($outTradeNo, 8, 4) . '-' . 
                        substr($outTradeNo, 12, 4) . '-' . 
                        substr($outTradeNo, 16, 4) . '-' . 
                        substr($outTradeNo, 20, 12);
                        
            // Check if exact UUID exists (for old format compatibility)
            $exactMatch = OrderTransaction::query()
                ->where('uuid', $exactUuid)
                ->exists();
                
            if ($exactMatch) {
                return $exactUuid;
            }
            
            // New format: find by pattern matching
            $transaction = OrderTransaction::query()
                ->where('uuid', 'LIKE', $pattern)
                ->where('payment_method', 'wechat')
                ->orderBy('id', 'DESC')
                ->first();
                
            if ($transaction) {
                return $transaction->uuid;
            }
        }

        // Unknown format, log warning and return as-is
        Logger::warning('Unknown out_trade_no Format', [
            'out_trade_no' => $outTradeNo,
            'length' => strlen($outTradeNo)
        ]);
        
        return $outTradeNo;
    }



    /**
     * Convert XML to array
     * 
     * @param string $xml XML string
     * @return array|false
     */
    private function xmlToArray(string $xml)
    {
        try {
            // Note: libxml_disable_entity_loader() is deprecated in PHP 8.0+
            // External entity loading is disabled by default since libxml 2.9.0
            // Use LIBXML_NOENT flag to ensure entities are substituted
            $xmlObj = simplexml_load_string($xml, 'SimpleXMLElement', LIBXML_NOCDATA | LIBXML_NONET);

            if ($xmlObj === false) {
                return false;
            }

            $json = json_encode($xmlObj);
            $array = json_decode($json, true);

            return $array;
        } catch (\Exception $e) {
            Logger::error('XML Parse Error', $e->getMessage());
            return false;
        }
    }

    /**
     * Return success response to WeChat Pay
     * 
     * @return void
     */
    private function returnSuccess()
    {
        $xml = '<xml><return_code><![CDATA[SUCCESS]]></return_code><return_msg><![CDATA[OK]]></return_msg></xml>';
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- WeChat Pay expects specific XML format
        echo $xml;
        exit;
    }

    /**
     * Return failure response to WeChat Pay
     * 
     * @param string $message Error message
     * @return void
     */
    private function returnFailure(string $message)
    {
        $xml = sprintf(
            '<xml><return_code><![CDATA[FAIL]]></return_code><return_msg><![CDATA[%s]]></return_msg></xml>',
            htmlspecialchars($message)
        );
        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- WeChat Pay expects specific XML format, message already escaped
        echo $xml;
        exit;
    }
}
