<?php

namespace WPKJFluentCart\Wechat\Utils;

use FluentCart\App\Helpers\Helper as FluentCartHelper;

/**
 * Helper Utility
 * 
 * General helper functions for WeChat Pay gateway
 */
class Helper
{
    /**
     * Convert cents to decimal amount
     * 
     * @param int $cents Amount in cents
     * @return string Decimal amount
     */
    public static function toDecimal($cents)
    {
        return FluentCartHelper::toDecimalWithoutComma($cents);
    }

    /**
     * Convert decimal to cents
     * 
     * @param float $amount Decimal amount
     * @return int Amount in cents
     */
    public static function toCents($amount)
    {
        return FluentCartHelper::toCent($amount);
    }

    /**
     * Generate nonce_str (random string for WeChat Pay)
     * 
     * WeChat Pay requires a nonce_str parameter for security.
     * It should be a random string of 32 characters or less.
     * 
     * @param int $length Length of random string (default: 32)
     * @return string Random string
     */
    public static function generateNonceStr(int $length = 32): string
    {
        $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $str = '';
        for ($i = 0; $i < $length; $i++) {
            $str .= $chars[random_int(0, strlen($chars) - 1)];
        }
        return $str;
    }

    /**
     * Generate out_trade_no (order number for WeChat Pay)
     * 
     * IMPORTANT: WeChat Pay requires out_trade_no to be unique for each payment attempt.
     * Even if it's the same product/customer, each payment must have a unique out_trade_no.
     * 
     * Format: {transaction_uuid_without_dashes}_{timestamp_microseconds}
     * This ensures absolute uniqueness even for rapid repeated orders.
     * 
     * @param string $transactionUuid Transaction UUID
     * @return string Unique trade number (max 32 chars)
     */
    public static function generateOutTradeNo($transactionUuid)
    {
        // Remove dashes from UUID (32 chars)
        $baseUuid = str_replace('-', '', $transactionUuid);
        
        // Add timestamp to ensure uniqueness (10 chars)
        // WeChat Pay limits out_trade_no to 32 chars, so we keep it shorter
        $uniqueSuffix = substr((string)time(), -10);
        
        // Total length: 22 + 10 = 32 chars (WeChat Pay's limit)
        return substr($baseUuid, 0, 22) . $uniqueSuffix;
    }

    /**
     * Sanitize WeChat Pay response data
     * 
     * @param array $data Response data
     * @return array Sanitized data
     */
    public static function sanitizeResponseData($data)
    {
        $sanitized = [];
        
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $sanitized[$key] = self::sanitizeResponseData($value);
            } else {
                $sanitized[$key] = sanitize_text_field($value);
            }
        }
        
        return $sanitized;
    }

    /**
     * Check if HTTPS is enabled
     * 
     * @return bool
     */
    public static function isHttps()
    {
        return is_ssl();
    }

    /**
     * Get current site URL with protocol
     * 
     * @return string
     */
    public static function getSiteUrl()
    {
        return home_url('/');
    }

    /**
     * Format amount for display
     * 
     * @param int $cents Amount in cents
     * @param string $currency Currency code
     * @return string Formatted amount with currency symbol
     */
    public static function formatAmount($cents, $currency = 'CNY')
    {
        $amount = self::toDecimal($cents);
        
        $currencySymbols = [
            'CNY' => '¥',
            'USD' => '$',
            'EUR' => '€',
            'GBP' => '£',
            'JPY' => '¥',
            'HKD' => 'HK$',
            'KRW' => '₩',
            'SGD' => 'S$',
            'AUD' => 'A$',
            'CAD' => 'C$',
            'CHF' => 'CHF',
            'NZD' => 'NZ$',
            'THB' => '฿',
            'MYR' => 'RM'
        ];
        
        $symbol = $currencySymbols[$currency] ?? $currency . ' ';
        
        return $symbol . number_format((float)$amount, 2, '.', ',');
    }
}
