<?php

namespace WPKJFluentCart\Wechat\Subscription;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Models\OrderTransaction;
use FluentCart\App\Models\Subscription;
use FluentCart\App\Modules\PaymentMethods\Core\AbstractSubscriptionModule;
use FluentCart\App\Modules\Subscriptions\Services\SubscriptionService as FluentCartSubscriptionService;
use FluentCart\Framework\Support\Arr;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * WeChat Pay Subscriptions Module
 * 
 * Handles recurring payment subscriptions for WeChat Pay
 * Note: WeChat Pay doesn't have native recurring payment API
 * This implementation uses manual payment reminders and scheduled billing
 */
class WechatSubscriptions extends AbstractSubscriptionModule
{
    /**
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * Constructor
     * 
     * @param WechatSettingsBase $settings Settings instance
     */
    public function __construct(WechatSettingsBase $settings)
    {
        $this->settings = $settings;
    }

    /**
     * Re-sync subscription status from WeChat
     * 
     * Since WeChat doesn't have native subscription API,
     * we check the latest payment transaction status
     * 
     * @param Subscription $subscriptionModel Subscription model
     * @return \WP_Error|array Sync result
     */
    public function reSyncSubscriptionFromRemote(Subscription $subscriptionModel)
    {
        if ($subscriptionModel->current_payment_method !== 'wechat') {
            return new \WP_Error(
                'invalid_payment_method',
                __('This subscription is not using WeChat Pay as payment method.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            );
        }

        $order = $subscriptionModel->order;

        try {
            // Get the latest transaction for this subscription
            $latestTransaction = OrderTransaction::query()
                ->where('order_id', $order->id)
                ->where('payment_method', 'wechat')
                ->orderBy('id', 'DESC')
                ->first();

            if (!$latestTransaction || !$latestTransaction->vendor_charge_id) {
                Logger::warning('No valid transaction found for subscription sync', [
                    'subscription_id' => $subscriptionModel->id,
                    'order_id' => $order->id
                ]);
                
                return new \WP_Error(
                    'no_transaction',
                    __('No valid WeChat Pay transaction found for this subscription.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            // Query transaction status from WeChat
            $api = new WechatAPI($this->settings);
            $outTradeNo = $latestTransaction->meta['out_trade_no'] ?? str_replace('-', '', $latestTransaction->uuid);
            
            $orderData = $api->queryOrder($outTradeNo);

            if (is_wp_error($orderData)) {
                return $orderData;
            }

            $tradeState = Arr::get($orderData, 'trade_state');
            $transactionId = Arr::get($orderData, 'transaction_id');

            Logger::info('Subscription Sync from WeChat', [
                'subscription_id' => $subscriptionModel->id,
                'trade_state' => $tradeState,
                'transaction_id' => $transactionId
            ]);

            // Map WeChat status to FluentCart subscription status
            $subscriptionStatus = $this->mapWechatStatusToSubscription($tradeState);
            
            if ($subscriptionStatus) {
                // Use FluentCart's syncSubscriptionStates for standard state management
                $updateArgs = [
                    'status' => $subscriptionStatus
                ];

                Logger::info('Syncing Subscription State via FluentCart Standard Method', [
                    'subscription_id' => $subscriptionModel->id,
                    'new_status' => $subscriptionStatus,
                    'wechat_status' => $tradeState
                ]);

                $subscriptionModel = FluentCartSubscriptionService::syncSubscriptionStates(
                    $subscriptionModel,
                    $updateArgs
                );

                Logger::info('Subscription State Synced Successfully', [
                    'subscription_id' => $subscriptionModel->id,
                    'status' => $subscriptionModel->status,
                    'bill_count' => $subscriptionModel->bill_count
                ]);
            }

            return [
                'status' => 'success',
                'message' => __('Subscription synced successfully.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'data' => [
                    'trade_state' => $tradeState,
                    'subscription_status' => $subscriptionModel->status
                ]
            ];

        } catch (\Exception $e) {
            Logger::error('Subscription Sync Error', [
                'subscription_id' => $subscriptionModel->id,
                'error' => $e->getMessage()
            ]);

            return new \WP_Error('sync_error', $e->getMessage());
        }
    }

    /**
     * Cancel subscription
     * 
     * For WeChat subscriptions, we just update the local status
     * since there's no remote subscription to cancel
     * 
     * @param string $vendorSubscriptionId Vendor subscription ID
     * @param array $args Additional arguments
     * @return array|\WP_Error Cancel result
     */
    public function cancel($vendorSubscriptionId, $args = [])
    {
        try {
            // Find subscription by vendor ID or use it directly
            $subscription = null;
            
            if (is_numeric($vendorSubscriptionId)) {
                $subscription = Subscription::find($vendorSubscriptionId);
            } else {
                $subscription = Subscription::query()
                    ->where('vendor_subscription_id', $vendorSubscriptionId)
                    ->first();
            }

            if (!$subscription) {
                return new \WP_Error(
                    'subscription_not_found',
                    __('Subscription not found.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            Logger::info('WeChat Subscription Cancellation', [
                'subscription_id' => $subscription->id,
                'vendor_subscription_id' => $vendorSubscriptionId,
                'reason' => Arr::get($args, 'reason', 'User requested')
            ]);

            // Update local subscription status
            $subscription->status = Status::SUBSCRIPTION_CANCELED;
            $subscription->canceled_at = current_time('mysql');
            $subscription->save();

            return [
                'status' => 'success',
                'message' => __('Subscription cancelled successfully.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ];

        } catch (\Exception $e) {
            Logger::error('Subscription Cancellation Error', [
                'vendor_subscription_id' => $vendorSubscriptionId,
                'error' => $e->getMessage()
            ]);

            return new \WP_Error('cancel_error', $e->getMessage());
        }
    }

    /**
     * Cancel subscription (alternative method signature)
     * 
     * @param array $data Request data
     * @param object $order Order object
     * @param Subscription $subscription Subscription model
     * @return void
     * @throws \Exception
     */
    public function cancelSubscription($data, $order, $subscription)
    {
        $result = $this->cancel($subscription->id, [
            'reason' => Arr::get($data, 'reason', 'User requested cancellation')
        ]);

        if (is_wp_error($result)) {
            throw new \Exception(esc_html($result->get_error_message()));
        }
    }

    /**
     * Map WeChat trade status to subscription status
     * 
     * @param string $wechatStatus WeChat trade state
     * @return string|null Subscription status
     */
    private function mapWechatStatusToSubscription($wechatStatus)
    {
        $statusMap = [
            'SUCCESS' => Status::SUBSCRIPTION_ACTIVE,
            'NOTPAY' => Status::SUBSCRIPTION_PENDING,
            'CLOSED' => Status::SUBSCRIPTION_CANCELED,
            'REFUND' => Status::SUBSCRIPTION_CANCELED,
            'PAYERROR' => Status::SUBSCRIPTION_PAST_DUE,
        ];

        return $statusMap[$wechatStatus] ?? null;
    }

    /**
     * Reactivate subscription
     * 
     * @param array $data Request data
     * @param int $subscriptionId Subscription ID
     * @return void
     * @throws \Exception
     */
    public function reactivateSubscription($data, $subscriptionId)
    {
        $subscription = Subscription::find($subscriptionId);
        
        if (!$subscription) {
            throw new \Exception(esc_html__('Subscription not found.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
        }

        if ($subscription->current_payment_method !== 'wechat') {
            throw new \Exception(esc_html__('This subscription is not using WeChat Pay.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
        }

        Logger::info('WeChat Subscription Reactivation', [
            'subscription_id' => $subscriptionId,
            'current_status' => $subscription->status
        ]);

        // Update subscription status directly
        $updateArgs = [
            'status' => Status::SUBSCRIPTION_ACTIVE,
            'canceled_at' => null,
            'next_billing_date' => $subscription->guessNextBillingDate(true)
        ];

        $subscription = FluentCartSubscriptionService::syncSubscriptionStates(
            $subscription,
            $updateArgs
        );

        Logger::info('Subscription Reactivated Successfully', [
            'subscription_id' => $subscriptionId,
            'new_status' => $subscription->status,
            'next_billing_date' => $subscription->next_billing_date
        ]);
    }
}
