<?php

namespace WPKJFluentCart\Wechat\Subscription;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Models\Subscription;
use FluentCart\App\Services\Payments\PaymentInstance;
use FluentCart\Framework\Support\Arr;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Config\WechatConfig;
use WPKJFluentCart\Wechat\Detector\ClientDetector;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Services\EncodingService;
use WPKJFluentCart\Wechat\Utils\Helper;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * WeChat Pay Subscription Processor
 * 
 * Handles subscription payment processing for WeChat Pay
 * 
 * IMPORTANT: WeChat Pay doesn't have native recurring payment API like Alipay
 * Implementation strategy:
 * 1. Initial payment: Process like regular payment
 * 2. Store subscription information in transaction meta
 * 3. For renewals: Create new payment with stored customer info
 * 4. Use FluentCart's cron system to schedule renewals
 * 5. Manual renewal mode only (customer must complete payment for each renewal)
 */
class WechatSubscriptionProcessor
{
    /**
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * @var WechatAPI
     */
    private $api;

    /**
     * Constructor
     * 
     * @param WechatSettingsBase $settings
     */
    public function __construct(WechatSettingsBase $settings)
    {
        $this->settings = $settings;
        
        // Do not initialize API client here to avoid errors when settings are not configured
        // API client will be initialized lazily when needed
    }

    /**
     * Initialize API client lazily
     * 
     * @return void
     */
    private function initializeApi()
    {
        if ($this->api !== null) {
            return;
        }

        // Get payment mode from FluentCart global settings
        $paymentMode = fluent_cart_get_option('order_mode', 'live');

        // Build API config
        $config = [
            'appid' => $this->settings->getAppId(),
            'mch_id' => $this->settings->getMchId(),
            'api_key' => $this->settings->getApiKey(),
            'sign_type' => $this->settings->getSignType(),
            'ssl_cert_path' => $this->settings->getSslCertPath(),
            'ssl_key_path' => $this->settings->getSslKeyPath(),
            'mode' => $paymentMode,
        ];
        
        $this->api = new WechatAPI($config);
    }

    /**
     * Process subscription payment
     * 
     * @param PaymentInstance $paymentInstance
     * @return array
     */
    public function processSubscription(PaymentInstance $paymentInstance)
    {
        try {
            // Initialize API client
            $this->initializeApi();
            
            $order = $paymentInstance->order;
            $transaction = $paymentInstance->transaction;
            $subscription = $paymentInstance->subscription;

            if (!$subscription) {
                return [
                    'status' => 'failed',
                    'message' => __('No subscription found.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                ];
            }

            Logger::info('Processing WeChat Subscription Payment', [
                'order_id' => $order->id,
                'subscription_id' => $subscription->id,
                'order_type' => $order->type,
                'billing_interval' => $subscription->billing_interval
            ]);

            // WeChat Pay only supports manual renewal mode
            // No automatic recurring payment like Alipay's agreement system
            return $this->processManualPayment($subscription, $paymentInstance);

        } catch (\Exception $e) {
            Logger::error('Subscription Processing Exception', [
                'order_id' => $order->id ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate vendor subscription ID
     * 
     * @param Subscription $subscription
     * @return string
     */
    private function generateVendorSubscriptionId(Subscription $subscription)
    {
        // Format: wechat_sub_{subscription_id}_{timestamp}
        return 'wechat_sub_' . $subscription->id . '_' . time();
    }

    /**
     * Prepare payment parameters for subscription
     * 
     * @param PaymentInstance $paymentInstance
     * @param int $amount Amount in cents
     * @param Subscription $subscription
     * @return array
     */
    private function preparePaymentParams(PaymentInstance $paymentInstance, $amount, Subscription $subscription)
    {
        $order = $paymentInstance->order;
        $transaction = $paymentInstance->transaction;

        // Generate unique out_trade_no
        $outTradeNo = Helper::generateOutTradeNo($transaction->uuid);

        // Store subscription info in transaction meta
        $transaction->meta = array_merge($transaction->meta ?? [], [
            'out_trade_no' => $outTradeNo,
            'is_subscription' => true,
            'subscription_id' => $subscription->id
        ]);
        $transaction->save();

        // Prepare subject and body
        $subject = $this->buildSubscriptionSubject($order, $subscription);
        $body = EncodingService::sanitizeForWechat($subject, WechatConfig::MAX_BODY_LENGTH);

        // Build notify URL
        $notifyUrl = add_query_arg([
            'fct_payment_listener' => '1',
            'method' => 'wechat'
        ], site_url('/'));

        // Calculate timeout
        $timeoutMinutes = max(WechatConfig::PAYMENT_TIMEOUT_MINUTES, WechatConfig::MIN_PAYMENT_TIMEOUT_MINUTES);
        
        // Calculate time_expire using WordPress timezone functions
        // WeChat Pay expects YmdHis format in GMT+8 (Beijing time)
        $beijingTimestamp = current_time('timestamp') + (8 * HOUR_IN_SECONDS) + ($timeoutMinutes * 60);
        $timeExpire = gmdate('YmdHis', $beijingTimestamp);

        return [
            'body' => $body,
            'out_trade_no' => $outTradeNo,
            'total_fee' => $amount,
            'notify_url' => $notifyUrl,
            'time_expire' => $timeExpire
        ];
    }

    /**
     * Build subscription payment subject
     * 
     * @param object $order
     * @param Subscription $subscription
     * @return string
     */
    private function buildSubscriptionSubject($order, Subscription $subscription)
    {
        // Get product title from subscription
        $productTitle = '';
        
        // Try to get from order items first
        $orderItems = $order->order_items ?? [];
        
        if (!empty($orderItems)) {
            // Handle if order_items is a collection or array
            if (is_array($orderItems) && isset($orderItems[0])) {
                $firstItem = $orderItems[0];
                $productTitle = is_object($firstItem) ? ($firstItem->title ?? $firstItem->post_title ?? '') : ($firstItem['title'] ?? $firstItem['post_title'] ?? '');
            } else if (is_object($orderItems) && method_exists($orderItems, 'first')) {
                // If it's a Laravel Collection
                $firstItem = $orderItems->first();
                if ($firstItem) {
                    $productTitle = $firstItem->title ?? $firstItem->post_title ?? '';
                }
            }
        }
        
        // Fallback: try to get from subscription itself
        if (empty($productTitle) && $subscription->product_id) {
            $product = get_post($subscription->product_id);
            if ($product) {
                $productTitle = $product->post_title;
            }
        }
        
        // Final fallback
        if (empty($productTitle)) {
            $productTitle = __('Subscription', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
        }
        
        // Build subject with subscription type
        if ($order->type === 'renewal') {
            $subject = sprintf(
                /* translators: %s: product title */
                __('%s - Renewal', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $productTitle
            );
        } else {
            $subject = sprintf(
                /* translators: %s: product title */
                __('%s - Subscription', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $productTitle
            );
        }
        
        // Truncate if too long (WeChat Pay body limit)
        if (strlen($subject) > WechatConfig::MAX_BODY_LENGTH) {
            $subject = mb_substr($subject, 0, floor(WechatConfig::MAX_BODY_LENGTH / 3));
            while (strlen($subject) > WechatConfig::MAX_BODY_LENGTH - 3) {
                $subject = mb_substr($subject, 0, mb_strlen($subject, 'UTF-8') - 1, 'UTF-8');
            }
            $subject .= '...';
        }

        return $subject;
    }

    /**
     * Process payment by client type
     * 
     * @param string $tradeType
     * @param array $params
     * @param PaymentInstance $paymentInstance
     * @return array
     */
    private function processPaymentByTradeType($tradeType, $params, PaymentInstance $paymentInstance)
    {
        switch ($tradeType) {
            case WechatConfig::TRADE_TYPE_NATIVE:
                return $this->processNativePayment($params, $paymentInstance);
            
            case WechatConfig::TRADE_TYPE_MWEB:
                return $this->processH5Payment($params, $paymentInstance);
            
            case WechatConfig::TRADE_TYPE_JSAPI:
                return $this->processJsapiPayment($params, $paymentInstance);
            
            default:
                return $this->processNativePayment($params, $paymentInstance);
        }
    }

    /**
     * Process Native payment (PC QR code)
     * 
     * @param array $params
     * @param PaymentInstance $paymentInstance
     * @return array
     */
    private function processNativePayment($params, PaymentInstance $paymentInstance)
    {
        $transaction = $paymentInstance->transaction;
        $order = $paymentInstance->order;

        try {
            $result = $this->api->createNativePayment($params);

            if (is_wp_error($result)) {
                throw new \Exception($result->get_error_message());
            }

            // Save payment info to transaction meta
            $transaction->meta = array_merge($transaction->meta ?? [], [
                'code_url' => $result['code_url'],
                'prepay_id' => $result['prepay_id'],
                'trade_type' => WechatConfig::TRADE_TYPE_NATIVE,
                'out_trade_no' => $params['out_trade_no'],
                'is_subscription' => true,
                'subscription_id' => $paymentInstance->subscription->id
            ]);
            $transaction->save();

            Logger::info('Subscription Native Payment Created', [
                'transaction_uuid' => $transaction->uuid,
                'subscription_id' => $paymentInstance->subscription->id
            ]);

            // Build QR code page URL
            $qrPageUrl = home_url('/');
            $qrCodeEncoded = base64_encode($result['code_url']);
            $qrPageUrl = add_query_arg([
                'fluent-cart' => 'wechat_qr_payment',
                'order_hash' => $order->uuid,
                'qr_code' => $qrCodeEncoded,
                'trx_uuid' => $transaction->uuid
            ], $qrPageUrl);

            return [
                'status' => 'success',
                'message' => __('Redirecting to payment page...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'redirect_to' => $qrPageUrl
            ];

        } catch (\Exception $e) {
            Logger::error('Subscription Native Payment Error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id,
                'subscription_id' => $paymentInstance->subscription->id
            ]);

            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Process H5 payment (Mobile browser)
     * 
     * @param array $params
     * @param PaymentInstance $paymentInstance
     * @return array
     */
    private function processH5Payment($params, PaymentInstance $paymentInstance)
    {
        $transaction = $paymentInstance->transaction;
        $order = $paymentInstance->order;

        try {
            // Add scene_info for H5 payment
            $params['scene_info'] = [
                'h5_info' => [
                    'type' => 'Wap',
                    'wap_url' => home_url('/'),
                    'wap_name' => get_bloginfo('name')
                ]
            ];

            $result = $this->api->createH5Payment($params);

            if (is_wp_error($result)) {
                throw new \Exception($result->get_error_message());
            }

            // Save payment info to transaction meta
            $transaction->meta = array_merge($transaction->meta ?? [], [
                'mweb_url' => $result['mweb_url'],
                'prepay_id' => $result['prepay_id'],
                'trade_type' => WechatConfig::TRADE_TYPE_MWEB,
                'out_trade_no' => $params['out_trade_no'],
                'is_subscription' => true,
                'subscription_id' => $paymentInstance->subscription->id
            ]);
            $transaction->save();

            Logger::info('Subscription H5 Payment Created', [
                'transaction_uuid' => $transaction->uuid,
                'subscription_id' => $paymentInstance->subscription->id
            ]);

            // Add redirect_url parameter for returning to site after payment
            $returnUrl = add_query_arg([
                'trx_hash' => $transaction->uuid,
                'fct_redirect' => 'yes',
                'out_trade_no' => $params['out_trade_no']
            ], home_url('/'));

            $mwebUrl = $result['mweb_url'] . '&redirect_url=' . urlencode($returnUrl);

            return [
                'status' => 'success',
                'nextAction' => 'redirect',
                'actionName' => 'custom',
                'message' => __('Redirecting to WeChat Pay...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'redirect_to' => $mwebUrl
            ];

        } catch (\Exception $e) {
            Logger::error('Subscription H5 Payment Error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id,
                'subscription_id' => $paymentInstance->subscription->id
            ]);

            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Process JSAPI payment (WeChat browser)
     * 
     * @param array $params
     * @param PaymentInstance $paymentInstance
     * @return array
     */
    private function processJsapiPayment($params, PaymentInstance $paymentInstance)
    {
        // JSAPI payment requires openid, which needs additional implementation
        // For now, return error message
        Logger::error('Subscription JSAPI Payment Not Implemented', [
            'order_id' => $paymentInstance->order->id,
            'subscription_id' => $paymentInstance->subscription->id
        ]);

        return [
            'status' => 'failed',
            'message' => __('JSAPI payment is not yet implemented. Please use a different browser or device.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
        ];
    }

    /**
     * Manual payment mode (WeChat Pay only supports this for subscriptions)
     * 
     * @param Subscription $subscription
     * @param PaymentInstance $paymentInstance
     * @return array
     */
    private function processManualPayment(Subscription $subscription, PaymentInstance $paymentInstance)
    {
        $order = $paymentInstance->order;
        $transaction = $paymentInstance->transaction;

        // Determine payment amount based on order type
        if ($order->type === 'renewal') {
            // Renewal payment: only charge recurring amount
            $paymentAmount = $subscription->getCurrentRenewalAmount();
        } else {
            // Initial payment: signup fee + first period (may include trial)
            $paymentAmount = (int)$subscription->signup_fee + (int)$subscription->recurring_total;
            
            // If there's a trial period, first charge is only signup fee
            if ($subscription->trial_days > 0) {
                $paymentAmount = (int)$subscription->signup_fee;
            }
        }

        // Ensure minimum amount
        if ($paymentAmount < WechatConfig::MIN_PAYMENT_AMOUNT_CENTS) {
            $paymentAmount = $transaction->total;
        }

        Logger::info('Subscription Manual Payment Amount Calculated', [
            'order_type' => $order->type,
            'amount' => $paymentAmount,
            'subscription_id' => $subscription->id
        ]);

        // Generate vendor subscription ID (for local tracking)
        if (empty($subscription->vendor_subscription_id)) {
            $vendorSubscriptionId = $this->generateVendorSubscriptionId($subscription);
            $subscription->update([
                'vendor_subscription_id' => $vendorSubscriptionId,
                'current_payment_method' => 'wechat'
            ]);
        }

        // Create payment parameters
        $paymentParams = $this->preparePaymentParams($paymentInstance, $paymentAmount, $subscription);

        // Detect trade type and process payment
        $tradeType = ClientDetector::getTradeType($this->settings);
        return $this->processPaymentByTradeType($tradeType, $paymentParams, $paymentInstance);
    }
}
