<?php

namespace WPKJFluentCart\Wechat\Services;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Models\OrderTransaction;
use FluentCart\App\Models\Subscription;
use FluentCart\App\Modules\Subscriptions\Services\SubscriptionService as FluentCartSubscriptionService;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * Subscription Service
 * 
 * Centralized service for handling subscription-related operations
 * Eliminates code duplication across NotifyHandler, PaymentStatusChecker, and PaymentProcessor
 */
class SubscriptionService
{
    /**
     * Check if transaction is for a subscription
     * 
     * @param OrderTransaction $transaction
     * @return bool
     */
    public static function isSubscriptionTransaction($transaction)
    {
        // Check transaction meta
        if (isset($transaction->meta['is_subscription']) && $transaction->meta['is_subscription']) {
            return true;
        }

        // Check if order has subscription
        $order = $transaction->order;
        if ($order && $order->subscription_id) {
            return true;
        }

        return false;
    }

    /**
     * Handle subscription payment success
     * 
     * Uses FluentCart's SubscriptionService::recordRenewalPayment() for renewal orders
     * and SubscriptionService::syncSubscriptionStates() for status management
     * 
     * Unified logic for activating subscriptions and managing billing cycles
     * Used by NotifyHandler, PaymentStatusChecker, and other payment confirmation handlers
     * 
     * @param OrderTransaction $transaction
     * @param array $paymentData Payment data from WeChat Pay
     * @param string $source Confirmation source (webhook, polling, return_handler)
     * @return bool True if subscription was processed, false otherwise
     */
    public static function handleSubscriptionPaymentSuccess($transaction, $paymentData, $source = 'unknown')
    {
        // Get subscription
        $subscriptionId = self::getSubscriptionId($transaction);
        
        if (!$subscriptionId) {
            Logger::warning('Subscription ID Not Found in Transaction', [
                'transaction_uuid' => $transaction->uuid,
                'source' => $source
            ]);
            return false;
        }

        $subscription = Subscription::find($subscriptionId);
        
        if (!$subscription) {
            Logger::error('Subscription Not Found', [
                'subscription_id' => $subscriptionId,
                'transaction_uuid' => $transaction->uuid,
                'source' => $source
            ]);
            return false;
        }

        $order = $transaction->order;
        $transactionId = $paymentData['transaction_id'] ?? $transaction->vendor_charge_id ?? '';
        $totalFee = $paymentData['total_fee'] ?? $transaction->total;

        Logger::info('Processing Subscription Payment Success', [
            'subscription_id' => $subscriptionId,
            'transaction_id' => $transactionId,
            'order_type' => $order->type ?? 'unknown',
            'current_status' => $subscription->status,
            'source' => $source
        ]);

        // Handle renewal orders using FluentCart standard method
        if ($order && $order->type === 'renewal') {
            return self::handleRenewalWithFluentCart(
                $subscription,
                $transaction,
                $paymentData,
                $source
            );
        }

        // For initial subscription, use FluentCart's syncSubscriptionStates
        return self::handleInitialSubscriptionWithFluentCart(
            $subscription,
            $transaction,
            $source
        );
    }

    /**
     * Handle renewal payment using FluentCart standard method
     * 
     * @param Subscription $subscription
     * @param OrderTransaction $transaction
     * @param array $paymentData
     * @param string $source
     * @return bool
     */
    private static function handleRenewalWithFluentCart($subscription, $transaction, $paymentData, $source)
    {
        // Check if already processed to avoid duplicate renewal records
        if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
            Logger::info('Renewal Transaction Already Processed', [
                'subscription_id' => $subscription->id,
                'transaction_id' => $transaction->id,
                'source' => $source
            ]);
            return true;
        }

        // Prepare transaction data for FluentCart
        $transactionData = [
            'subscription_id' => $subscription->id,
            'vendor_charge_id' => $paymentData['transaction_id'] ?? $transaction->vendor_charge_id,
            'total' => $paymentData['total_fee'] ?? $transaction->total,
            'status' => Status::TRANSACTION_SUCCEEDED,
            'payment_method' => 'wechat',
            'meta' => array_merge($transaction->meta ?? [], [
                'wechat_transaction_id' => $paymentData['transaction_id'] ?? '',
                'wechat_bank_type' => $paymentData['bank_type'] ?? '',
                'wechat_time_end' => $paymentData['time_end'] ?? '',
                'confirmed_via' => $source
            ])
        ];

        // Calculate next billing date
        $nextBillingDate = self::calculateNextBillingDate($subscription);

        // Prepare subscription update args
        $subscriptionUpdateArgs = [
            'next_billing_date' => $nextBillingDate,
            'status' => Status::SUBSCRIPTION_ACTIVE
        ];

        Logger::info('Recording Renewal Payment via FluentCart Standard Method', [
            'subscription_id' => $subscription->id,
            'transaction_uuid' => $transaction->uuid,
            'next_billing_date' => $nextBillingDate,
            'source' => $source
        ]);

        // Use FluentCart's standard method - automatically creates renewal order, order items,
        // transaction record, syncs subscription state, and triggers SubscriptionRenewed event
        $createdTransaction = FluentCartSubscriptionService::recordRenewalPayment(
            $transactionData,
            $subscription,
            $subscriptionUpdateArgs
        );

        if (is_wp_error($createdTransaction)) {
            Logger::error('Failed to Record Renewal Payment via FluentCart', [
                'subscription_id' => $subscription->id,
                'error' => $createdTransaction->get_error_message(),
                'source' => $source
            ]);
            return false;
        }

        Logger::info('Renewal Payment Recorded Successfully via FluentCart', [
            'subscription_id' => $subscription->id,
            'transaction_id' => $createdTransaction->id,
            'wechat_transaction_id' => $paymentData['transaction_id'] ?? '',
            'source' => $source
        ]);

        return true;
    }

    /**
     * Handle initial subscription using FluentCart standard method
     * 
     * @param Subscription $subscription
     * @param OrderTransaction $transaction
     * @param string $source
     * @return bool
     */
    private static function handleInitialSubscriptionWithFluentCart($subscription, $transaction, $source)
    {
        // Prepare subscription update data
        $updateArgs = [
            'status' => Status::SUBSCRIPTION_ACTIVE,
            'next_billing_date' => self::calculateNextBillingDate($subscription)
        ];

        Logger::info('Syncing Initial Subscription State via FluentCart', [
            'subscription_id' => $subscription->id,
            'previous_status' => $subscription->status,
            'source' => $source
        ]);

        // Use FluentCart's syncSubscriptionStates - automatically handles:
        // - Bill count calculation
        // - EOT (End of Term) detection
        // - Next billing date calculation
        // - Status change events
        $subscription = FluentCartSubscriptionService::syncSubscriptionStates(
            $subscription,
            $updateArgs
        );

        Logger::info('Initial Subscription State Synced via FluentCart', [
            'subscription_id' => $subscription->id,
            'status' => $subscription->status,
            'next_billing_date' => $subscription->next_billing_date,
            'source' => $source
        ]);

        return true;
    }

    /**
     * Calculate next billing date based on subscription interval
     * 
     * Uses FluentCart's built-in guessNextBillingDate() method for consistency
     * 
     * @param Subscription $subscription
     * @return string Y-m-d H:i:s format
     */
    private static function calculateNextBillingDate($subscription)
    {
        // Use FluentCart's built-in method which handles:
        // - Trial period calculation
        // - Interval-based date calculation  
        // - Last order date tracking
        // - Edge cases and timezone handling
        return $subscription->guessNextBillingDate(true);
    }

    /**
     * Get subscription ID from transaction
     * 
     * @param OrderTransaction $transaction
     * @return int|null
     */
    public static function getSubscriptionId($transaction)
    {
        // Try transaction meta first
        if (isset($transaction->meta['subscription_id'])) {
            return $transaction->meta['subscription_id'];
        }

        // Try order subscription_id
        if ($transaction->order && $transaction->order->subscription_id) {
            return $transaction->order->subscription_id;
        }

        return null;
    }



    /**
     * Log subscription payment confirmation
     * 
     * Standardized logging for subscription payments
     * 
     * @param int $subscriptionId
     * @param string $transactionId
     * @param string $orderType
     * @param string $source
     * @return void
     */
    public static function logSubscriptionConfirmation($subscriptionId, $transactionId, $orderType, $source)
    {
        Logger::info('Subscription Payment Confirmed', [
            'subscription_id' => $subscriptionId,
            'transaction_id' => $transactionId,
            'order_type' => $orderType,
            'source' => $source
        ]);
    }
}
