<?php

namespace WPKJFluentCart\Wechat\Services;

use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * Payment Page Service
 * 
 * Centralized service for payment page rendering and resource management
 * Eliminates code duplication across H5WaitingPageHandler and NativePageHandler
 */
class PaymentPageService
{
    /**
     * Get sanitized GET parameter
     * 
     * @param string $key Parameter key
     * @param string $default Default value
     * @return string Sanitized parameter value
     */
    public static function getGetParameter(string $key, string $default = ''): string
    {
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Payment page parameters are validated by order/transaction lookup
        return isset($_GET[$key]) ? sanitize_text_field(wp_unslash($_GET[$key])) : $default;
    }

    /**
     * Get multiple GET parameters at once
     * 
     * @param array $keys Array of parameter keys
     * @return array Associative array of sanitized parameters
     */
    public static function getGetParameters(array $keys): array
    {
        $params = [];
        foreach ($keys as $key) {
            $params[$key] = self::getGetParameter($key);
        }
        return $params;
    }

    /**
     * Find order by UUID with error logging
     * 
     * @param string $orderHash Order UUID
     * @param string $context Context for logging (e.g., 'H5 Waiting Page')
     * @return Order|null Order instance or null if not found
     */
    public static function findOrderByHash(string $orderHash, string $context = 'Payment Page'): ?Order
    {
        if (empty($orderHash)) {
            Logger::error("{$context} Missing Order Hash");
            return null;
        }

        $order = Order::query()->where('uuid', $orderHash)->first();
        
        if (!$order) {
            Logger::error("{$context} Order Not Found", [
                'order_hash' => $orderHash
            ]);
            return null;
        }

        return $order;
    }

    /**
     * Find transaction by UUID with error logging
     * 
     * @param string $trxUuid Transaction UUID
     * @param string $context Context for logging
     * @return OrderTransaction|null Transaction instance or null if not found
     */
    public static function findTransactionByUuid(string $trxUuid, string $context = 'Payment Page'): ?OrderTransaction
    {
        if (empty($trxUuid)) {
            Logger::error("{$context} Missing Transaction UUID");
            return null;
        }

        $transaction = OrderTransaction::query()->where('uuid', $trxUuid)->first();
        
        if (!$transaction) {
            Logger::error("{$context} Transaction Not Found", [
                'trx_uuid' => $trxUuid
            ]);
            return null;
        }

        return $transaction;
    }

    /**
     * Calculate file version based on file modification time
     * 
     * @param string $filePath Full file path
     * @param string $fallbackVersion Fallback version if file doesn't exist
     * @return string Version string
     */
    public static function getFileVersion(string $filePath, string $fallbackVersion = WPKJ_FC_WECHAT_VERSION): string
    {
        return file_exists($filePath) ? (string) filemtime($filePath) : $fallbackVersion;
    }

    /**
     * Enqueue CSS file with automatic versioning
     * 
     * @param string $handle Unique handle for the stylesheet
     * @param string $fileName CSS filename (e.g., 'native-payment.css')
     * @param array $dependencies Array of dependency handles
     * @return void
     */
    public static function enqueueStyle(string $handle, string $fileName, array $dependencies = []): void
    {
        $cssFile = WPKJ_FC_WECHAT_PATH . 'assets/css/' . $fileName;
        $cssVersion = self::getFileVersion($cssFile);
        
        wp_enqueue_style(
            $handle,
            WPKJ_FC_WECHAT_URL . 'assets/css/' . $fileName,
            $dependencies,
            $cssVersion
        );
    }

    /**
     * Enqueue JavaScript file with automatic versioning
     * 
     * @param string $handle Unique handle for the script
     * @param string $fileName JS filename (e.g., 'native-payment.js')
     * @param array $dependencies Array of dependency handles
     * @param bool $inFooter Whether to enqueue in footer
     * @return void
     */
    public static function enqueueScript(string $handle, string $fileName, array $dependencies = ['jquery'], bool $inFooter = true): void
    {
        $jsFile = WPKJ_FC_WECHAT_PATH . 'assets/js/' . $fileName;
        $jsVersion = self::getFileVersion($jsFile);
        
        wp_enqueue_script(
            $handle,
            WPKJ_FC_WECHAT_URL . 'assets/js/' . $fileName,
            $dependencies,
            $jsVersion,
            $inFooter
        );
    }

    /**
     * Get common i18n strings for payment pages
     * 
     * @return array Array of translated strings
     */
    public static function getCommonI18n(): array
    {
        return [
            'page_title' => __('WeChat Pay Payment', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'payment_info' => __('Payment Information', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'order_number' => __('Order Number', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'amount' => __('Amount', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'waiting_payment' => __('Waiting for payment...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'payment_success' => __('Payment successful! Redirecting...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'payment_failed' => __('Payment failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
        ];
    }

    /**
     * Get formatted amount display
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @param Order $order Order instance
     * @return string Formatted amount
     */
    public static function getFormattedAmount(OrderTransaction $transaction, Order $order): string
    {
        $amountToDisplay = $transaction->total ?? $order->total ?? 0;
        return \WPKJFluentCart\Wechat\Utils\Helper::formatAmount($amountToDisplay, $order->currency);
    }

    /**
     * Get common AJAX data for payment pages
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @param Order $order Order instance
     * @return array AJAX data array
     */
    public static function getCommonAjaxData(OrderTransaction $transaction, Order $order): array
    {
        return [
            'transaction_uuid' => $transaction->uuid,
            'order_uuid' => $order->uuid,
            'ajax_url' => admin_url('admin-ajax.php'),
            'wpkj_wechat_pay_nonce' => wp_create_nonce('wpkj_wechat_pay_check_status'),
        ];
    }

    /**
     * Render order information section (common HTML)
     * 
     * @param Order $order Order instance
     * @param OrderTransaction $transaction Transaction instance
     * @param array $i18n Internationalization strings
     * @return void
     */
    public static function renderOrderInfo(Order $order, OrderTransaction $transaction, array $i18n): void
    {
        $formattedAmount = self::getFormattedAmount($transaction, $order);
        // Display invoice_no if available, otherwise fall back to ID
        $orderNumber = !empty($order->invoice_no) ? $order->invoice_no : '#' . $order->id;
        ?>
        <div class="order-info">
            <h3><?php echo esc_html($i18n['payment_info']); ?></h3>
            <div class="order-info-row">
                <span class="order-info-label"><?php echo esc_html($i18n['order_number']); ?>:</span>
                <span class="order-info-value"><?php echo esc_html($orderNumber); ?></span>
            </div>
            <div class="order-info-row">
                <span class="order-info-label"><?php echo esc_html($i18n['amount']); ?>:</span>
                <span class="order-info-value amount-value"><?php echo esc_html($formattedAmount); ?></span>
            </div>
        </div>
        <?php
    }

    /**
     * Render HTML head section
     * 
     * @param string $pageTitle Page title
     * @param string $siteName Site name
     * @return void
     */
    public static function renderHtmlHead(string $pageTitle, string $siteName): void
    {
        ?>
<!DOCTYPE html>
<html <?php language_attributes(); ?>>
<head>
    <meta charset="<?php bloginfo('charset'); ?>">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="robots" content="noindex, nofollow">
    <title><?php echo esc_html($pageTitle); ?> - <?php echo esc_html($siteName); ?></title>
    <?php wp_head(); ?>
</head>
<body>
        <?php
    }

    /**
     * Render HTML footer section
     * 
     * @return void
     */
    public static function renderHtmlFooter(): void
    {
        ?>
    <?php wp_footer(); ?>
</body>
</html>
        <?php
    }

    /**
     * Render page header section
     * 
     * @param string $siteName Site name
     * @param string $pageTitle Page title
     * @return void
     */
    public static function renderPageHeader(string $siteName, string $pageTitle): void
    {
        ?>
    <div class="wechat-container">
        <div class="wechat-header">
            <h1><?php echo esc_html($siteName); ?></h1>
            <p><?php echo esc_html($pageTitle); ?></p>
        </div>

        <div class="wechat-body">
        <?php
    }

    /**
     * Render page footer section
     * 
     * @return void
     */
    public static function renderPageFooter(): void
    {
        ?>
        </div>
    </div>
        <?php
    }

    /**
     * Check if payment is already completed and redirect if needed
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @param Order $order Order instance
     * @return bool True if already completed (redirected), false otherwise
     */
    public static function checkAndRedirectIfCompleted(OrderTransaction $transaction, Order $order): bool
    {
        if ($transaction->status === \FluentCart\App\Helpers\Status::TRANSACTION_SUCCEEDED) {
            // Use Transaction's getReceiptPageUrl() method (FluentCart standard)
            $receiptUrl = $transaction->getReceiptPageUrl(true);
            
            // Add order hash for additional tracking
            $receiptUrl = add_query_arg([
                'order_hash' => $order->uuid
            ], $receiptUrl);
            
            wp_safe_redirect($receiptUrl);
            exit;
        }
        return false;
    }
}

