<?php

namespace WPKJFluentCart\Wechat\Processor;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Utils\Helper;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * Refund Processor
 * 
 * Handles automatic refund processing when orders are cancelled
 * Adapted from Alipay RefundProcessor with WeChat Pay specific implementations
 */
class RefundProcessor
{
    /**
     * Settings instance
     * 
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * API instance
     * 
     * @var WechatAPI
     */
    private $api;

    /**
     * Constructor
     */
    public function __construct()
    {
        // Do not initialize API client here to avoid errors when settings are not configured
        // API client will be initialized lazily when needed
    }

    /**
     * Register hooks for automatic refund
     * 
     * @return void
     */
    public function register()
    {
        // Hook into order status change to canceled
        add_action('fluent_cart/order_status_changed_to_canceled', [$this, 'handleOrderStatusChanged'], 10, 1);
    }

    /**
     * Handle order status change to cancelled
     * 
     * @param array $data Order status change data
     * @return void
     */
    public function handleOrderStatusChanged($data)
    {
        if (!$this->isAutoRefundEnabled()) {
            return;
        }

        $order = $data['order'] ?? null;
        
        if (!$order) {
            return;
        }

        // Check if this order uses WeChat Pay gateway
        // Only process refunds for WeChat Pay orders
        if ($order->payment_method !== 'wechat') {
            Logger::info('Skipping WeChat Pay refund - different payment method', [
                'order_id' => $order->id,
                'payment_method' => $order->payment_method ?? 'unknown'
            ]);
            return;
        }

        $this->processAutoRefund($order);
    }

    /**
     * Process automatic refund for cancelled order
     * 
     * Uses FluentCart's standard Refund service for processing refunds.
     * The service handles transaction creation, order updates, and events automatically.
     * 
     * @param Order $order Order instance
     * @return void
     */
    private function processAutoRefund($order)
    {
        // Find successful WeChat Pay transaction
        $transaction = OrderTransaction::query()
            ->where('order_id', $order->id)
            ->where('payment_method', 'wechat')
            ->where('status', Status::TRANSACTION_SUCCEEDED)
            ->where('transaction_type', Status::TRANSACTION_TYPE_CHARGE)
            ->orderBy('id', 'DESC')
            ->first();

        if (!$transaction) {
            $this->addOrderLog($order, 
                __('WeChat Pay Auto-refund Skipped', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                __('No successful WeChat Pay payment transaction found for this order.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            );
            return;
        }

        // CRITICAL SECURITY CHECK: Prevent duplicate refunds
        // Check if this specific transaction has already been refunded automatically
        $existingAutoRefund = OrderTransaction::query()
            ->where('order_id', $order->id)
            ->where('transaction_type', Status::TRANSACTION_TYPE_REFUND)
            ->where('status', Status::TRANSACTION_REFUNDED)
            ->get();

        if ($existingAutoRefund && $existingAutoRefund->count() > 0) {
            foreach ($existingAutoRefund as $refund) {
                $refundMeta = $refund->meta ?? [];
                
                // Check if this is an automatic refund for the same original transaction
                if (isset($refundMeta['refund_type']) && 
                    $refundMeta['refund_type'] === 'automatic' &&
                    isset($refundMeta['original_transaction_uuid']) &&
                    $refundMeta['original_transaction_uuid'] === $transaction->uuid) {
                    
                    Logger::info('Auto-refund Already Processed - Duplicate Prevention', [
                        'order_id' => $order->id,
                        'original_transaction_uuid' => $transaction->uuid,
                        'existing_refund_id' => $refund->id,
                        'existing_refund_amount' => $refund->total,
                        'refunded_at' => $refundMeta['auto_refund_triggered_at'] ?? 'unknown'
                    ]);
                    
                    $this->addOrderLog($order,
                        __('WeChat Pay Auto-refund Skipped', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        sprintf(
                            /* translators: 1: refund ID, 2: refund amount */
                            __('Automatic refund already processed for this transaction (Refund ID: %1$d, Amount: %2$s)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                            $refund->id,
                            Helper::formatAmount($refund->total, $order->currency)
                        )
                    );
                    
                    return;
                }
            }
        }

        // Calculate refund amount
        $refundAmount = $order->total_paid - $order->total_refund;

        if ($refundAmount <= 0) {
            Logger::warning('Auto-refund Skipped - Invalid Refund Amount', [
                'order_id' => $order->id,
                'calculated_refund' => $refundAmount,
                'total_paid' => $order->total_paid,
                'total_refund' => $order->total_refund
            ]);
            return;
        }

        // All checks passed - proceed with refund using FluentCart's standard service
        Logger::info('Auto-refund Security Checks Passed - Using FluentCart Refund Service', [
            'order_id' => $order->id,
            'transaction_uuid' => $transaction->uuid,
            'refund_amount' => $refundAmount,
            'total_paid' => $order->total_paid,
            'total_refund' => $order->total_refund
        ]);

        try {
            // Use FluentCart's standard Refund service
            // This will automatically:
            // 1. Validate the refund amount
            // 2. Call WechatGateway->processRefund() to execute the WeChat Pay API call
            // 3. Create refund transaction record
            // 4. Update order payment status and total_refund
            // 5. Trigger fluent_cart/order_refund_completed event
            // 6. Send refund notification email (if enabled)
            // 7. Restore inventory (if manageStock is enabled)
            $refundService = new \FluentCart\App\Services\Payments\Refund();
            $result = $refundService->processRefund($transaction, $refundAmount, [
                'reason' => 'Order cancelled - automatic refund',
                'manageStock' => false, // Don't restore inventory for cancelled orders
                'refund_type' => 'automatic' // Custom meta to identify automatic refunds
            ]);

            if (isset($result['refund_transaction'])) {
                // Update refund transaction meta to mark as automatic
                $refundTransaction = $result['refund_transaction'];
                $meta = $refundTransaction->meta ?? [];
                $meta['refund_type'] = 'automatic';
                $meta['original_transaction_uuid'] = $transaction->uuid;
                $meta['auto_refund_triggered_at'] = current_time('mysql');
                $refundTransaction->meta = $meta;
                $refundTransaction->save();

                Logger::info('Automatic Refund Successful via FluentCart Service', [
                    'order_id' => $order->id,
                    'refund_transaction_id' => $refundTransaction->id,
                    'vendor_refund_id' => $result['vendor_refund_id'] ?? '',
                    'refund_amount' => $refundAmount
                ]);

                $this->addOrderLog($order,
                    __('WeChat Pay Auto-refund Successful', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    sprintf(
                        /* translators: 1: refund amount with currency, 2: WeChat Pay refund ID */
                        __('Refunded %1$s via WeChat Pay (Refund ID: %2$s)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        Helper::formatAmount($refundAmount, $order->currency),
                        $result['vendor_refund_id'] ?? 'N/A'
                    )
                );
            }

        } catch (\Exception $e) {
            Logger::error('Auto-refund Exception via FluentCart Service', [
                'order_id' => $order->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->addOrderLog($order,
                __('WeChat Pay Auto-refund Exception', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                sprintf(
                    /* translators: %s: exception error message */
                    __('Error: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    $e->getMessage()
                )
            );
        }
    }



    /**
     * Check if auto-refund is enabled
     * 
     * @return bool
     */
    private function isAutoRefundEnabled(): bool
    {
        // Initialize settings if needed
        if ($this->settings === null) {
            $this->settings = new WechatSettingsBase();
        }
        
        return $this->settings->get('auto_refund_on_cancel') === 'yes';
    }



    /**
     * Add order log entry
     * 
     * @param Order $order Order instance
     * @param string $title Log title
     * @param string $message Log message
     * @return void
     */
    private function addOrderLog($order, $title, $message)
    {
        fluent_cart_add_log(
            $title,
            $message,
            strpos($title, 'Failed') !== false || strpos($title, 'Exception') !== false ? 'error' : 'info',
            [
                'module_name' => 'order',
                'module_id' => $order->id,
            ]
        );
    }
}
