<?php

namespace WPKJFluentCart\Wechat\Processor;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Helpers\StatusHelper;
use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use FluentCart\App\Services\Payments\PaymentHelper;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Services\OrderService;
use WPKJFluentCart\Wechat\Services\SubscriptionService;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * Payment Status Checker
 * 
 * Handles AJAX requests to check Native payment status
 * Adapted from Alipay PaymentStatusChecker
 */
class PaymentStatusChecker
{
    /**
     * Settings instance
     * 
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * API instance
     * 
     * @var WechatAPI
     */
    private $api;

    /**
     * Constructor
     */
    public function __construct()
    {
        // Do not initialize API client here to avoid errors when settings are not configured
        // API client will be initialized lazily when needed
    }

    /**
     * Register AJAX hooks
     * 
     * @return void
     */
    public function register()
    {
        add_action('wp_ajax_wpkj_wechat_check_payment_status', [$this, 'checkPaymentStatus']);
        add_action('wp_ajax_nopriv_wpkj_wechat_check_payment_status', [$this, 'checkPaymentStatus']);
    }

    /**
     * Initialize API client lazily
     * 
     * @return void
     */
    private function initializeApi()
    {
        if ($this->api !== null) {
            return;
        }

        $this->settings = new WechatSettingsBase();
        
        // Get payment mode from FluentCart global settings
        $paymentMode = fluent_cart_get_option('order_mode', 'live');
        
        // Build API config
        $config = [
            'appid' => $this->settings->getAppId(),
            'mch_id' => $this->settings->getMchId(),
            'api_key' => $this->settings->getApiKey(),
            'sign_type' => $this->settings->getSignType(),
            'mode' => $paymentMode,
        ];
        
        $this->api = new WechatAPI($config);
    }

    /**
     * Check payment status via AJAX
     * 
     * @return void
     */
    public function checkPaymentStatus()
    {
        try {
            // Initialize API client
            $this->initializeApi();
            // Verify nonce for CSRF protection
            $nonce = isset($_POST['wpkj_wechat_pay_nonce']) ? sanitize_text_field(wp_unslash($_POST['wpkj_wechat_pay_nonce'])) : '';
            if (empty($nonce) || !wp_verify_nonce($nonce, 'wpkj_wechat_pay_check_status')) {
                wp_send_json_error([
                    'message' => __('Security verification failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                ], 403);
                return;
            }
            
            $transactionUuid = isset($_POST['transaction_uuid']) ? sanitize_text_field(wp_unslash($_POST['transaction_uuid'])) : '';

            if (empty($transactionUuid)) {
                wp_send_json_error([
                    'message' => __('Invalid transaction ID', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                ]);
                return;
            }

            $transaction = OrderTransaction::query()
                ->where('uuid', $transactionUuid)
                ->first();

            if (!$transaction) {
                wp_send_json_error([
                    'message' => __('Transaction not found', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                ]);
                return;
            }

            // Check if payment is already completed
            if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
                $order = Order::find($transaction->order_id);
                
                // Use Transaction's getReceiptPageUrl() method (FluentCart standard)
                $receiptUrl = $transaction->getReceiptPageUrl(true);
                
                // Add order hash for additional tracking
                $receiptUrl = add_query_arg([
                    'order_hash' => $order->uuid
                ], $receiptUrl);
                
                wp_send_json_success([
                    'status' => 'paid',
                    'message' => __('Payment completed successfully', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    'redirect_url' => $receiptUrl
                ]);
                return;
            }

            // Get out_trade_no from transaction meta
            $outTradeNo = $transaction->meta['out_trade_no'] ?? null;
            
            if (empty($outTradeNo)) {
                Logger::warning('Missing out_trade_no in Transaction Meta', [
                    'transaction_uuid' => $transaction->uuid,
                    'transaction_id' => $transaction->id
                ]);
                
                wp_send_json_success([
                    'status' => 'waiting',
                    'message' => __('Waiting for payment', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                ]);
                return;
            }
            
            // Query payment status from WeChat Pay
            $result = $this->api->queryPayment($outTradeNo);

            Logger::info('WeChat Query Result', [
                'out_trade_no' => $outTradeNo,
                'is_error' => is_wp_error($result),
                'trade_state' => is_array($result) ? ($result['trade_state'] ?? 'N/A') : 'ERROR',
                'result_code' => is_array($result) ? ($result['result_code'] ?? 'N/A') : 'ERROR'
            ]);

            if (is_wp_error($result)) {
                Logger::warning('Query Payment WP_Error', [
                    'code' => $result->get_error_code(),
                    'message' => $result->get_error_message()
                ]);
                wp_send_json_success([
                    'status' => 'waiting',
                    'message' => __('Waiting for payment', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                ]);
                return;
            }

            // Check trade_state
            if (isset($result['trade_state'])) {
                $tradeState = $result['trade_state'];

                if ($tradeState === 'SUCCESS') {
                    // Process payment confirmation immediately
                    $confirmed = $this->processPaymentConfirmation($transaction, $result);
                    
                    if (!$confirmed) {
                        wp_send_json_success([
                            'status' => 'waiting',
                            'message' => __('Payment verification in progress...', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                        ]);
                        return;
                    }
                    
                    // Reload transaction and order
                    $transaction = $transaction->fresh();
                    $order = $transaction->order;
                    
                    Logger::info('Native Payment Confirmed via Status Check', [
                        'transaction_uuid' => $transaction->uuid,
                        'order_id' => $order->id,
                        'trade_state' => $tradeState
                    ]);
                    
                    // Use Transaction's getReceiptPageUrl() method (FluentCart standard)
                    $receiptUrl = $transaction->getReceiptPageUrl(true);
                    
                    // Add order hash for additional tracking
                    $receiptUrl = add_query_arg([
                        'order_hash' => $order->uuid
                    ], $receiptUrl);
                    
                    wp_send_json_success([
                        'status' => 'paid',
                        'message' => __('Payment completed successfully', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        'redirect_url' => $receiptUrl
                    ]);
                    return;
                    
                } elseif ($tradeState === 'CLOSED' || $tradeState === 'REVOKED' || $tradeState === 'PAYERROR') {
                    wp_send_json_success([
                        'status' => 'failed',
                        'message' => __('Payment cancelled or failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                    ]);
                    return;
                }
            }

            wp_send_json_success([
                'status' => 'waiting',
                'message' => __('Waiting for payment', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ]);

        } catch (\Exception $e) {
            Logger::error('Payment Status Check Error', [
                'message' => $e->getMessage(),
                'transaction_uuid' => $transactionUuid ?? ''
            ]);

            wp_send_json_error([
                'message' => __('Failed to check payment status', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ]);
        }
    }



    /**
     * Process payment confirmation
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @param array $tradeData Trade data from WeChat Pay
     * @return bool True if confirmation succeeded, false otherwise
     */
    private function processPaymentConfirmation($transaction, $tradeData)
    {
        // Check if already processed
        if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
            Logger::info('Transaction Already Completed', [
                'transaction_uuid' => $transaction->uuid
            ]);
            return true;
        }

        $order = Order::find($transaction->order_id);
        if (!$order) {
            Logger::error('Order Not Found for Payment Confirmation', [
                'transaction_id' => $transaction->id,
                'order_id' => $transaction->order_id
            ]);
            return false;
        }

        // Verify amount
        if (isset($tradeData['total_fee'])) {
            $receivedAmount = (int) $tradeData['total_fee'];
            $expectedAmount = (int) $transaction->total; // Use 'total' field (in cents)
            
            if ($expectedAmount !== $receivedAmount) {
                Logger::error('Amount Mismatch in Status Check', [
                    'expected' => $expectedAmount,
                    'received' => $receivedAmount,
                    'transaction_uuid' => $transaction->uuid,
                    'transaction_id' => $tradeData['transaction_id'] ?? 'N/A'
                ]);
                return false;
            }
        }

        // Update transaction
        $transaction->status = Status::TRANSACTION_SUCCEEDED;
        $transaction->vendor_charge_id = $tradeData['transaction_id'] ?? '';
        $transaction->meta = array_merge($transaction->meta ?? [], [
            'wechat_transaction_id' => $tradeData['transaction_id'] ?? '',
            'wechat_bank_type' => $tradeData['bank_type'] ?? '',
            'wechat_time_end' => $tradeData['time_end'] ?? '',
            'wechat_cash_fee' => $tradeData['cash_fee'] ?? $tradeData['total_fee'],
            'confirmed_via' => 'status_check'
        ]);
        $transaction->save();

        // Add log to order activity
        fluent_cart_add_log(
            __('WeChat Pay Payment Confirmation', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            sprintf(
                /* translators: %s: WeChat Pay transaction ID */
                __('Payment confirmed from WeChat Pay. Transaction ID: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $tradeData['transaction_id'] ?? ''
            ),
            'info',
            [
                'module_name' => 'order',
                'module_id' => $order->id,
            ]
        );

        // Sync order statuses
        (new StatusHelper($order))->syncOrderStatuses($transaction);

        // Clear cart order association
        OrderService::clearCartOrderAssociation($order, 'wechat_status_check');

        // Handle subscription status update
        if (SubscriptionService::isSubscriptionTransaction($transaction)) {
            SubscriptionService::handleSubscriptionPaymentSuccess($transaction, $tradeData, 'polling');
        }

        Logger::info('Payment Confirmed via Status Check', [
            'order_uuid' => $order->uuid,
            'transaction_uuid' => $transaction->uuid,
            'transaction_id' => $tradeData['transaction_id'] ?? '',
            'total_fee' => $tradeData['total_fee'] ?? 0
        ]);

        return true;
    }


}
