<?php

namespace WPKJFluentCart\Wechat\Processor;

use FluentCart\App\Helpers\Status;
use FluentCart\App\Helpers\StatusHelper;
use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use FluentCart\App\Services\Payments\PaymentInstance;
use FluentCart\App\Services\Payments\PaymentHelper;
use WPKJFluentCart\Wechat\API\WechatAPI;
use WPKJFluentCart\Wechat\Config\WechatConfig;
use WPKJFluentCart\Wechat\Gateway\WechatSettingsBase;
use WPKJFluentCart\Wechat\Detector\ClientDetector;
use WPKJFluentCart\Wechat\Services\OrderService;
use WPKJFluentCart\Wechat\Services\EncodingService;
use WPKJFluentCart\Wechat\Utils\Helper;
use WPKJFluentCart\Wechat\Utils\Logger;
use FluentCart\Framework\Support\Arr;

/**
 * Payment Processor
 * 
 * Handles payment processing logic for WeChat Pay
 * Adapted from Alipay PaymentProcessor with WeChat Pay specific implementations
 */
class PaymentProcessor
{
    /**
     * Settings instance
     * 
     * @var WechatSettingsBase
     */
    private $settings;

    /**
     * API instance
     * 
     * @var WechatAPI
     */
    private $api;

    /**
     * Constructor
     * 
     * @param WechatSettingsBase $settings Settings instance
     */
    public function __construct(WechatSettingsBase $settings)
    {
        $this->settings = $settings;
        
        // Do not initialize API client here to avoid errors when settings are not configured
        // API client will be initialized lazily when needed
    }

    /**
     * Initialize API client lazily
     * 
     * @return void
     */
    private function initializeApi()
    {
        if ($this->api !== null) {
            return;
        }

        // Get payment mode from FluentCart global settings
        $paymentMode = fluent_cart_get_option('order_mode', 'live');

        // Build API config
        $config = [
            'appid' => $this->settings->getAppId(),
            'mch_id' => $this->settings->getMchId(),
            'api_key' => $this->settings->getApiKey(),
            'sign_type' => $this->settings->getSignType(),
            'ssl_cert_path' => $this->settings->getSslCertPath(),
            'ssl_key_path' => $this->settings->getSslKeyPath(),
            'mode' => $paymentMode,
        ];
        
        $this->api = new WechatAPI($config);
    }

    /**
     * Process single payment
     * 
     * @param PaymentInstance $paymentInstance Payment instance
     * @return array Payment response
     */
    public function processSinglePayment(PaymentInstance $paymentInstance)
    {
        $transaction = $paymentInstance->transaction;
        $order = $paymentInstance->order;

        try {
            // Initialize API client
            $this->initializeApi();
            
            // Check if this is a subscription order
            if ($paymentInstance->subscription) {
                Logger::info('Routing to Subscription Processor', [
                    'order_id' => $order->id,
                    'subscription_id' => $paymentInstance->subscription->id
                ]);
                
                $subscriptionProcessor = new \WPKJFluentCart\Wechat\Subscription\WechatSubscriptionProcessor($this->settings);
                return $subscriptionProcessor->processSubscription($paymentInstance);
            }
            // Check if transaction is already completed
            if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
                throw new \Exception(
                    __('Transaction has already been completed.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            // Log attempt on completed orders
            if (in_array($order->status, ['completed', 'processing'])) {
                Logger::warning('Payment Attempt on Completed Order', [
                    'order_uuid' => $order->uuid,
                    'order_status' => $order->status,
                    'transaction_status' => $transaction->status
                ]);
            }

            // Build payment data
            $paymentData = $this->buildPaymentData($paymentInstance);

            // Detect client type and choose appropriate payment method
            $tradeType = ClientDetector::getTradeType($this->settings);

            // Process based on trade type
            if ($tradeType === WechatConfig::TRADE_TYPE_NATIVE) {
                // PC QR code payment
                return $this->processNativePayment($paymentInstance, $paymentData);
            } elseif ($tradeType === WechatConfig::TRADE_TYPE_MWEB) {
                // Mobile H5 payment
                return $this->processH5Payment($paymentInstance, $paymentData);
            } elseif ($tradeType === WechatConfig::TRADE_TYPE_JSAPI) {
                // WeChat browser JSAPI payment
                return $this->processJsapiPayment($paymentInstance, $paymentData);
            }

            throw new \Exception(__('Unable to determine payment method', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));

        } catch (\Exception $e) {
            Logger::error('Payment Processing Error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id,
                'transaction_id' => $transaction->id
            ]);

            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Process Native payment (PC QR code)
     * 
     * @param PaymentInstance $paymentInstance Payment instance
     * @param array $paymentData Payment data
     * @return array Payment response
     */
    private function processNativePayment(PaymentInstance $paymentInstance, array $paymentData)
    {
        $transaction = $paymentInstance->transaction;
        $order = $paymentInstance->order;

        try {
            $result = $this->api->createNativePayment($paymentData);

            if (is_wp_error($result)) {
                throw new \Exception($result->get_error_message());
            }

            // Save payment info to transaction meta
            $transaction->meta = array_merge($transaction->meta ?? [], [
                'code_url' => $result['code_url'],
                'prepay_id' => $result['prepay_id'],
                'trade_type' => WechatConfig::TRADE_TYPE_NATIVE,
                'out_trade_no' => $paymentData['out_trade_no']
            ]);
            $transaction->save();

            // Build QR code page URL
            $qrPageUrl = home_url('/');
            $qrCodeEncoded = base64_encode($result['code_url']);
            $qrPageUrl = add_query_arg([
                'fluent-cart' => 'wechat_qr_payment',
                'order_hash' => $order->uuid,
                'qr_code' => $qrCodeEncoded,
                'trx_uuid' => $transaction->uuid
            ], $qrPageUrl);

            return [
                'status' => 'success',
                'message' => __('Redirecting to payment page...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'redirect_to' => $qrPageUrl
            ];

        } catch (\Exception $e) {
            Logger::error('Native Payment Error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id
            ]);

            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Process H5 payment (Mobile browser)
     * 
     * @param PaymentInstance $paymentInstance Payment instance
     * @param array $paymentData Payment data
     * @return array Payment response
     */
    private function processH5Payment(PaymentInstance $paymentInstance, array $paymentData)
    {
        $transaction = $paymentInstance->transaction;
        $order = $paymentInstance->order;

        try {
            // Add scene_info for H5 payment
            $paymentData['scene_info'] = [
                'h5_info' => [
                    'type' => 'Wap',
                    'wap_url' => home_url('/'),
                    'wap_name' => get_bloginfo('name')
                ]
            ];

            $result = $this->api->createH5Payment($paymentData);

            if (is_wp_error($result)) {
                throw new \Exception($result->get_error_message());
            }

            // Save payment info to transaction meta
            $transaction->meta = array_merge($transaction->meta ?? [], [
                'mweb_url' => $result['mweb_url'],
                'prepay_id' => $result['prepay_id'],
                'trade_type' => WechatConfig::TRADE_TYPE_MWEB,
                'out_trade_no' => $paymentData['out_trade_no']
            ]);
            $transaction->save();

            // Add redirect_url parameter for returning to site after payment
            $returnUrl = add_query_arg([
                'trx_hash' => $transaction->uuid,
                'fct_redirect' => 'yes',
                'out_trade_no' => $paymentData['out_trade_no']
            ], home_url('/'));

            $mwebUrl = $result['mweb_url'] . '&redirect_url=' . urlencode($returnUrl);

            return [
                'status' => 'success',
                'nextAction' => 'redirect',
                'actionName' => 'custom',
                'message' => __('Redirecting to WeChat Pay...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'data' => [
                    'order' => [
                        'uuid' => $order->uuid,
                    ],
                    'transaction' => [
                        'uuid' => $transaction->uuid,
                    ]
                ],
                'redirect_to' => $mwebUrl,
                'custom_payment_url' => PaymentHelper::getCustomPaymentLink($order->uuid)
            ];

        } catch (\Exception $e) {
            Logger::error('H5 Payment Error', [
                'message' => $e->getMessage(),
                'order_id' => $order->id
            ]);

            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Process JSAPI payment (WeChat browser)
     * 
     * @param PaymentInstance $paymentInstance Payment instance
     * @param array $paymentData Payment data
     * @return array Payment response
     */
    private function processJsapiPayment(PaymentInstance $paymentInstance, array $paymentData)
    {
        // JSAPI payment requires openid, which needs additional implementation
        // For now, return error message
        Logger::error('JSAPI Payment Not Implemented', [
            'order_id' => $paymentInstance->order->id
        ]);

        return [
            'status' => 'failed',
            'message' => __('JSAPI payment is not yet implemented. Please use a different browser or device.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
        ];
    }

    /**
     * Build payment data array
     * 
     * @param PaymentInstance $paymentInstance Payment instance
     * @return array Payment data
     */
    private function buildPaymentData(PaymentInstance $paymentInstance): array
    {
        $transaction = $paymentInstance->transaction;
        $order = $paymentInstance->order;

        // Generate unique out_trade_no
        $outTradeNo = Helper::generateOutTradeNo($transaction->uuid);

        // Get order subject/body
        $subject = $this->getOrderSubject($order);
        $body = EncodingService::sanitizeForWechat($subject, WechatConfig::MAX_BODY_LENGTH);

        // Build notify URL
        $notifyUrl = add_query_arg([
            'fct_payment_listener' => '1',
            'method' => 'wechat'
        ], site_url('/'));

        // Amount in cents (WeChat Pay requires integer cents)
        // FluentCart stores amounts in cents, use 'total' field
        $totalFee = (int) $transaction->total;

        // Validate minimum amount
        if ($totalFee < WechatConfig::MIN_PAYMENT_AMOUNT_CENTS) {
            throw new \Exception(
                sprintf(
                    /* translators: 1: current payment amount, 2: minimum required amount */
                    esc_html__('Payment amount (%1$s) is below minimum. Minimum: %2$s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html(Helper::formatAmount($totalFee)),
                    esc_html(Helper::formatAmount(WechatConfig::MIN_PAYMENT_AMOUNT_CENTS))
                )
            );
        }

        // Build payment data
        $paymentData = [
            'body' => $body,
            'out_trade_no' => $outTradeNo,
            'total_fee' => $totalFee,
            'notify_url' => $notifyUrl,
        ];

        // Add timeout (ensure at least 5 minutes for non-MICROPAY transactions)
        // WeChat Pay requires: MICROPAY >= 1min, others >= 5min
        // IMPORTANT: WeChat Pay uses Beijing time (GMT+8), calculate based on WordPress timezone
        $timeoutMinutes = max(WechatConfig::PAYMENT_TIMEOUT_MINUTES, WechatConfig::MIN_PAYMENT_TIMEOUT_MINUTES);
        
        // Calculate time_expire using WordPress timezone functions
        // WeChat Pay expects YmdHis format in GMT+8 (Beijing time)
        $beijingTimestamp = current_time('timestamp') + (8 * HOUR_IN_SECONDS) + ($timeoutMinutes * 60);
        $timeExpire = gmdate('YmdHis', $beijingTimestamp);
        
        $paymentData['time_expire'] = $timeExpire;

        return $paymentData;
    }

    /**
     * Get order subject
     * 
     * @param Order $order Order instance
     * @return string Order subject
     */
    private function getOrderSubject(Order $order): string
    {        // Try to get first item name
        if (!empty($order->orderMetas) && is_array($order->orderMetas)) {
            $items = Arr::get($order->orderMetas, 'cart_items', []);
            if (!empty($items) && is_array($items)) {
                $firstItem = reset($items);
                if (isset($firstItem['title'])) {
                    $itemName = $firstItem['title'];
                    $itemCount = count($items);
                    
                    if ($itemCount > 1) {
                        return sprintf(
                            /* translators: 1: first item name, 2: number of additional items */
                            __('%1$s and %2$d more items', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                            $itemName,
                            $itemCount - 1
                        );
                    }
                    
                    return $itemName;
                }
            }
        }

        // Fallback to order number or order ID
        if (!empty($order->order_no)) {
            return sprintf(
                /* translators: %s: order number */
                __('Order #%s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $order->order_no
            );
        }
        
        // Use order ID as last resort
        return sprintf(
            /* translators: %d: order ID */
            __('Order %d', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            $order->id
        );
    }

    /**
     * Confirm payment success
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @param array $wechatData WeChat Pay response data
     * @return void
     */
    public function confirmPaymentSuccess(OrderTransaction $transaction, $wechatData)
    {
        // Check if already processed
        if ($transaction->status === Status::TRANSACTION_SUCCEEDED) {
            return;
        }

        $order = Order::query()->where('id', $transaction->order_id)->first();

        if (!$order) {
            Logger::error('Order Not Found', [
                'transaction_id' => $transaction->id,
                'order_id' => $transaction->order_id
            ]);
            return;
        }

        // Verify payment amount
        $totalFee = isset($wechatData['total_fee']) ? (int) $wechatData['total_fee'] : 0;
        
        // Convert both to integers for comparison to avoid type mismatch
        $expectedAmount = (int)$transaction->total;
        $receivedAmount = (int)$totalFee;
        
        if ($expectedAmount !== $receivedAmount) {
            Logger::error('Amount mismatch', [
                'expected' => $expectedAmount,
                'received' => $receivedAmount,
                'transaction_uuid' => $transaction->uuid
            ]);
            return;
        }

        // Check if this is a subscription payment
        $isSubscription = isset($transaction->meta['is_subscription']) && $transaction->meta['is_subscription'];
        $subscriptionId = $transaction->meta['subscription_id'] ?? null;

        // Update transaction
        $transaction->status = Status::TRANSACTION_SUCCEEDED;
        $transaction->vendor_charge_id = $wechatData['transaction_id'] ?? '';
        $transaction->meta = array_merge($transaction->meta ?? [], [
            'wechat_transaction_id' => $wechatData['transaction_id'] ?? '',
            'wechat_bank_type' => $wechatData['bank_type'] ?? '',
            'wechat_time_end' => $wechatData['time_end'] ?? '',
            'wechat_cash_fee' => $wechatData['cash_fee'] ?? $totalFee,
            'confirmed_via' => 'return_handler'
        ]);
        $transaction->save();

        // Log subscription payment if applicable
        if ($isSubscription && $subscriptionId) {
            Logger::info('Subscription Payment Confirmed', [
                'subscription_id' => $subscriptionId,
                'transaction_id' => $wechatData['transaction_id'] ?? '',
                'order_type' => $order->type ?? 'unknown'
            ]);
        }

        // Add log to order activity
        fluent_cart_add_log(
            __('WeChat Pay Payment Confirmation', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            sprintf(
                /* translators: %s: WeChat Pay transaction ID */
                __('Payment confirmed from WeChat Pay. Transaction ID: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $wechatData['transaction_id'] ?? ''
            ),
            'info',
            [
                'module_name' => 'order',
                'module_id' => $order->id,
            ]
        );

        // Sync order statuses
        (new StatusHelper($order))->syncOrderStatuses($transaction);
        
        // Clear cart order association
        OrderService::clearCartOrderAssociation($order, 'payment_confirmation');
    }

    /**
     * Process failed payment
     * 
     * @param OrderTransaction $transaction Transaction instance
     * @param array $data Failure data
     * @return void
     */
    public function processFailedPayment(OrderTransaction $transaction, $data)
    {
        $order = Order::query()->where('id', $transaction->order_id)->first();

        if (!$order) {
            return;
        }

        Logger::error('Payment Failed', [
            'transaction_uuid' => $transaction->uuid,
            'reason' => $data['reason'] ?? 'Unknown'
        ]);

        // Add log to order activity
        fluent_cart_add_log(
            __('WeChat Pay Payment Failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            sprintf(
                /* translators: %s: failure reason */
                __('Payment failed. Reason: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                $data['reason'] ?? 'Unknown'
            ),
            'error',
            [
                'module_name' => 'order',
                'module_id' => $order->id,
            ]
        );
    }
}
