<?php

namespace WPKJFluentCart\Wechat\Processor;

use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use WPKJFluentCart\Wechat\Services\PaymentPageService;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * Native Payment Page Handler
 * 
 * Intercepts custom checkout page requests and renders QR code payment interface
 * Adapted from Alipay FaceToFacePageHandler
 */
class NativePageHandler
{
    /**
     * Register hooks
     * 
     * @return void
     */
    public function register()
    {
        add_action('template_redirect', [$this, 'handleNativePayment'], 1);
    }

    /**
     * Handle Native payment page display
     * 
     * @return void
     */
    public function handleNativePayment()
    {
        // Check for our custom Native payment route
        $fluentCartParam = PaymentPageService::getGetParameter('fluent-cart');
        if ($fluentCartParam !== 'wechat_qr_payment') {
            return;
        }

        $params = PaymentPageService::getGetParameters(['order_hash', 'qr_code', 'trx_uuid']);
        $orderHash = $params['order_hash'];
        $qrCodeEncoded = $params['qr_code'];
        $trxUuid = $params['trx_uuid'];

        if (empty($orderHash) || empty($qrCodeEncoded) || empty($trxUuid)) {
            Logger::error('Native Payment Handler Missing Parameters', [
                'order_hash' => empty($orderHash) ? 'EMPTY' : 'OK',
                'qr_code' => empty($qrCodeEncoded) ? 'EMPTY' : 'OK',
                'trx_uuid' => empty($trxUuid) ? 'EMPTY' : 'OK'
            ]);
            return;
        }

        $order = PaymentPageService::findOrderByHash($orderHash, 'Native Payment Handler');
        if (!$order) {
            return;
        }

        $transaction = PaymentPageService::findTransactionByUuid($trxUuid, 'Native Payment Handler');
        if (!$transaction) {
            return;
        }

        $qrCode = base64_decode($qrCodeEncoded);
        if (empty($qrCode)) {
            Logger::error('Native Payment Handler QR Code Decode Failed', [
                'encoded_length' => strlen($qrCodeEncoded)
            ]);
            return;
        }

        // If payment already completed, redirect immediately
        PaymentPageService::checkAndRedirectIfCompleted($transaction, $order);

        $this->renderPaymentPage($order, $transaction, $qrCode);
        exit;
    }

    /**
     * Render Native payment page
     * 
     * @param Order $order Order instance
     * @param OrderTransaction $transaction Transaction instance
     * @param string $qrCode QR code URL
     * @return void
     */
    private function renderPaymentPage($order, $transaction, $qrCode)
    {
        $siteName = get_bloginfo('name');
        
        // Enqueue jQuery
        wp_enqueue_script('jquery');
        
        // Enqueue QRCode.js library (local copy)
        $qrCodeFile = WPKJ_FC_WECHAT_PATH . 'assets/js/vendor/qrcode.min.js';
        $qrCodeVersion = file_exists($qrCodeFile) ? filemtime($qrCodeFile) : WPKJ_FC_WECHAT_VERSION;
        wp_enqueue_script(
            'qrcodejs',
            WPKJ_FC_WECHAT_URL . 'assets/js/vendor/qrcode.min.js',
            [],
            $qrCodeVersion,
            true
        );
        
        // Enqueue CSS
        PaymentPageService::enqueueStyle(
            'wpkj-fc-wechat-native-styles',
            'native-payment.css'
        );
        
        // Enqueue JavaScript with QRCode.js as dependency
        $jsFile = WPKJ_FC_WECHAT_PATH . 'assets/js/native-payment.js';
        $jsVersion = file_exists($jsFile) ? filemtime($jsFile) : WPKJ_FC_WECHAT_VERSION;
        wp_enqueue_script(
            'wpkj-fc-wechat-native-script',
            WPKJ_FC_WECHAT_URL . 'assets/js/native-payment.js',
            ['jquery', 'qrcodejs'], // Ensure QRCode.js loads before our script
            $jsVersion,
            true
        );

        // Prepare i18n strings
        $i18n = array_merge(PaymentPageService::getCommonI18n(), [
            'scan_title' => __('Scan QR Code to Pay with WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'scan_instruction' => __('Please use WeChat to scan the QR code and complete payment', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'payment_timeout' => __('Payment timeout, please refresh and try again', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'auto_refresh_notice' => __('The page will automatically refresh after payment is completed', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'security_notice' => __('Secure payment service provided by WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
        ]);
        
        // Localize script data
        $ajaxData = array_merge(PaymentPageService::getCommonAjaxData($transaction, $order), [
            'qr_code' => $qrCode,
            'i18n' => $i18n
        ]);
        
        wp_localize_script(
            'wpkj-fc-wechat-native-script',
            'wpkj_wechat_native_data',
            $ajaxData
        );

        // Render HTML
        PaymentPageService::renderHtmlHead($i18n['scan_title'], $siteName);
        ?>
    <div class="wechat-container">
        <div class="wechat-header">
            <h1><?php echo esc_html($i18n['scan_title']); ?></h1>
            <p><?php echo esc_html($i18n['scan_instruction']); ?></p>
        </div>

        <div class="wechat-body">
        <?php
        
        // Render order info
        PaymentPageService::renderOrderInfo($order, $transaction, $i18n);

        // Render QR code and status
        ?>
            <div class="qr-container">
                <div id="wpkj-wechat-qrcode"></div>
            </div>

            <div class="status-container">
                <div id="payment-status">
                    <span class="loading-spinner"></span>
                    <span class="status-text"><?php echo esc_html($i18n['waiting_payment']); ?></span>
                </div>
            </div>

            <div class="instruction">
                <?php echo esc_html($i18n['auto_refresh_notice']); ?>
            </div>

            <div class="security-footer">
                <svg class="security-shield" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="16" height="16" fill="#09BB07">
                    <path d="M12 2L4 5v6.59c0 5.05 3.41 9.76 8 10.91 4.59-1.15 8-5.86 8-10.91V5l-8-3zm6 9.09c0 4-2.55 7.7-6 8.83-3.45-1.13-6-4.82-6-8.83V6.31l6-2.12 6 2.12v4.78z"/>
                    <path d="M9.5 11.5l1.41 1.41L15.5 8.34 14.09 6.93l-3.18 3.18-1.41-1.42z"/>
                </svg>
                <span class="security-text"><?php echo esc_html($i18n['security_notice']); ?></span>
            </div>
        </div>
    </div>
        <?php
        
        PaymentPageService::renderHtmlFooter();
    }
}
