<?php

namespace WPKJFluentCart\Wechat\Processor;

use FluentCart\App\Models\Order;
use FluentCart\App\Models\OrderTransaction;
use WPKJFluentCart\Wechat\Services\PaymentPageService;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * H5 Payment Waiting Page Handler
 * 
 * Handles H5 payment waiting page with status polling
 */
class H5WaitingPageHandler
{
    /**
     * Register hooks
     * 
     * @return void
     */
    public function register()
    {
        add_action('template_redirect', [$this, 'handleWaitingPage'], 1);
    }

    /**
     * Handle waiting page display
     * 
     * @return void
     */
    public function handleWaitingPage()
    {
        // Check for our custom waiting page route
        $fluentCartParam = PaymentPageService::getGetParameter('fluent-cart');
        if ($fluentCartParam !== 'wechat_payment_waiting') {
            return;
        }

        $params = PaymentPageService::getGetParameters(['order_hash', 'trx_uuid']);
        $orderHash = $params['order_hash'];
        $trxUuid = $params['trx_uuid'];

        if (empty($orderHash) || empty($trxUuid)) {
            Logger::error('H5 Waiting Page Handler Missing Parameters', [
                'order_hash' => empty($orderHash) ? 'EMPTY' : 'OK',
                'trx_uuid' => empty($trxUuid) ? 'EMPTY' : 'OK'
            ]);
            return;
        }

        $order = PaymentPageService::findOrderByHash($orderHash, 'H5 Waiting Page Handler');
        if (!$order) {
            return;
        }

        $transaction = PaymentPageService::findTransactionByUuid($trxUuid, 'H5 Waiting Page Handler');
        if (!$transaction) {
            return;
        }

        // If payment already completed, redirect immediately
        PaymentPageService::checkAndRedirectIfCompleted($transaction, $order);

        $this->renderWaitingPage($order, $transaction);
        exit;
    }

    /**
     * Render H5 payment waiting page
     * 
     * @param Order $order Order instance
     * @param OrderTransaction $transaction Transaction instance
     * @return void
     */
    private function renderWaitingPage($order, $transaction)
    {
        $siteName = get_bloginfo('name');
        
        // Enqueue jQuery
        wp_enqueue_script('jquery');
        
        // Enqueue CSS (reuse Native payment CSS)
        PaymentPageService::enqueueStyle(
            'wpkj-fc-wechat-waiting-styles',
            'native-payment.css'
        );
        
        // Enqueue JavaScript for status polling
        PaymentPageService::enqueueScript(
            'wpkj-fc-wechat-h5-waiting-script',
            'h5-waiting.js'
        );

        // Prepare i18n strings
        $i18n = array_merge(PaymentPageService::getCommonI18n(), [
            'waiting_payment' => __('Waiting for payment confirmation...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'payment_failed' => __('Payment not completed', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'instruction' => __('Please wait while we confirm your payment...', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'retry_payment' => __('Pay Again', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
        ]);
        
        // Localize script data
        $ajaxData = array_merge(PaymentPageService::getCommonAjaxData($transaction, $order), [
            'custom_payment_url' => \FluentCart\App\Services\Payments\PaymentHelper::getCustomPaymentLink($order->uuid),
            'i18n' => $i18n
        ]);
        
        wp_localize_script(
            'wpkj-fc-wechat-h5-waiting-script',
            'wpkj_wechat_h5_data',
            $ajaxData
        );

        // Render HTML
        PaymentPageService::renderHtmlHead($i18n['page_title'], $siteName);
        PaymentPageService::renderPageHeader($siteName, $i18n['page_title']);
        
        // Render order info
        PaymentPageService::renderOrderInfo($order, $transaction, $i18n);

        // Render status container
        ?>
            <div class="status-container">
                <div id="payment-status">
                    <span class="loading-spinner"></span>
                    <span class="status-text"><?php echo esc_html($i18n['waiting_payment']); ?></span>
                </div>
            </div>

            <p class="instruction"><?php echo esc_html($i18n['instruction']); ?></p>
        <?php
        
        PaymentPageService::renderPageFooter();
        PaymentPageService::renderHtmlFooter();
    }
}
