<?php

namespace WPKJFluentCart\Wechat\Gateway;

use FluentCart\App\Helpers\Helper as FluentCartHelper;
use FluentCart\App\Modules\PaymentMethods\Core\BaseGatewaySettings;
use FluentCart\Framework\Support\Arr;
use FluentCart\Api\StoreSettings;

/**
 * WeChat Pay Settings Handler
 * 
 * Manages gateway configuration and credentials storage
 * Adapted from AlipaySettingsBase with WeChat Pay specific configurations
 */
class WechatSettingsBase extends BaseGatewaySettings
{
    /**
     * Meta key for settings storage in database
     * 
     * @var string
     */
    public $methodHandler = 'fluent_cart_payment_settings_wechat';

    /**
     * Settings array
     * 
     * @var array
     */
    public $settings;

    /**
     * Store settings instance
     * 
     * @var StoreSettings|null
     */
    public $storeSettings = null;

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();

        $settings = $this->getCachedSettings();
        $defaults = static::getDefaults();

        if (!$settings || !is_array($settings) || empty($settings)) {
            $settings = $defaults;
        } else {
            $settings = wp_parse_args($settings, $defaults);
        }

        $this->settings = $settings;

        if (!$this->storeSettings) {
            $this->storeSettings = new StoreSettings();
        }
    }

    /**
     * Get default settings
     * 
     * @return array
     */
    public static function getDefaults(): array
    {
        return [
            'is_active' => 'no',
            'gateway_description' => '',
            'appid' => '',
            'mch_id' => '',
            'api_key' => '',
            'app_secret' => '', // For JSAPI payment (optional)
            'sign_type' => 'no', // Checkbox: 'yes' for HMAC-SHA256, 'no' for MD5
            'notify_url_verification' => 'yes',
            'auto_refund_on_cancel' => 'no',
            'enable_jsapi' => 'no', // Enable JSAPI payment (WeChat browser)
            'ssl_cert_path' => '', // Path to SSL certificate (for refund)
            'ssl_key_path' => '', // Path to SSL key (for refund)
        ];
    }

    /**
     * Check if gateway is active
     * 
     * @return bool
     */
    public function isActive(): bool
    {
        $settings = $this->get();

        if ($settings['is_active'] !== 'yes') {
            return false;
        }

        // Check if credentials are available
        return !empty($this->getAppId()) && !empty($this->getMchId());
    }

    /**
     * Get setting value
     * 
     * @param string $key Setting key
     * @return mixed
     */
    public function get($key = '')
    {
        $settings = $this->settings;
        if ($key) {
            return $this->settings[$key] ?? null;
        }
        return $settings;
    }

    /**
     * Get payment mode
     * 
     * WeChat Pay does not provide a public sandbox environment,
     * so we always return 'live' mode.
     * 
     * @return string Always returns 'live'
     */
    public function getMode()
    {
        return 'live';
    }

    /**
     * Get WeChat App ID
     * 
     * @return string
     */
    public function getAppId()
    {
        return defined('WPKJ_WECHAT_APPID') 
            ? WPKJ_WECHAT_APPID 
            : ($this->get()['appid'] ?? '');
    }

    /**
     * Get Merchant ID
     * 
     * @return string
     */
    public function getMchId()
    {
        return defined('WPKJ_WECHAT_MCH_ID') 
            ? WPKJ_WECHAT_MCH_ID 
            : ($this->get()['mch_id'] ?? '');
    }

    /**
     * Get API Key
     * 
     * WeChat Pay API Key is used for signature generation
     * Unlike RSA keys in Alipay, this is a simple string key
     * 
     * @return string
     */
    public function getApiKey()
    {
        if (defined('WPKJ_WECHAT_API_KEY')) {
            return WPKJ_WECHAT_API_KEY;
        }
        
        $encryptedKey = $this->get()['api_key'] ?? '';
        if (empty($encryptedKey)) {
            throw new \Exception(esc_html__('API key is not configured', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
        }
        
        // Check if this is an encrypted value (FluentCart encrypted keys are very long base64)
        $isEncrypted = strlen($encryptedKey) > 200 && preg_match('/^[A-Za-z0-9+\/]+=*$/', $encryptedKey);
        
        if (!$isEncrypted) {
            // Key is stored in plain text - log warning in development only
            if (defined('WP_DEBUG') && WP_DEBUG) {
                \WPKJFluentCart\Wechat\Utils\Logger::warning('API Key stored unencrypted', [
                    'key_length' => strlen($encryptedKey)
                ]);
            }
            return $encryptedKey;
        }
        
        // Decrypt the key (stored encrypted in database)
        $decrypted = FluentCartHelper::decryptKey($encryptedKey);
        
        if ($decrypted === false || empty($decrypted)) {
            \WPKJFluentCart\Wechat\Utils\Logger::error('API Key Decryption Failed', [
                'encrypted_key_length' => strlen($encryptedKey),
                'is_encrypted_format' => $isEncrypted
            ]);
            throw new \Exception(esc_html__('Unable to decrypt API key. Please re-enter your credentials.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
        }
        
        return $decrypted;
    }

    /**
     * Get App Secret (for JSAPI payment)
     * 
     * App Secret is required for JSAPI payment to get user's openid
     * It's optional for other payment methods
     * 
     * @return string
     */
    public function getAppSecret()
    {
        if (defined('WPKJ_WECHAT_APP_SECRET')) {
            return WPKJ_WECHAT_APP_SECRET;
        }
        
        $encryptedSecret = $this->get()['app_secret'] ?? '';
        if (empty($encryptedSecret)) {
            return '';
        }
        
        // Check if this is an encrypted value
        $isEncrypted = strlen($encryptedSecret) > 200 && preg_match('/^[A-Za-z0-9+\/]+=*$/', $encryptedSecret);
        
        if (!$isEncrypted) {
            // Return plain text value
            return $encryptedSecret;
        }
        
        // Decrypt the secret
        $decrypted = FluentCartHelper::decryptKey($encryptedSecret);
        return $decrypted !== false ? $decrypted : '';
    }

    /**
     * Get sign type
     * 
     * @return string MD5 or HMAC-SHA256
     */
    public function getSignType()
    {
        // 'sign_type' field is now a checkbox:
        // - checked ('yes') => use HMAC-SHA256
        // - unchecked ('no' or empty) => use MD5 (default)
        $useHmacSha256 = $this->get('sign_type') === 'yes';
        return $useHmacSha256 ? 'HMAC-SHA256' : 'MD5';
    }

    /**
     * Get SSL certificate path (for refund operations)
     * 
     * @return string
     */
    public function getSslCertPath()
    {
        $certPath = $this->get('ssl_cert_path') ?? '';
        
        if (empty($certPath)) {
            return '';
        }

        // Normalize path (handles different OS path separators)
        $normalizedPath = wp_normalize_path($certPath);
        
        // SSL certificates must use absolute paths
        // Do not attempt to convert relative paths - return empty if not absolute
        if (!$this->isAbsolutePath($normalizedPath)) {
            return '';
        }

        return file_exists($normalizedPath) ? $normalizedPath : '';
    }

    /**
     * Get SSL key path (for refund operations)
     * 
     * @return string
     */
    public function getSslKeyPath()
    {
        $keyPath = $this->get('ssl_key_path') ?? '';
        
        if (empty($keyPath)) {
            return '';
        }

        // Normalize path (handles different OS path separators)
        $normalizedPath = wp_normalize_path($keyPath);
        
        // SSL key must use absolute paths
        // Do not attempt to convert relative paths - return empty if not absolute
        if (!$this->isAbsolutePath($normalizedPath)) {
            return '';
        }

        return file_exists($normalizedPath) ? $normalizedPath : '';
    }

    /**
     * Check if a path is absolute
     * 
     * @param string $path File path to check
     * @return bool True if path is absolute
     */
    private function isAbsolutePath($path)
    {
        // Unix-like absolute path (starts with /)
        if (strpos($path, '/') === 0) {
            return true;
        }
        
        // Windows absolute path (e.g., C:\, D:\)
        if (preg_match('/^[a-zA-Z]:\\\\/', $path)) {
            return true;
        }
        
        return false;
    }

    /**
     * Check if JSAPI payment is enabled
     * 
     * @return bool
     */
    public function isJsapiEnabled(): bool
    {
        return $this->get('enable_jsapi') === 'yes';
    }
}
