<?php

namespace WPKJFluentCart\Wechat\Gateway;

use FluentCart\Api\CurrencySettings;
use FluentCart\App\Helpers\CartCheckoutHelper;
use FluentCart\App\Helpers\Helper as FluentCartHelper;
use FluentCart\App\Modules\PaymentMethods\Core\AbstractPaymentGateway;
use FluentCart\App\Services\Payments\PaymentInstance;
use FluentCart\Framework\Support\Arr;
use WPKJFluentCart\Wechat\Config\WechatConfig;
use WPKJFluentCart\Wechat\Processor\PaymentProcessor;
use WPKJFluentCart\Wechat\Webhook\NotifyHandler;
use WPKJFluentCart\Wechat\Utils\Helper;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * WeChat Pay Payment Gateway
 * 
 * Main gateway class for WeChat Pay integration with FluentCart
 * Adapted from AlipayGateway with WeChat Pay specific implementations
 */
class WechatGateway extends AbstractPaymentGateway
{
    /**
     * Gateway slug
     * 
     * @var string
     */
    private $methodSlug = 'wechat';

    /**
     * Supported features
     * 
     * @var array
     */
    public array $supportedFeatures = ['payment', 'refund', 'webhook', 'subscriptions'];

    /**
     * Constructor
     */
    public function __construct()
    {
        $settings = new WechatSettingsBase();
        $subscriptions = new \WPKJFluentCart\Wechat\Subscription\WechatSubscriptions($settings);
        
        parent::__construct(
            $settings,
            $subscriptions
        );
    }

    /**
     * Gateway metadata
     * 
     * @return array
     */
    public function meta(): array
    {
        return [
            'title' => 'WeChat Pay',
            'route' => 'wechat',
            'slug' => 'wechat',
            'label' => 'WeChat Pay',
            'description' => esc_html__('Pay securely with WeChat Pay - Support PC QR code, Mobile H5, and WeChat browser payments', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'logo' => WPKJ_FC_WECHAT_URL . 'assets/images/wechat-logo.svg',
            'icon' => WPKJ_FC_WECHAT_URL . 'assets/images/wechat-icon.svg',
            'brand_color' => '#09BB07',
            'status' => $this->settings->get('is_active') === 'yes',
            'upcoming' => false,
            'supported_features' => $this->supportedFeatures
        ];
    }

    /**
     * Boot the gateway
     * 
     * This method is called during gateway registration.
     * We check for WeChat Pay return parameters and trigger the handler immediately.
     * 
     * @return void
     */
    public function boot()
    {
        // WeChat Pay typically uses async notification only
        // Return URL is optional and mainly for user navigation
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WeChat Pay return URL verification handled by signature
        $trxHash = isset($_GET['trx_hash']) ? sanitize_text_field(wp_unslash($_GET['trx_hash'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WeChat Pay return URL verification handled by signature
        $fctRedirect = isset($_GET['fct_redirect']) ? sanitize_text_field(wp_unslash($_GET['fct_redirect'])) : '';
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- WeChat Pay return URL verification handled by signature
        $outTradeNo = isset($_GET['out_trade_no']) ? sanitize_text_field(wp_unslash($_GET['out_trade_no'])) : '';
        
        if (!empty($trxHash) && 
            !empty($fctRedirect) && 
            $fctRedirect === 'yes' &&
            !empty($outTradeNo)) {
            
            $returnHandler = new \WPKJFluentCart\Wechat\Webhook\ReturnHandler();
            $returnHandler->handleReturn();
        }
    }

    /**
     * Make payment from payment instance
     * 
     * @param PaymentInstance $paymentInstance Payment instance
     * @return array Payment response
     */
    public function makePaymentFromPaymentInstance(PaymentInstance $paymentInstance)
    {
        try {
            // Process payment (supports both single and subscription orders)
            $processor = new PaymentProcessor($this->settings);
            return $processor->processSinglePayment($paymentInstance);

        } catch (\Exception $e) {
            Logger::error('Payment Processing Exception', $e->getMessage());
            
            return [
                'status' => 'failed',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Handle IPN/Webhook notification
     * 
     * @return void
     */
    public function handleIPN()
    {
        // Verify this is a POST request
        if (!isset($_SERVER['REQUEST_METHOD']) || sanitize_text_field(wp_unslash($_SERVER['REQUEST_METHOD'])) !== 'POST') {
            return;
        }

        $notifyHandler = new NotifyHandler($this->settings);
        $notifyHandler->processNotify();
    }

    /**
     * Get order information for checkout
     * 
     * @param array $data Request data
     * @return void
     */
    public function getOrderInfo(array $data)
    {
        $checkOutHelper = CartCheckoutHelper::make();
        $items = $checkOutHelper->getItems();

        // Validate currency support
        $this->checkCurrencySupport();

        wp_send_json([
            'status' => 'success',
            'message' => __('Ready to process payment', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
            'data' => [
                'gateway' => 'wechat',
                'currency' => CurrencySettings::get('currency')
            ]
        ], 200);
    }

    /**
     * Settings form fields
     * 
     * @return array
     */
    public function fields()
    {
        $notifyUrl = add_query_arg([
            'fct_payment_listener' => '1',
            'method' => 'wechat'
        ], site_url('/'));

        return [
            'notice' => [
                'value' => $this->renderStoreModeNotice(),
                'label' => __('WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'type' => 'notice'
            ],
            'gateway_description' => [
                'type' => 'text',
                'label' => __('Gateway Description', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => __('Pay securely with WeChat Pay - Support PC QR code, Mobile H5, and WeChat browser payments', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'help' => __('This description will be displayed on the checkout page when customers select WeChat Pay. Leave empty to use default.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'payment_credentials' => [
                'type' => 'html_attr',
                'label' => __('Payment Credentials', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'value' => '<div class="mb-4"><p class="text-sm text-gray-600">' . esc_html__('⚠️ WeChat Pay does not provide a public sandbox environment. Please use your production credentials and test with small amounts (e.g., ¥0.01).', 'wpkj-payment-gateway-for-fluentcart-with-wechat') . '</p></div>'
            ],
            'appid' => [
                'type' => 'text',
                'label' => __('App ID (AppID)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => 'wx1234567890abcdef',
                'required' => true,
                'help' => __('Your WeChat Pay App ID (starts with "wx")', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'mch_id' => [
                'type' => 'text',
                'label' => __('Merchant ID (Mch ID)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => '1234567890',
                'required' => true,
                'help' => __('Your WeChat Pay Merchant ID (10 digits)', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'api_key' => [
                'type' => 'password',
                'label' => __('API Key', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => '32-character API key',
                'required' => true,
                'help' => __('Your WeChat Pay API Key (32 characters, used for signature)', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'app_secret' => [
                'type' => 'password',
                'label' => __('App Secret (Optional)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => 'App Secret for JSAPI',
                'help' => __('Required for JSAPI payment (WeChat browser). Leave empty if not using JSAPI.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'notify_url_info' => [
                'type' => 'html_attr',
                'label' => __('Notify URL (Webhook)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'value' => sprintf(
                    '<div class="mt-3"><p class="mb-2">%s</p><code class="copyable-content">%s</code><p class="mt-2 text-sm text-gray-600">%s</p></div>',
                    esc_html__('Configure this URL in your WeChat Pay merchant settings:', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html($notifyUrl),
                    esc_html__('This URL will receive payment notifications from WeChat Pay.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                )
            ],
            'sign_type' => [
                'type' => 'checkbox',
                'label' => __('Use HMAC-SHA256 signature (recommended for enhanced security, leave unchecked for MD5)', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'auto_refund_on_cancel' => [
                'type' => 'checkbox',
                'label' => __('Enable automatic refund when order is cancelled (Requires SSL certificates)', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'enable_jsapi' => [
                'type' => 'checkbox',
                'label' => __('Enable JSAPI payment for WeChat browser (Requires App Secret configuration)', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'subscription_settings_header' => [
                'type' => 'html_attr',
                'label' => __('Subscription Settings', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'value' => '<hr class="my-4"><h3 class="text-lg font-semibold mb-2">' . esc_html__('Subscription Management', 'wpkj-payment-gateway-for-fluentcart-with-wechat') . '</h3><p class="text-sm text-gray-600 mb-3">' . esc_html__('Configure how subscription orders are handled when the parent order is canceled.', 'wpkj-payment-gateway-for-fluentcart-with-wechat') . '</p>'
            ],
            'auto_cancel_subscription_on_order_cancel' => [
                'type' => 'checkbox',
                'label' => __('Automatically cancel subscription when parent order is canceled (Initial order only)', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'help' => __('When enabled, canceling the initial subscription order will automatically cancel the associated subscription. This does NOT affect renewal orders - canceling a renewal order will never cancel the subscription.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'subscription_cancel_info' => [
                'type' => 'html_attr',
                'label' => __('How It Works', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'value' => sprintf(
                    '<div class="mt-2 p-4 bg-amber-50 border border-amber-200 rounded">'
                    . '<p class="text-sm mb-2"><strong>%s</strong></p>'
                    . '<ul class="text-sm list-disc list-inside space-y-1">'
                    . '<li>%s</li>'
                    . '<li>%s</li>'
                    . '<li>%s</li>'
                    . '<li>%s</li>'
                    . '</ul>'
                    . '<p class="text-sm mt-3 text-blue-600"><strong>%s</strong> %s</p>'
                    . '<p class="text-sm mt-2 text-gray-600"><strong>%s</strong> %s</p>'
                    . '</div>',
                    esc_html__('Automatic Subscription Cancellation Rules:', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('✅ WILL cancel: Initial subscription order (type=subscription) is canceled', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('❌ Will NOT cancel: Renewal order (type=renewal) is canceled', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('❌ Will NOT cancel: Subscription already in canceled status', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('❌ Will NOT cancel: Orders from other payment methods', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Note:', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('This feature helps maintain data consistency between orders and subscriptions. All operations are logged in the order activity log.', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Default:', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('This feature is DISABLED by default. Enable it only if you need automatic synchronization between order and subscription status.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                )
            ],
            'ssl_settings_header' => [
                'type' => 'html_attr',
                'label' => __('SSL Certificate Settings', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'value' => '<hr class="my-4"><h3 class="text-lg font-semibold mb-2">' . esc_html__('SSL Certificates (For Refund)', 'wpkj-payment-gateway-for-fluentcart-with-wechat') . '</h3><p class="text-sm text-gray-600 mb-3">' . esc_html__('WeChat Pay requires SSL certificates for refund operations. Download from WeChat Pay merchant platform.', 'wpkj-payment-gateway-for-fluentcart-with-wechat') . '</p>'
            ],
            'ssl_cert_path' => [
                'type' => 'text',
                'label' => __('SSL Certificate Path', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => '/path/to/apiclient_cert.pem',
                'help' => __('Absolute path to apiclient_cert.pem file. Example: /var/www/certs/apiclient_cert.pem', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'ssl_key_path' => [
                'type' => 'text',
                'label' => __('SSL Key Path', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'placeholder' => '/path/to/apiclient_key.pem',
                'help' => __('Absolute path to apiclient_key.pem file. Example: /var/www/certs/apiclient_key.pem', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ],
            'ssl_info' => [
                'type' => 'html_attr',
                'label' => __('Certificate Setup Guide', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                'value' => sprintf(
                    '<div class="mt-2 p-4 bg-blue-50 border border-blue-200 rounded">'
                    . '<p class="text-sm mb-2"><strong>%s</strong></p>'
                    . '<ol class="text-sm list-decimal list-inside space-y-1">'
                    . '<li>%s</li>'
                    . '<li>%s</li>'
                    . '<li>%s</li>'
                    . '<li>%s</li>'
                    . '</ol>'
                    . '<p class="text-sm mt-3 text-red-600"><strong>%s</strong> %s</p>'
                    . '</div>',
                    esc_html__('How to Setup SSL Certificates:', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Login to WeChat Pay merchant platform', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Navigate to Account Center → API Security → Download Certificates', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Upload apiclient_cert.pem and apiclient_key.pem to a secure directory on your server', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Enter the absolute file paths in the fields above', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Security Notice:', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html__('Store certificates outside the web root directory and set proper file permissions (600 or 400).', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                )
            ]
        ];
    }

    /**
     * Validate settings before save
     * 
     * @param array $data Settings data
     * @return array Validation result
     */
    public static function validateSettings($data): array
    {
        $appId = Arr::get($data, 'appid');
        $mchId = Arr::get($data, 'mch_id');
        $apiKey = Arr::get($data, 'api_key');

        // Check if App ID is provided
        if (empty($appId)) {
            return [
                'status' => 'failed',
                'message' => __('App ID is required!', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ];
        }

        // Validate App ID format (starts with "wx")
        if (!preg_match('/^wx[a-z0-9]{16}$/i', $appId)) {
            return [
                'status' => 'failed',
                'message' => __('Invalid App ID format. It should start with "wx" followed by 16 characters.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ];
        }

        // Check if Merchant ID is provided
        if (empty($mchId)) {
            return [
                'status' => 'failed',
                'message' => __('Merchant ID is required!', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ];
        }

        // Validate Merchant ID format (10 digits)
        if (!preg_match('/^\d{10}$/', $mchId)) {
            return [
                'status' => 'failed',
                'message' => __('Invalid Merchant ID format. It should be 10 digits.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
            ];
        }

        // Only validate API Key if it's being changed (not encrypted)
        if (!empty($apiKey)) {
            // Check if this is an encrypted value (FluentCart encrypted keys are very long base64)
            $isEncrypted = strlen($apiKey) > 200 && preg_match('/^[A-Za-z0-9+\/]+=*$/', $apiKey);
            
            // Only validate format if it's a new/updated key (not encrypted)
            if (!$isEncrypted) {
                // WeChat Pay API Key should be 32 characters
                if (strlen($apiKey) !== 32) {
                    return [
                        'status' => 'failed',
                        'message' => __('Invalid API Key format. It should be exactly 32 characters.', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                    ];
                }
            }
        }

        // Validate SSL certificate paths if provided
        $sslCertPath = Arr::get($data, 'ssl_cert_path');
        $sslKeyPath = Arr::get($data, 'ssl_key_path');

        if (!empty($sslCertPath)) {
            // Normalize path (handles different OS path separators)
            $normalizedPath = wp_normalize_path($sslCertPath);
            
            // SSL certificates must use absolute paths
            if (!self::isAbsolutePath($normalizedPath)) {
                return [
                    'status' => 'failed',
                    'message' => sprintf(
                        /* translators: %s: file path */
                        __('SSL certificate path must be absolute (e.g., /var/www/certs/cert.pem): %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $sslCertPath
                    )
                ];
            }
            
            if (!file_exists($normalizedPath)) {
                return [
                    'status' => 'failed',
                    'message' => sprintf(
                        /* translators: %s: file path */
                        __('SSL certificate file not found: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $sslCertPath
                    )
                ];
            }
        }

        if (!empty($sslKeyPath)) {
            // Normalize path (handles different OS path separators)
            $normalizedPath = wp_normalize_path($sslKeyPath);
            
            // SSL key must use absolute paths
            if (!self::isAbsolutePath($normalizedPath)) {
                return [
                    'status' => 'failed',
                    'message' => sprintf(
                        /* translators: %s: file path */
                        __('SSL key path must be absolute (e.g., /var/www/certs/key.pem): %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $sslKeyPath
                    )
                ];
            }
            
            if (!file_exists($normalizedPath)) {
                return [
                    'status' => 'failed',
                    'message' => sprintf(
                        /* translators: %s: file path */
                        __('SSL key file not found: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $sslKeyPath
                    )
                ];
            }
        }

        return [
            'status' => 'success',
            'message' => __('Settings validated successfully!', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
        ];
    }

    /**
     * Check if a path is absolute
     * 
     * @param string $path File path to check
     * @return bool True if path is absolute
     */
    private static function isAbsolutePath($path)
    {
        // Unix-like absolute path (starts with /)
        if (strpos($path, '/') === 0) {
            return true;
        }
            
        // Windows absolute path (e.g., C:\, D:\)
        if (preg_match('/^[a-zA-Z]:\\\\/', $path)) {
            return true;
        }
            
        return false;
    }

    /**
     * Check currency support
     * 
     * @return void
     * @throws \Exception If currency is not supported
     */
    private function checkCurrencySupport()
    {
        $currency = CurrencySettings::get('currency');
        $supportedCurrencies = WechatConfig::getSupportedCurrencies();

        if (!in_array($currency, $supportedCurrencies)) {
            throw new \Exception(
                sprintf(
                    /* translators: 1: current currency code, 2: comma-separated list of supported currencies */
                    esc_html__('WeChat Pay does not support %1$s. Supported currencies: %2$s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html($currency),
                    esc_html(implode(', ', $supportedCurrencies))
                )
            );
        }
    }

    /**
     * Enqueue frontend scripts
     * 
     * @param string $hasSubscription Has subscription flag
     * @return array
     */
    public function getEnqueueScriptSrc($hasSubscription = 'no'): array
    {
        return [
            [
                'handle' => 'wpkj-fc-wechat-checkout',
                'src' => WPKJ_FC_WECHAT_URL . 'assets/js/wechat-checkout.js',
                'deps' => ['jquery']
            ]
        ];
    }

    /**
     * Get localized data for scripts
     * 
     * @return array
     */
    public function getLocalizeData(): array
    {
        $customDescription = $this->settings->get('gateway_description');
        $description = !empty($customDescription) 
            ? $customDescription 
            : __('Pay securely with WeChat Pay - Support PC QR code, Mobile H5, and WeChat browser payments', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
        
        return [
            'wpkj_fc_wechat_data' => [
                'description' => $description
            ]
        ];
    }

    /**
     * Process refund through WeChat Pay API
     * 
     * This method is called by FluentCart's Refund service when processing refunds.
     * It only handles the WeChat Pay API call and returns the vendor refund ID.
     * FluentCart automatically handles transaction creation, order updates, and events.
     * 
     * @param \FluentCart\App\Models\OrderTransaction $transaction Original payment transaction
     * @param int $amount Refund amount in cents
     * @param array $args Additional arguments (reason, etc.)
     * @return string|\WP_Error WeChat Pay refund ID (refund_id) or WP_Error on failure
     */
    public function processRefund($transaction, $amount, $args)
    {
        try {
            // Get transaction identifiers
            $outTradeNo = $transaction->meta['out_trade_no'] ?? null;
            $transactionId = $transaction->meta['wechat_transaction_id'] ?? $transaction->vendor_charge_id;
            
            if (empty($outTradeNo) && empty($transactionId)) {
                return new \WP_Error(
                    'missing_transaction_id',
                    __('Missing transaction identifiers (out_trade_no and transaction_id)', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            // Initialize API client
            // Get payment mode from FluentCart global settings
            $paymentMode = fluent_cart_get_option('order_mode', 'live');
            
            // Build API config
            $config = [
                'appid' => $this->settings->getAppId(),
                'mch_id' => $this->settings->getMchId(),
                'api_key' => $this->settings->getApiKey(),
                'sign_type' => $this->settings->getSignType(),
                'ssl_cert_path' => $this->settings->getSslCertPath(),
                'ssl_key_path' => $this->settings->getSslKeyPath(),
                'mode' => $paymentMode,
            ];
            
            $api = new \WPKJFluentCart\Wechat\API\WechatAPI($config);

            // Generate unique refund number
            $outRefundNo = $transaction->uuid . '-' . time() . '-' . substr(md5(uniqid()), 0, 8);

            // Prepare refund parameters
            $refundParams = [
                'out_refund_no' => $outRefundNo,
                'total_fee' => (int) $transaction->total, // Original payment amount in cents
                'refund_fee' => (int) $amount, // Refund amount in cents
                'refund_desc' => $args['reason'] ?? 'Order refund'
            ];
            
            // Prefer transaction_id over out_trade_no
            if (!empty($transactionId)) {
                $refundParams['transaction_id'] = $transactionId;
            } else {
                $refundParams['out_trade_no'] = $outTradeNo;
            }

            Logger::info('Processing Refund via WeChat Pay API', [
                'transaction_uuid' => $transaction->uuid,
                'refund_fee' => $amount,
                'total_fee' => $transaction->total,
                'out_refund_no' => $outRefundNo
            ]);

            // Call WeChat Pay refund API
            $result = $api->refund($refundParams);

            if (is_wp_error($result)) {
                Logger::error('WeChat Pay Refund API Error', [
                    'transaction_uuid' => $transaction->uuid,
                    'error' => $result->get_error_message()
                ]);
                return $result;
            }

            // Check refund result
            if (!isset($result['return_code']) || $result['return_code'] !== 'SUCCESS' ||
                !isset($result['result_code']) || $result['result_code'] !== 'SUCCESS') {
                
                $errorMsg = $result['err_code_des'] ?? $result['return_msg'] ?? 'Unknown error';
                
                Logger::error('WeChat Pay Refund Failed', [
                    'transaction_uuid' => $transaction->uuid,
                    'error' => $errorMsg
                ]);
                
                return new \WP_Error(
                    'wechat_refund_failed',
                    sprintf(
                        /* translators: %s: WeChat Pay error message */
                        __('WeChat Pay refund failed: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $errorMsg
                    )
                );
            }

            // Return WeChat Pay refund_id as vendor_refund_id
            // FluentCart will:
            // - Create refund transaction record
            // - Store this refund_id in vendor_charge_id field
            // - Update order payment status and total_refund
            // - Trigger fluent_cart/order_refund_completed event
            // - Send refund notification email (if enabled)
            // - Restore inventory (if manageStock is enabled)
            $vendorRefundId = $result['refund_id'] ?? $outRefundNo;

            Logger::info('WeChat Pay Refund Successful', [
                'transaction_uuid' => $transaction->uuid,
                'refund_id' => $vendorRefundId,
                'refund_amount' => $amount
            ]);

            return $vendorRefundId;

        } catch (\Exception $e) {
            Logger::error('WeChat Pay Refund Exception', [
                'transaction_uuid' => $transaction->uuid ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return new \WP_Error(
                'wechat_refund_exception',
                sprintf(
                    /* translators: %s: exception error message */
                    __('WeChat Pay refund exception: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    $e->getMessage()
                )
            );
        }
    }
}
