<?php

namespace WPKJFluentCart\Wechat\Detector;

use WPKJFluentCart\Wechat\Config\WechatConfig;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * Client Detector
 * 
 * Detects user environment to choose appropriate WeChat Pay payment interface
 */
class ClientDetector
{
    /**
     * Detect if user is in WeChat browser
     * 
     * This is critical for choosing the correct payment method:
     * - WeChat browser -> JSAPI payment
     * - Mobile browser (non-WeChat) -> H5 payment
     * - PC browser -> Native payment (QR code)
     * 
     * @return bool
     */
    public static function isWechatBrowser(): bool
    {
        $userAgent = self::getUserAgent();
        
        if (empty($userAgent)) {
            return false;
        }
        
        // WeChat browser contains "MicroMessenger" in user agent
        // Example: Mozilla/5.0 (iPhone; CPU iPhone OS 14_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148 MicroMessenger/8.0.15(0x18000f2b) NetType/WIFI Language/zh_CN
        return stripos($userAgent, 'MicroMessenger') !== false;
    }

    /**
     * Detect if user is on mobile device
     * 
     * @return bool
     */
    public static function isMobile(): bool
    {
        $userAgent = self::getUserAgent();
        
        // Handle empty user agent
        if (empty($userAgent)) {
            return false;
        }
        
        $mobileAgents = [
            'Android', 'iPhone', 'iPad', 'iPod', 'BlackBerry', 
            'Windows Phone', 'Mobile', 'Symbian', 'Opera Mini',
            'Windows CE', 'Nokia', 'SonyEricsson', 'Samsung',
            'LG', 'HTC', 'Motorola', 'Kindle'
        ];
        
        foreach ($mobileAgents as $agent) {
            if (stripos($userAgent, $agent) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Detect client type
     * 
     * Returns simplified client type string for payment routing
     * 
     * @return string 'wechat'|'mobile'|'pc'
     */
    public static function detect(): string
    {
        if (self::isWechatBrowser()) {
            return 'wechat';
        }
        
        if (self::isMobile()) {
            return 'mobile';
        }
        
        return 'pc';
    }

    /**
     * Get appropriate trade type based on client environment
     * 
     * This maps client type to WeChat Pay's trade_type parameter:
     * - WeChat browser -> JSAPI (requires openid)
     * - Mobile browser -> MWEB (H5 payment)
     * - PC browser -> NATIVE (QR code scanning)
     * 
     * @param array|null $settings Settings array (optional, for future customization)
     * @return string JSAPI|MWEB|NATIVE
     */
    public static function getTradeType($settings = null): string
    {
        if (self::isWechatBrowser()) {
            // JSAPI payment - requires user's openid
            // This is used when user is browsing in WeChat app
            return WechatConfig::TRADE_TYPE_JSAPI;
        }
        
        if (self::isMobile()) {
            // MWEB (H5) payment - for mobile browsers outside WeChat
            // User will be redirected to WeChat Pay H5 page
            return WechatConfig::TRADE_TYPE_MWEB;
        }
        
        // NATIVE payment - generates QR code for PC users
        // User scans the QR code with WeChat app to complete payment
        return WechatConfig::TRADE_TYPE_NATIVE;
    }

    /**
     * Get user agent string
     * 
     * @return string
     */
    private static function getUserAgent(): string
    {
        // Check if HTTP_USER_AGENT exists (may not in CLI or proxied environments)
        if (!isset($_SERVER['HTTP_USER_AGENT'])) {
            return '';
        }
        
        // Sanitize user agent for safe use
        return sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT']));
    }

    /**
     * Log client detection result for debugging
     * 
     * This is useful for troubleshooting payment issues
     * 
     * @return void
     */
    public static function logDetection(): void
    {
        $userAgent = self::getUserAgent();
        $clientType = self::detect();
        $tradeType = self::getTradeType();
        
        Logger::info('Client Detection', [
            'user_agent' => $userAgent,
            'client_type' => $clientType,
            'trade_type' => $tradeType,
            'is_wechat_browser' => self::isWechatBrowser() ? 'yes' : 'no',
            'is_mobile' => self::isMobile() ? 'yes' : 'no',
            // Sanitize IP address before logging
            'ip' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : 'unknown'
        ]);
    }
}
