<?php
/**
 * Wechat Pay Configuration Constants
 * 
 * Centralized configuration management for all Wechat Pay related constants.
 * This eliminates magic numbers and improves code maintainability.
 * 
 * All limits and constraints are based on Wechat Pay official documentation:
 * @link https://pay.weixin.qq.com/wiki/doc/api/index.html
 * 
 * @package WPKJFluentCart\Wechat\Config
 * @since 1.0.0
 */

namespace WPKJFluentCart\Wechat\Config;

class WechatConfig
{
    /**
     * Wechat Pay API limits and constraints
     */
    
    /**
     * Minimum payment amount (in cents)
     * 
     * Wechat Pay requires minimum 0.01 CNY (1 cent) for payment.
     * This prevents invalid zero-amount payments.
     * 
     * @var int
     */
    const MIN_PAYMENT_AMOUNT_CENTS = 1;
    
    /**
     * Maximum single transaction amount limit (CNY)
     * 
     * Wechat Pay enforces this limit on all payment requests.
     * 
     * @var int
     */
    const MAX_SINGLE_TRANSACTION_AMOUNT = 500000;
    
    /**
     * Maximum length for payment description
     * 
     * Used in unified order body parameter.
     * 
     * @var int
     */
    const MAX_BODY_LENGTH = 128;
    
    /**
     * Maximum length for merchant order number (out_trade_no)
     * 
     * Wechat Pay requires out_trade_no to be unique and within this length.
     * 
     * @var int
     */
    const MAX_OUT_TRADE_NO_LENGTH = 32;
    
    /**
     * Timeout settings
     */
    
    /**
     * Minimum payment timeout in minutes (required by WeChat Pay)
     * 
     * WeChat Pay requires:
     * - MICROPAY (刷卡支付): >= 1 minute
     * - NATIVE/JSAPI/MWEB: >= 5 minutes
     * 
     * @var int
     */
    const MIN_PAYMENT_TIMEOUT_MINUTES = 5;
    
    /**
     * Payment timeout in minutes
     * 
     * After this time, unpaid orders will be automatically cancelled by Wechat Pay.
     * Must be >= MIN_PAYMENT_TIMEOUT_MINUTES (5 minutes)
     * 
     * @var int
     */
    const PAYMENT_TIMEOUT_MINUTES = 30;
    
    /**
     * Default payment timeout string
     * 
     * Format used in Wechat Pay API time_expire parameter.
     * Format: yyyyMMddHHmmss
     * 
     * @var string
     */
    const DEFAULT_PAYMENT_TIMEOUT = '30m';
    
    /**
     * Cache TTL (Time To Live)
     */
    
    /**
     * Notify deduplication cache TTL
     * 
     * Prevents duplicate processing of Wechat Pay notify callbacks.
     * Uses WordPress transient API.
     * 
     * @var int Seconds (86400 = 24 hours)
     */
    const NOTIFY_DEDUP_TTL = DAY_IN_SECONDS;
    
    /**
     * Query result cache TTL
     * 
     * Caches Wechat Pay trade query results to reduce API calls.
     * Short TTL ensures near-real-time status updates.
     * 
     * @var int Seconds
     */
    const QUERY_CACHE_TTL = 5;
    
    /**
     * Polling settings for Native payment
     */
    
    /**
     * Status check interval in seconds
     * 
     * How often to poll Wechat Pay for payment status on Native payment page.
     * 
     * @var int Seconds
     */
    const STATUS_CHECK_INTERVAL = 3;
    
    /**
     * Maximum status check attempts
     * 
     * Maximum number of status checks before timing out.
     * (200 attempts × 3 seconds = 600 seconds = 10 minutes)
     * 
     * @var int
     */
    const STATUS_CHECK_MAX_ATTEMPTS = 200;
    
    /**
     * Gateway URLs
     */
    const API_URL_PROD = 'https://api.mch.weixin.qq.com';
    
    /**
     * Trade types
     */
    const TRADE_TYPE_NATIVE = 'NATIVE';    // PC QR code payment
    const TRADE_TYPE_JSAPI = 'JSAPI';      // Wechat browser payment
    const TRADE_TYPE_MWEB = 'MWEB';        // Mobile H5 payment
    const TRADE_TYPE_APP = 'APP';          // Native app payment
    
    /**
     * Sign types
     */
    const SIGN_TYPE_MD5 = 'MD5';
    const SIGN_TYPE_HMAC_SHA256 = 'HMAC-SHA256';
    
    /**
     * Supported currencies
     * 
     * @return array
     */
    public static function getSupportedCurrencies(): array
    {
        return [
            'CNY',  // Chinese Yuan (primary)
            'USD',  // US Dollar
            'EUR',  // Euro
            'GBP',  // British Pound
            'HKD',  // Hong Kong Dollar
            'JPY',  // Japanese Yen
        ];
    }
    
    /**
     * Get API URL based on mode
     * 
     * @param string $mode Payment mode (test/live)
     * @return string API URL
     */
    public static function getApiUrl(string $mode): string
    {
        // Use production URL for both test and live modes, because Wechat Pay does not support sandbox mode at the moment
        return self::API_URL_PROD;
    }
    
    /**
     * Get payment timeout with buffer
     * 
     * @return int Total timeout in minutes
     */
    public static function getPaymentTimeoutWithBuffer(): int
    {
        return self::PAYMENT_TIMEOUT_MINUTES + 5; // Add 5 minutes buffer
    }
}
