<?php

namespace WPKJFluentCart\Wechat\API;

use WPKJFluentCart\Wechat\Config\WechatConfig;
use WPKJFluentCart\Wechat\Utils\Helper;
use WPKJFluentCart\Wechat\Utils\Logger;

/**
 * WeChat Pay API Handler
 * 
 * Handles all communication with WeChat Pay payment gateway
 * Adapted from AlipayAPI with key differences:
 * - XML format instead of JSON
 * - MD5/HMAC-SHA256 signature instead of RSA
 * - Different API endpoints and parameters
 */
class WechatAPI
{
    /**
     * Gateway configuration
     * 
     * @var array
     */
    private $config;

    /**
     * WeChat Pay gateway URLs
     */
    const API_URL_PROD = WechatConfig::API_URL_PROD;

    /**
     * Constructor
     * 
     * @param array $config Gateway configuration
     *   Required keys:
     *   - appid: WeChat Pay App ID
     *   - mch_id: Merchant ID
     *   - api_key: API Key for signature
     *   - sign_type: Signature type (MD5 or HMAC-SHA256)
     *   - mode: Environment mode (prod or test)
     */
    public function __construct(array $config)
    {
        $this->config = $config;
    }

    /**
     * Create Native payment (QR code for PC)
     * 
     * @param array $orderData Order data
     * @return array|\WP_Error QR code data or error
     */
    public function createNativePayment(array $orderData)
    {
        try {
            $params = [
                'appid' => $this->config['appid'],
                'mch_id' => $this->config['mch_id'],
                'nonce_str' => Helper::generateNonceStr(),
                'body' => $orderData['body'],
                'out_trade_no' => $orderData['out_trade_no'],
                'total_fee' => $orderData['total_fee'], // Amount in cents
                'spbill_create_ip' => $this->getClientIp(),
                'notify_url' => $orderData['notify_url'],
                'trade_type' => WechatConfig::TRADE_TYPE_NATIVE,
            ];

            if (!empty($orderData['detail'])) {
                $params['detail'] = $orderData['detail'];
            }

            if (!empty($orderData['time_expire'])) {
                $params['time_expire'] = $orderData['time_expire'];
            }

            // Generate signature
            $params['sign'] = $this->generateSign($params);

            // Convert to XML
            $xml = $this->arrayToXml($params);

            // Send request
            $apiUrl = $this->getApiUrl('/pay/unifiedorder');
            $response = wp_remote_post($apiUrl, [
                'body' => $xml,
                'headers' => [
                    'Content-Type' => 'application/xml; charset=UTF-8'
                ],
                'timeout' => 30,
            ]);

            if (is_wp_error($response)) {
                Logger::error('WP_Error in Native Payment Request', [
                    'error_code' => $response->get_error_code(),
                    'error_message' => $response->get_error_message()
                ]);
                return $response;
            }

            $httpCode = wp_remote_retrieve_response_code($response);
            if ($httpCode !== 200) {
                $responseBody = wp_remote_retrieve_body($response);
                Logger::error('HTTP Request Failed', [
                    'http_code' => $httpCode,
                    'api_url' => $apiUrl,
                    'response_body' => substr($responseBody, 0, 500)
                ]);
                return new \WP_Error(
                    'wechat_http_error',
                    sprintf(
                        /* translators: %d: HTTP status code */
                        __('HTTP %d error from WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $httpCode
                    )
                );
            }

            $body = wp_remote_retrieve_body($response);
            
            if (empty($body)) {
                return new \WP_Error('wechat_native_error', __('Empty response from WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
            }

            // Parse XML response
            $result = $this->xmlToArray($body);

            if (!is_array($result)) {
                Logger::error('Invalid XML Response', [
                    'body_preview' => substr($body, 0, 500)
                ]);
                return new \WP_Error(
                    'wechat_native_error',
                    __('Invalid XML response from WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            // Check return_code
            if (!isset($result['return_code']) || $result['return_code'] !== 'SUCCESS') {
                $errorMsg = $result['return_msg'] ?? __('Native payment creation failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
                Logger::error('Native Payment Failed (return_code)', [
                    'out_trade_no' => $orderData['out_trade_no'],
                    'return_code' => $result['return_code'] ?? 'unknown',
                    'return_msg' => $errorMsg
                ]);
                return new \WP_Error('wechat_native_error', $errorMsg);
            }

            // Check result_code
            if (!isset($result['result_code']) || $result['result_code'] !== 'SUCCESS') {
                $errorMsg = $result['err_code_des'] ?? $result['err_code'] ?? __('Native payment creation failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
                Logger::error('Native Payment Failed (result_code)', [
                    'out_trade_no' => $orderData['out_trade_no'],
                    'result_code' => $result['result_code'] ?? 'unknown',
                    'err_code' => $result['err_code'] ?? '',
                    'err_code_des' => $result['err_code_des'] ?? ''
                ]);
                return new \WP_Error('wechat_native_error', $errorMsg);
            }

            // Verify signature
            if (!$this->verifySign($result)) {
                Logger::error('Signature Verification Failed', [
                    'out_trade_no' => $orderData['out_trade_no']
                ]);
                return new \WP_Error(
                    'wechat_native_error',
                    __('Signature verification failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            if (empty($result['code_url'])) {
                Logger::error('Missing code_url in Response', [
                    'out_trade_no' => $orderData['out_trade_no']
                ]);
                return new \WP_Error(
                    'wechat_native_error',
                    __('QR code URL not found in response', 'wpkj-payment-gateway-for-fluentcart-with-wechat')
                );
            }

            return [
                'code_url' => $result['code_url'],
                'prepay_id' => $result['prepay_id'] ?? '',
                'out_trade_no' => $orderData['out_trade_no'],
                'trade_type' => WechatConfig::TRADE_TYPE_NATIVE
            ];

        } catch (\Exception $e) {
            Logger::error('Create Native Payment Error', $e->getMessage());
            return new \WP_Error('wechat_native_error', $e->getMessage());
        }
    }

    /**
     * Create H5 payment (for mobile browsers)
     * 
     * @param array $orderData Order data
     * @return array|\WP_Error Payment data or error
     */
    public function createH5Payment(array $orderData)
    {
        try {
            $params = [
                'appid' => $this->config['appid'],
                'mch_id' => $this->config['mch_id'],
                'nonce_str' => Helper::generateNonceStr(),
                'body' => $orderData['body'],
                'out_trade_no' => $orderData['out_trade_no'],
                'total_fee' => $orderData['total_fee'],
                'spbill_create_ip' => $this->getClientIp(),
                'notify_url' => $orderData['notify_url'],
                'trade_type' => WechatConfig::TRADE_TYPE_MWEB,
            ];

            // H5 payment requires scene_info
            if (isset($orderData['scene_info'])) {
                $params['scene_info'] = json_encode($orderData['scene_info'], JSON_UNESCAPED_UNICODE);
            }

            // Generate signature
            $params['sign'] = $this->generateSign($params);

            // Convert to XML
            $xml = $this->arrayToXml($params);

            // Send request
            $apiUrl = $this->getApiUrl('/pay/unifiedorder');
            $response = wp_remote_post($apiUrl, [
                'body' => $xml,
                'headers' => [
                    'Content-Type' => 'application/xml; charset=UTF-8'
                ],
                'timeout' => 30,
            ]);

            if (is_wp_error($response)) {
                Logger::error('WP_Error in H5 Payment Request', [
                    'error_code' => $response->get_error_code(),
                    'error_message' => $response->get_error_message()
                ]);
                return $response;
            }

            $httpCode = wp_remote_retrieve_response_code($response);
            if ($httpCode !== 200) {
                return new \WP_Error(
                    'wechat_http_error',
                    sprintf(
                        /* translators: %d: HTTP status code */
                        __('HTTP %d error from WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $httpCode
                    )
                );
            }

            $body = wp_remote_retrieve_body($response);
            $result = $this->xmlToArray($body);

            if (!is_array($result)) {
                return new \WP_Error('wechat_h5_error', __('Invalid XML response', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
            }

            // Check response
            if ($result['return_code'] !== 'SUCCESS' || $result['result_code'] !== 'SUCCESS') {
                $errorMsg = $result['err_code_des'] ?? $result['return_msg'] ?? __('H5 payment creation failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
                return new \WP_Error('wechat_h5_error', $errorMsg);
            }

            // Verify signature
            if (!$this->verifySign($result)) {
                return new \WP_Error('wechat_h5_error', __('Signature verification failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
            }

            Logger::info('H5 Payment Created', [
                'out_trade_no' => $orderData['out_trade_no'],
                'mweb_url' => $result['mweb_url'] ?? ''
            ]);

            return [
                'mweb_url' => $result['mweb_url'] ?? '',
                'prepay_id' => $result['prepay_id'] ?? '',
                'out_trade_no' => $orderData['out_trade_no'],
                'trade_type' => WechatConfig::TRADE_TYPE_MWEB
            ];

        } catch (\Exception $e) {
            Logger::error('Create H5 Payment Error', $e->getMessage());
            return new \WP_Error('wechat_h5_error', $e->getMessage());
        }
    }

    /**
     * Query payment status
     * 
     * @param string $outTradeNo Order number
     * @return array|\WP_Error Payment status or error
     */
    public function queryPayment(string $outTradeNo)
    {
        try {
            // Check cache first
            $cacheKey = 'wechat_query_' . md5($outTradeNo);
            $cached = get_transient($cacheKey);
            
            if ($cached !== false) {
                return $cached;
            }

            $params = [
                'appid' => $this->config['appid'],
                'mch_id' => $this->config['mch_id'],
                'out_trade_no' => $outTradeNo,
                'nonce_str' => Helper::generateNonceStr(),
            ];

            // Generate signature
            $params['sign'] = $this->generateSign($params);

            // Convert to XML
            $xml = $this->arrayToXml($params);

            // Send request
            $apiUrl = $this->getApiUrl('/pay/orderquery');
            $response = wp_remote_post($apiUrl, [
                'body' => $xml,
                'headers' => [
                    'Content-Type' => 'application/xml; charset=UTF-8'
                ],
                'timeout' => 15,
            ]);

            if (is_wp_error($response)) {
                return $response;
            }

            $httpCode = wp_remote_retrieve_response_code($response);
            if ($httpCode !== 200) {
                return new \WP_Error(
                    'wechat_http_error',
                    sprintf(
                        /* translators: %d: HTTP status code */
                        __('HTTP %d error from WeChat Pay', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                        $httpCode
                    )
                );
            }

            $body = wp_remote_retrieve_body($response);
            $result = $this->xmlToArray($body);

            if (!is_array($result)) {
                return new \WP_Error('wechat_query_error', __('Invalid XML response', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
            }

            // Check return_code
            if ($result['return_code'] !== 'SUCCESS') {
                $errorMsg = $result['return_msg'] ?? __('Query failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
                return new \WP_Error('wechat_query_error', $errorMsg);
            }

            // Trade not found - payment still pending
            if ($result['result_code'] !== 'SUCCESS' && isset($result['err_code']) && $result['err_code'] === 'ORDERNOTEXIST') {
                $pendingResult = [
                    'trade_state' => 'NOTPAY',
                    'trade_state_desc' => 'Order not found, payment may still be pending'
                ];
                // Cache pending status for shorter time
                set_transient($cacheKey, $pendingResult, 2);
                return $pendingResult;
            }

            // Verify signature
            if (!$this->verifySign($result)) {
                return new \WP_Error('wechat_query_error', __('Signature verification failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
            }

            // Cache successful result
            if (isset($result['trade_state'])) {
                set_transient($cacheKey, $result, WechatConfig::QUERY_CACHE_TTL);
            }

            return $result;

        } catch (\Exception $e) {
            Logger::error('Query Payment Error', $e->getMessage());
            return new \WP_Error('wechat_query_error', $e->getMessage());
        }
    }

    /**
     * Process refund
     * 
     * @param array $refundData Refund data
     * @return array|\WP_Error Refund result or error
     */
    public function refund(array $refundData)
    {
        try {
            $params = [
                'appid' => $this->config['appid'],
                'mch_id' => $this->config['mch_id'],
                'nonce_str' => Helper::generateNonceStr(),
                'out_refund_no' => $refundData['out_refund_no'],
                'total_fee' => $refundData['total_fee'],
                'refund_fee' => $refundData['refund_fee'],
                'op_user_id' => $this->config['mch_id'], // Operator ID, default to merchant ID
            ];

            // Prefer transaction_id over out_trade_no
            if (!empty($refundData['transaction_id'])) {
                $params['transaction_id'] = $refundData['transaction_id'];
            } elseif (!empty($refundData['out_trade_no'])) {
                $params['out_trade_no'] = $refundData['out_trade_no'];
            } else {
                throw new \Exception('Refund requires either transaction_id or out_trade_no');
            }

            if (!empty($refundData['refund_desc'])) {
                $params['refund_desc'] = $refundData['refund_desc'];
            }

            // Generate signature
            $params['sign'] = $this->generateSign($params);

            // Convert to XML
            $xml = $this->arrayToXml($params);
            
            Logger::info('Refund Request Details', [
                'params' => array_keys($params),
                'total_fee' => $params['total_fee'],
                'refund_fee' => $params['refund_fee'],
                'identifier' => $params['transaction_id'] ?? $params['out_trade_no'] ?? 'N/A',
                'has_ssl_cert' => !empty($this->config['ssl_cert_path']),
                'has_ssl_key' => !empty($this->config['ssl_key_path'])
            ]);

            // Refund API requires SSL certificate, must use cURL
            $apiUrl = $this->getApiUrl('/secapi/pay/refund');
            $responseBody = $this->sendCurlRequest($apiUrl, $xml);
            
            $result = $this->xmlToArray($responseBody);

            if (!is_array($result)) {
                Logger::error('Invalid Refund Response', [
                    'response_preview' => substr($responseBody, 0, 500)
                ]);
                return new \WP_Error('wechat_refund_error', __('Invalid XML response', 'wpkj-payment-gateway-for-fluentcart-with-wechat'));
            }

            // Check response
            if ($result['return_code'] !== 'SUCCESS' || $result['result_code'] !== 'SUCCESS') {
                $errorMsg = $result['err_code_des'] ?? $result['return_msg'] ?? __('Refund failed', 'wpkj-payment-gateway-for-fluentcart-with-wechat');
                Logger::error('Refund Failed', [
                    'identifier' => $refundData['transaction_id'] ?? $refundData['out_trade_no'] ?? 'unknown',
                    'return_code' => $result['return_code'] ?? 'N/A',
                    'result_code' => $result['result_code'] ?? 'N/A',
                    'err_code' => $result['err_code'] ?? 'N/A',
                    'error' => $errorMsg
                ]);
                return new \WP_Error('wechat_refund_error', $errorMsg);
            }

            Logger::info('Refund Processed Successfully', [
                'identifier' => $refundData['transaction_id'] ?? $refundData['out_trade_no'] ?? 'unknown',
                'refund_fee' => $refundData['refund_fee'],
                'refund_id' => $result['refund_id'] ?? 'N/A'
            ]);

            return $result;

        } catch (\Exception $e) {
            Logger::error('Refund Exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return new \WP_Error('wechat_refund_error', $e->getMessage());
        }
    }

    /**
     * Generate signature
     * 
     * @param array $params Parameters to sign
     * @return string Signature
     */
    private function generateSign(array $params): string
    {
        // Remove sign if exists
        unset($params['sign']);

        // Remove empty values and convert arrays to JSON strings
        $params = array_filter($params, function($value) {
            return $value !== '' && $value !== null;
        });

        // Convert array values to strings (for scene_info, etc.)
        foreach ($params as $key => $value) {
            if (is_array($value)) {
                $params[$key] = json_encode($value, JSON_UNESCAPED_UNICODE);
            }
        }

        // Sort parameters by key
        ksort($params);

        // Build sign string
        $signString = '';
        foreach ($params as $key => $value) {
            $signString .= $key . '=' . $value . '&';
        }

        // Append API key
        $signString .= 'key=' . $this->config['api_key'];

        // Generate signature based on sign type
        $signType = $this->config['sign_type'] ?? WechatConfig::SIGN_TYPE_MD5;

        if ($signType === WechatConfig::SIGN_TYPE_HMAC_SHA256) {
            $signature = strtoupper(hash_hmac('sha256', $signString, $this->config['api_key']));
        } else {
            // MD5 (default)
            $signature = strtoupper(md5($signString));
        }

        return $signature;
    }

    /**
     * Verify signature from WeChat Pay notification
     * 
     * @param array $data Notification data
     * @return bool Verification result
     */
    public function verifySign(array $data): bool
    {
        if (!isset($data['sign'])) {
            Logger::error('Signature Verification Failed', 'Missing sign parameter');
            return false;
        }

        $receivedSign = $data['sign'];
        
        // Remove sign field for verification
        $dataForSign = $data;
        unset($dataForSign['sign']);
        
        // Remove empty values
        $dataForSign = array_filter($dataForSign, function($value) {
            return $value !== '' && $value !== null;
        });
        
        // Sort parameters by key
        ksort($dataForSign);
        
        // Build sign string
        $signString = '';
        foreach ($dataForSign as $key => $value) {
            // Skip array values (shouldn't happen in WeChat responses, but just in case)
            if (is_array($value)) {
                continue;
            }
            $signString .= $key . '=' . $value . '&';
        }
        
        // Append API key
        $signString .= 'key=' . $this->config['api_key'];
        
        // Generate signature based on sign type
        $signType = $data['sign_type'] ?? $this->config['sign_type'] ?? WechatConfig::SIGN_TYPE_MD5;
        
        if ($signType === WechatConfig::SIGN_TYPE_HMAC_SHA256) {
            $calculatedSign = strtoupper(hash_hmac('sha256', $signString, $this->config['api_key']));
        } else {
            // MD5 (default)
            $calculatedSign = strtoupper(md5($signString));
        }
        
        $result = ($receivedSign === $calculatedSign);

        if (!$result) {
            Logger::error('Signature Verification Failed', [
                'expected' => $calculatedSign,
                'received' => $receivedSign,
                'sign_type' => $signType,
                'sign_string_preview' => substr($signString, 0, 200)
            ]);
        }

        return $result;
    }

    /**
     * Convert array to XML
     * 
     * @param array $data Data array
     * @return string XML string
     */
    private function arrayToXml(array $data): string
    {
        $xml = '<xml>';
        foreach ($data as $key => $value) {
            if (is_numeric($value)) {
                $xml .= '<' . $key . '>' . $value . '</' . $key . '>';
            } else {
                $xml .= '<' . $key . '><![CDATA[' . $value . ']]></' . $key . '>';
            }
        }
        $xml .= '</xml>';

        return $xml;
    }

    /**
     * Convert XML to array
     * 
     * @param string $xml XML string
     * @return array|false Array data or false on error
     */
    private function xmlToArray(string $xml)
    {
        try {
            // PHP 8.0+ has XXE protection enabled by default
            // No need to use deprecated libxml_disable_entity_loader()
            
            // Load XML with security flags
            $xmlObj = simplexml_load_string(
                $xml, 
                'SimpleXMLElement', 
                LIBXML_NOCDATA | LIBXML_NONET | LIBXML_NOENT
            );

            if ($xmlObj === false) {
                Logger::error('XML Parse Error', [
                    'xml_preview' => substr($xml, 0, 200)
                ]);
                return false;
            }

            $json = json_encode($xmlObj);
            $array = json_decode($json, true);

            return $array;

        } catch (\Exception $e) {
            Logger::error('XML Conversion Error', $e->getMessage());
            return false;
        }
    }

    /**
     * Get API URL for specific endpoint
     * 
     * @param string $endpoint API endpoint
     * @return string Full API URL
     */
    private function getApiUrl(string $endpoint): string
    {
        $baseUrl = self::API_URL_PROD;
        return $baseUrl . $endpoint;
    }

    /**
     * Send HTTP request for refund (requires client certificate)
     * Uses WordPress HTTP API with client certificate support
     * 
     * @param string $url Request URL
     * @param string $xml XML data
     * @return string Response
     * @throws \Exception On request error
     */
    private function sendCurlRequest(string $url, string $xml): string
    {
        // Check if certificate files exist
        $certPath = $this->config['ssl_cert_path'] ?? '';
        $keyPath = $this->config['ssl_key_path'] ?? '';
        
        if (empty($certPath) || empty($keyPath)) {
            throw new \Exception('SSL certificate and key paths are required for refund API calls.');
        }
        
        if (!file_exists($certPath)) {
            throw new \Exception(
                sprintf(
                    /* translators: %s: file path */
                    esc_html__('SSL certificate file not found: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html($certPath)
                )
            );
        }
        
        if (!file_exists($keyPath)) {
            throw new \Exception(
                sprintf(
                    /* translators: %s: SSL private key file path */
                    esc_html__('SSL private key file not found: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html($keyPath)
                )
            );
        }

        // Determine if sandbox mode
        $isSandbox = ($this->config['mode'] ?? 'live') === 'test';
        
        // Store cert paths for the http_api_curl filter
        $sslConfig = [
            'cert_path' => $certPath,
            'key_path' => $keyPath,
            'verify_peer' => !$isSandbox,
            'verify_host' => !$isSandbox ? 2 : 0,
        ];
        
        // Add filter to configure SSL client certificate
        // phpcs:disable WordPress.WP.AlternativeFunctions.curl_curl_setopt -- Using http_api_curl hook is the WordPress-recommended way to configure client certificates
        $curlFilter = function($handle) use ($sslConfig) {
            curl_setopt($handle, CURLOPT_SSLCERT, $sslConfig['cert_path']);
            curl_setopt($handle, CURLOPT_SSLKEY, $sslConfig['key_path']);
            curl_setopt($handle, CURLOPT_SSLCERTTYPE, 'PEM');
            curl_setopt($handle, CURLOPT_SSLKEYTYPE, 'PEM');
            curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, $sslConfig['verify_peer']);
            curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, $sslConfig['verify_host']);
        };
        // phpcs:enable WordPress.WP.AlternativeFunctions.curl_curl_setopt
        
        add_action('http_api_curl', $curlFilter);

        Logger::info('WordPress HTTP API Refund Request', [
            'url' => $url,
            'cert_path' => $certPath,
            'key_path' => $keyPath,
            'sandbox_mode' => $isSandbox,
            'xml_length' => strlen($xml)
        ]);

        // Send request using WordPress HTTP API
        $response = wp_remote_post($url, [
            'body' => $xml,
            'headers' => [
                'Content-Type' => 'application/xml; charset=utf-8',
            ],
            'timeout' => 30,
            'sslverify' => !$isSandbox,
        ]);
        
        // Remove filter after request
        remove_action('http_api_curl', $curlFilter);

        // Check for WP_Error
        if (is_wp_error($response)) {
            Logger::error('WordPress HTTP API Error', [
                'error_code' => $response->get_error_code(),
                'error_message' => $response->get_error_message(),
            ]);
            throw new \Exception(
                sprintf(
                    /* translators: 1: error code, 2: error message */
                    esc_html__('HTTP request error (%1$s): %2$s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html($response->get_error_code()),
                    esc_html($response->get_error_message())
                )
            );
        }
        
        $httpCode = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        Logger::info('WordPress HTTP API Refund Response', [
            'http_code' => $httpCode,
            'response_length' => strlen($body),
            'response_preview' => substr($body, 0, 300)
        ]);

        // Check HTTP status code
        if ($httpCode !== 200) {
            Logger::error('HTTP Error Response', [
                'http_code' => $httpCode,
                'response' => substr($body, 0, 500)
            ]);
            throw new \Exception(
                sprintf(
                    /* translators: %s: HTTP status code */
                    esc_html__('HTTP error: %s', 'wpkj-payment-gateway-for-fluentcart-with-wechat'),
                    esc_html((string)$httpCode)
                )
            );
        }

        return $body;
    }

    /**
     * Get client IP address
     * 
     * @return string Client IP
     */
    private function getClientIp(): string
    {
        $ip = '';

        if (isset($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR']));
        } elseif (isset($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['HTTP_CLIENT_IP']));
        } elseif (isset($_SERVER['REMOTE_ADDR'])) {
            $ip = sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR']));
        }

        // Get first IP if multiple IPs are present
        if (strpos($ip, ',') !== false) {
            $ips = explode(',', $ip);
            $ip = trim($ips[0]);
        }

        // Validate IP
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            $ip = '127.0.0.1';
        }

        return $ip;
    }
}
