/**
 * Native Payment Page Script
 * Handles QR code display and payment status polling
 * Matches Alipay Face-to-Face payment script structure
 */

(function($) {
    'use strict';

    // Ensure wpkj_wechat_native_data is available
    if (typeof wpkj_wechat_native_data === 'undefined') {
        console.error('WeChat Native payment data not loaded');
        return;
    }

    $(document).ready(function() {
        initQRCode();
        startPaymentStatusPolling();
    });

    /**
     * Initialize QR code display
     */
    function initQRCode() {
        // Use QRCode.js library for client-side generation
        if (typeof QRCode !== 'undefined') {
            new QRCode(document.getElementById('wpkj-wechat-qrcode'), {
                text: wpkj_wechat_native_data.qr_code,
                width: 280,
                height: 280,
                colorDark: '#000000',
                colorLight: '#ffffff',
                correctLevel: QRCode.CorrectLevel.H
            });
        } else {
            // QRCode library not loaded - show error
            console.error('QRCode.js library is not loaded');
            $('#wpkj-wechat-qrcode').html(
                '<div style="width: 280px; height: 280px; display: flex; align-items: center; justify-content: center; background: #f5f5f5; border: 1px solid #ddd;">' +
                '<p style="padding: 20px; text-align: center; color: #666;">Failed to load QR code. Please refresh the page.</p>' +
                '</div>'
            );
        }
    }

    /**
     * Start polling for payment status
     */
    function startPaymentStatusPolling() {
        var pollingInterval = 3000; // 3 seconds
        var maxAttempts = 200; // 10 minutes
        var currentAttempt = 0;
        var isRedirecting = false;

        /**
         * Check payment status via AJAX
         */
        function checkPaymentStatus() {
            if (isRedirecting) {
                return;
            }

            currentAttempt++;

            if (currentAttempt > maxAttempts) {
                clearInterval(pollingTimer);
                showTimeout();
                return;
            }

            $.ajax({
                url: wpkj_wechat_native_data.ajax_url,
                type: 'POST',
                dataType: 'json',
                data: {
                    action: 'wpkj_wechat_check_payment_status',
                    transaction_uuid: wpkj_wechat_native_data.transaction_uuid,
                    wpkj_wechat_pay_nonce: wpkj_wechat_native_data.wpkj_wechat_pay_nonce
                },
                success: function(response) {
                    handleStatusResponse(response);
                },
                error: function(xhr, status, error) {
                    console.error('Payment status check failed:', {
                        status: status,
                        error: error,
                        response: xhr.responseText
                    });
                    // Continue polling on error (network issues, etc.)
                }
            });
        }

        /**
         * Handle AJAX response
         */
        function handleStatusResponse(response) {
            if (isRedirecting) {
                return;
            }

            if (response.success && response.data) {
                var status = response.data.status;

                if (status === 'paid' || status === 'success') {
                    handlePaymentSuccess(response.data);
                } else if (status === 'failed' || status === 'cancelled') {
                    handlePaymentFailure(response.data);
                }
                // status === 'waiting' continues polling
            }
        }

        /**
         * Handle successful payment
         */
        function handlePaymentSuccess(data) {
            isRedirecting = true;
            clearInterval(pollingTimer);
            
            $('#payment-status').html(
                '<div class="status-text" style="color: #52c41a;">✓ ' + 
                wpkj_wechat_native_data.i18n.payment_success + 
                '</div>'
            );
            
            console.log('Payment successful, redirecting to:', data.redirect_url);
            
            setTimeout(function() {
                if (data.redirect_url) {
                    window.location.href = data.redirect_url;
                } else {
                    // Fallback: reload page which will trigger auto-redirect
                    window.location.reload();
                }
            }, 1000);
        }

        /**
         * Handle failed payment
         */
        function handlePaymentFailure(data) {
            clearInterval(pollingTimer);
            
            $('#payment-status').html(
                '<div class="status-text" style="color: #ff4d4f;">✗ ' + 
                (data.message || wpkj_wechat_native_data.i18n.payment_failed) + 
                '</div>'
            );
        }

        /**
         * Show timeout message
         */
        function showTimeout() {
            $('#payment-status').html(
                '<div class="status-text" style="color: #faad14;">⚠ ' + 
                wpkj_wechat_native_data.i18n.payment_timeout + 
                '</div>'
            );
        }

        // Start polling
        var pollingTimer = setInterval(checkPaymentStatus, pollingInterval);
        
        // Check immediately on load
        checkPaymentStatus();
    }

})(jQuery);
