/**
 * H5 Payment Waiting Page Handler
 * 
 * Polls payment status and redirects when payment is completed
 */
(function($) {
    'use strict';

    // Configuration
    const CONFIG = {
        POLL_INTERVAL: 2000,      // Poll every 2 seconds
        MAX_ATTEMPTS: 90,          // Maximum 90 attempts (3 minutes)
        REDIRECT_DELAY: 1000       // Delay before redirect (1 second)
    };

    let pollAttempts = 0;
    let pollTimer = null;

    /**
     * Initialize
     */
    $(document).ready(function() {
        console.log('[H5 Waiting] Initializing payment status polling');
        startPolling();
    });

    /**
     * Start polling payment status
     */
    function startPolling() {
        pollTimer = setInterval(checkPaymentStatus, CONFIG.POLL_INTERVAL);
        // Check immediately on start
        checkPaymentStatus();
    }

    /**
     * Stop polling
     */
    function stopPolling() {
        if (pollTimer) {
            clearInterval(pollTimer);
            pollTimer = null;
        }
    }

    /**
     * Check payment status via AJAX
     */
    function checkPaymentStatus() {
        pollAttempts++;

        console.log('[H5 Waiting] Checking payment status (attempt ' + pollAttempts + '/' + CONFIG.MAX_ATTEMPTS + ')');

        // Stop polling if max attempts reached
        if (pollAttempts >= CONFIG.MAX_ATTEMPTS) {
            stopPolling();
            showPaymentTimeout();
            return;
        }

        $.ajax({
            url: wpkj_wechat_h5_data.ajax_url,
            type: 'POST',
            data: {
                action: 'wpkj_wechat_check_payment_status',
                transaction_uuid: wpkj_wechat_h5_data.transaction_uuid,
                wpkj_wechat_pay_nonce: wpkj_wechat_h5_data.wpkj_wechat_pay_nonce
            },
            dataType: 'json',
            timeout: 10000,
            success: function(response) {
                console.log('[H5 Waiting] Status check response:', response);

                if (response.success && response.data) {
                    handleStatusResponse(response.data);
                } else {
                    console.warn('[H5 Waiting] Invalid response format');
                }
            },
            error: function(xhr, status, error) {
                console.error('[H5 Waiting] AJAX error:', status, error);
                // Continue polling on error
            }
        });
    }

    /**
     * Handle status check response
     */
    function handleStatusResponse(data) {
        const status = data.status;

        switch (status) {
            case 'paid':
                handlePaymentSuccess(data);
                break;
            case 'failed':
                handlePaymentFailed(data);
                break;
            case 'waiting':
                // Continue polling
                console.log('[H5 Waiting] Payment still pending, continuing to poll');
                break;
            default:
                console.warn('[H5 Waiting] Unknown status:', status);
        }
    }

    /**
     * Handle payment success
     */
    function handlePaymentSuccess(data) {
        console.log('[H5 Waiting] Payment successful!');
        stopPolling();

        // Update UI
        $('#payment-status')
            .removeClass('status-pending status-failed')
            .addClass('status-success')
            .html(
                '<span class="success-icon">✓</span>' +
                '<span class="status-text">' + wpkj_wechat_h5_data.i18n.payment_success + '</span>'
            );

        // Redirect to receipt page
        if (data.redirect_url) {
            setTimeout(function() {
                window.location.href = data.redirect_url;
            }, CONFIG.REDIRECT_DELAY);
        }
    }

    /**
     * Handle payment failed
     */
    function handlePaymentFailed(data) {
        console.log('[H5 Waiting] Payment failed');
        stopPolling();

        // Update UI
        $('#payment-status')
            .removeClass('status-pending status-success')
            .addClass('status-failed')
            .html(
                '<span class="error-icon">✗</span>' +
                '<span class="status-text">' + wpkj_wechat_h5_data.i18n.payment_failed + '</span>'
            );

        // Show retry button
        $('.instruction').html(
            '<a href="' + wpkj_wechat_h5_data.custom_payment_url + '" class="retry-button">' +
            wpkj_wechat_h5_data.i18n.retry_payment +
            '</a>'
        );
    }

    /**
     * Handle payment timeout
     */
    function showPaymentTimeout() {
        console.log('[H5 Waiting] Payment timeout');

        // Update UI
        $('#payment-status')
            .removeClass('status-pending status-success')
            .addClass('status-timeout')
            .html(
                '<span class="warning-icon">⚠</span>' +
                '<span class="status-text">' + wpkj_wechat_h5_data.i18n.payment_timeout + '</span>'
            );
    }

})(jQuery);
