<?php
/**
 * The core plugin class.
 *
 * @package WPG_LLMsTxt_Manager
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
	exit;
}

/**
 * Main plugin class.
 */
class WPG_LLMsTxt_Manager
{

	/**
	 * The loader that's responsible for maintaining and registering all hooks.
	 *
	 * @var object
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @var string
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @var string
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 */
	public function __construct()
	{
		$this->version = WPG_LLMSTXT_VERSION;
		$this->plugin_name = 'wpgeared-llms-txt-generator';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_generator_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 */
	private function load_dependencies()
	{
		require_once WPG_LLMSTXT_PATH . 'includes/class-cache.php';
		require_once WPG_LLMSTXT_PATH . 'includes/class-seo-integration.php';
		require_once WPG_LLMSTXT_PATH . 'includes/class-generator.php';
		require_once WPG_LLMSTXT_PATH . 'includes/class-crawler-detector.php';
		require_once WPG_LLMSTXT_PATH . 'admin/class-admin.php';
		require_once WPG_LLMSTXT_PATH . 'public/class-public.php';
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 * Note: WordPress.org automatically loads translations for hosted plugins.
	 */
	private function set_locale()
	{
		// WordPress.org automatically loads translations, so no action is needed here.
	}

	/**
	 * Register all of the hooks related to the admin area functionality.
	 */
	private function define_admin_hooks()
	{
		$plugin_admin = new WPG_LLMsTxt_Manager_Admin($this->get_plugin_name(), $this->get_version());

		add_action('admin_enqueue_scripts', array($plugin_admin, 'enqueue_styles'));
		add_action('admin_enqueue_scripts', array($plugin_admin, 'enqueue_scripts'));
	}

	/**
	 * Register all of the hooks related to the public-facing functionality.
	 */
	private function define_public_hooks()
	{
		$plugin_public = new WPG_LLMsTxt_Manager_Public($this->get_plugin_name(), $this->get_version());

		add_action('init', array($this, 'add_rewrite_rules'));
		add_action('template_redirect', array($this, 'serve_llms_txt_file'));
		add_action('template_redirect', array($plugin_public, 'detect_crawlers'));
	}

	/**
	 * Register all of the hooks related to the generator functionality.
	 */
	private function define_generator_hooks()
	{
		// Clear cache and regenerate on content updates
		add_action('save_post', array($this, 'clear_cache_on_update'));
		add_action('created_term', array($this, 'clear_cache_on_update'));
		add_action('edited_term', array($this, 'clear_cache_on_update'));
		add_action('delete_term', array($this, 'clear_cache_on_update'));
	}

	/**
	 * Add rewrite rules for llms.txt.
	 */
	public function add_rewrite_rules()
	{
		add_rewrite_rule('^llms\.txt$', 'index.php?wpg_llmstxt=1', 'top');
		add_rewrite_tag('%wpg_llmstxt%', '([^&]+)');
	}

	/**
	 * Serve the llms.txt file.
	 */
	public function serve_llms_txt_file()
	{
		$wpg_llmstxt = get_query_var('llms_txt');
		if ('1' === $wpg_llmstxt) {
			$generator = new WPG_LLMsTxt_Manager_Generator();
			$content = $generator->generate();

			header('Content-Type: text/plain; charset=utf-8');
			header('X-Robots-Tag: noindex');
			echo esc_html($content);
			exit;
		}
	}

	/**
	 * Clear cache on content update.
	 */
	public function clear_cache_on_update()
	{
		$cache = new WPG_LLMsTxt_Manager_Cache();
		$cache->clear();
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 */
	public function run()
	{
		// Flush rewrite rules on activation
		if (get_option('wpg_llmstxt_flush_rewrite_rules')) {
			flush_rewrite_rules();
			delete_option('wpg_llmstxt_flush_rewrite_rules');
		}
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of WordPress.
	 *
	 * @return string The name of the plugin.
	 */
	public function get_plugin_name()
	{
		return $this->plugin_name;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @return string The version number of the plugin.
	 */
	public function get_version()
	{
		return $this->version;
	}

	/**
	 * Activation function.
	 */
	public static function activate()
	{
		global $wpdb;

		// Create crawler logs table
		$table_name = $wpdb->prefix . 'wpg_llmstxt_crawler_logs';

		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE IF NOT EXISTS $table_name (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			crawler_name varchar(100) NOT NULL,
			user_agent text NOT NULL,
			ip_address varchar(45) NOT NULL,
			visit_date datetime NOT NULL,
			page_visited varchar(255) NOT NULL,
			PRIMARY KEY  (id),
			KEY crawler_name (crawler_name),
			KEY visit_date (visit_date)
		) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta($sql);

		// Set default options
		$defaults = array(
			'wpg_llmstxt_enabled' => 1,
			'wpg_llmstxt_post_types' => array('post', 'page'),
			'wpg_llmstxt_include_categories' => 1,
			'wpg_llmstxt_include_tags' => 1,
			'wpg_llmstxt_site_description' => '',
			'wpg_llmstxt_excluded_ids' => array(),
			'wpg_llmstxt_crawler_detection' => 1,
			'wpg_llmstxt_cache_duration' => 24,
		);

		foreach ($defaults as $key => $value) {
			if (false === get_option($key)) {
				add_option($key, $value);
			}
		}

		// Flag to flush rewrite rules
		add_option('wpg_llmstxt_flush_rewrite_rules', 1);
	}

	/**
	 * Deactivation function.
	 */
	public static function deactivate()
	{
		flush_rewrite_rules();
	}
}
