<?php
/**
 * LLMs.txt file generator class.
 *
 * @package WPG_LLMsTxt_Manager
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
	exit;
}

/**
 * Generator class.
 */
class WPG_LLMsTxt_Manager_Generator
{

	/**
	 * Cache instance.
	 *
	 * @var WPG_LLMsTxt_Manager_Cache
	 */
	private $cache;

	/**
	 * SEO Integration instance.
	 *
	 * @var WPG_LLMsTxt_Manager_SEO_Integration
	 */
	private $seo_integration;

	/**
	 * Constructor.
	 */
	public function __construct()
	{
		$this->cache = new WPG_LLMsTxt_Manager_Cache();
		$this->seo_integration = new WPG_LLMsTxt_Manager_SEO_Integration();
	}

	/**
	 * Generate LLMs.txt content.
	 *
	 * @return string Generated content.
	 */
	public function generate()
	{
		// Check if plugin is enabled
		if (!get_option('wpg_llmstxt_enabled', 1)) {
			return '';
		}

		// Check cache first
		if ($this->cache->exists()) {
			$cached = $this->cache->get();
			if (false !== $cached) {
				return $cached;
			}
		}

		$content = array();

		// Site Description
		$site_description = $this->get_site_description();
		if (!empty($site_description)) {
			$content[] = '# Site Description';
			$content[] = $site_description;
			$content[] = '';
		}

		// Main Navigation
		$navigation = $this->get_navigation();
		if (!empty($navigation)) {
			$content[] = '# Main Navigation';
			$content[] = $navigation;
			$content[] = '';
		}

		// Important Pages
		$pages = $this->get_pages();
		if (!empty($pages)) {
			$content[] = '# Important Pages';
			$content[] = $pages;
			$content[] = '';
		}

		// Categories
		if (get_option('wpg_llmstxt_include_categories', 1)) {
			$categories = $this->get_categories();
			if (!empty($categories)) {
				$content[] = '# Categories';
				$content[] = $categories;
				$content[] = '';
			}
		}

		// Tags
		if (get_option('wpg_llmstxt_include_tags', 1)) {
			$tags = $this->get_tags();
			if (!empty($tags)) {
				$content[] = '# Tags';
				$content[] = $tags;
				$content[] = '';
			}
		}

		// Recent Posts
		$posts = $this->get_posts();
		if (!empty($posts)) {
			$content[] = '# Recent Posts';
			$content[] = $posts;
			$content[] = '';
		}

		// Custom Post Types
		$custom_posts = $this->get_custom_post_types();
		if (!empty($custom_posts)) {
			$content[] = $custom_posts;
			$content[] = '';
		}

		// Sitemap
		$sitemap = $this->get_sitemap();
		if (!empty($sitemap)) {
			$content[] = '# Sitemap';
			$content[] = $sitemap;
		}

		$output = implode("\n", $content);

		// Cache the output
		$this->cache->set($output);

		return $output;
	}

	/**
	 * Get site description.
	 *
	 * @return string Site description.
	 */
	private function get_site_description()
	{
		$custom_description = get_option('wpg_llmstxt_site_description', '');
		if (!empty($custom_description)) {
			return sanitize_textarea_field($custom_description);
		}

		return get_bloginfo('description');
	}

	/**
	 * Get navigation links.
	 *
	 * @return string Navigation links.
	 */
	private function get_navigation()
	{
		$nav_items = array();
		$home_url = home_url('/');

		// Home
		$nav_items[] = '- Home: ' . esc_url($home_url);

		// Get menu items
		$locations = get_nav_menu_locations();
		if (!empty($locations)) {
			foreach ($locations as $location => $menu_id) {
				$menu = wp_get_nav_menu_object($menu_id);
				if ($menu) {
					$menu_items = wp_get_nav_menu_items($menu->term_id);
					if ($menu_items) {
						foreach ($menu_items as $item) {
							if ('custom' !== $item->type) {
								$nav_items[] = '- ' . esc_html($item->title) . ': ' . esc_url($item->url);
							}
						}
					}
				}
			}
		}

		return implode("\n", $nav_items);
	}

	/**
	 * Get pages.
	 *
	 * @return string Pages list.
	 */
	private function get_pages()
	{
		$post_types = get_option('wpg_llmstxt_post_types', array('post', 'page'));
		if (!in_array('page', $post_types, true)) {
			return '';
		}

		$excluded_ids = get_option('wpg_llmstxt_excluded_ids', array());
		if (!is_array($excluded_ids)) {
			$excluded_ids = array();
		}
		$excluded_ids = array_map('absint', $excluded_ids);

		// phpcs:disable WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- Required for user-defined post exclusions.
		$args = array(
			'post_type' => 'page',
			'post_status' => 'publish',
			'posts_per_page' => 50,
			'orderby' => 'menu_order',
			'order' => 'ASC',
			'post__not_in' => $excluded_ids,
		);
		// phpcs:enable WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in

		$query = new WP_Query($args);
		$items = array();

		if ($query->have_posts()) {
			while ($query->have_posts()) {
				$query->the_post();
				$post_id = get_the_ID();

				// Check SEO exclusion
				if ($this->seo_integration->should_exclude_post($post_id)) {
					continue;
				}

				$items[] = '- ' . esc_html(get_the_title()) . ': ' . esc_url(get_permalink());
			}
			wp_reset_postdata();
		}

		return implode("\n", $items);
	}

	/**
	 * Get categories.
	 *
	 * @return string Categories list.
	 */
	private function get_categories()
	{
		$categories = get_categories(
			array(
				'orderby' => 'name',
				'order' => 'ASC',
				'number' => 100,
			)
		);

		$items = array();
		foreach ($categories as $category) {
			$items[] = '- ' . esc_html($category->name) . ': ' . esc_url(get_category_link($category->term_id));
		}

		return implode("\n", $items);
	}

	/**
	 * Get tags.
	 *
	 * @return string Tags list.
	 */
	private function get_tags()
	{
		$tags = get_tags(
			array(
				'orderby' => 'count',
				'order' => 'DESC',
				'number' => 50,
			)
		);

		$items = array();
		foreach ($tags as $tag) {
			$items[] = '- ' . esc_html($tag->name) . ': ' . esc_url(get_tag_link($tag->term_id));
		}

		return implode("\n", $items);
	}

	/**
	 * Get posts.
	 *
	 * @return string Posts list.
	 */
	private function get_posts()
	{
		$post_types = get_option('wpg_llmstxt_post_types', array('post', 'page'));
		if (!in_array('post', $post_types, true)) {
			return '';
		}

		$excluded_ids = get_option('wpg_llmstxt_excluded_ids', array());
		if (!is_array($excluded_ids)) {
			$excluded_ids = array();
		}
		$excluded_ids = array_map('absint', $excluded_ids);

		// phpcs:disable WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- Required for user-defined post exclusions.
		$args = array(
			'post_type' => 'post',
			'post_status' => 'publish',
			'posts_per_page' => 50,
			'orderby' => 'date',
			'order' => 'DESC',
			'post__not_in' => $excluded_ids,
		);
		// phpcs:enable WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in

		$query = new WP_Query($args);
		$items = array();

		if ($query->have_posts()) {
			while ($query->have_posts()) {
				$query->the_post();
				$post_id = get_the_ID();

				// Check SEO exclusion
				if ($this->seo_integration->should_exclude_post($post_id)) {
					continue;
				}

				$items[] = '- ' . esc_html(get_the_title()) . ': ' . esc_url(get_permalink());
			}
			wp_reset_postdata();
		}

		return implode("\n", $items);
	}

	/**
	 * Get custom post types content.
	 *
	 * @return string Custom post types content.
	 */
	private function get_custom_post_types()
	{
		$post_types = get_option('wpg_llmstxt_post_types', array('post', 'page'));
		if (!is_array($post_types)) {
			$post_types = array();
		}

		// Filter out 'post' and 'page' as they're handled separately
		$custom_post_types = array_filter(
			$post_types,
			function ($type) {
				return !in_array($type, array('post', 'page'), true);
			}
		);

		if (empty($custom_post_types)) {
			return '';
		}

		$excluded_ids = get_option('wpg_llmstxt_excluded_ids', array());
		if (!is_array($excluded_ids)) {
			$excluded_ids = array();
		}
		$excluded_ids = array_map('absint', $excluded_ids);

		$all_items = array();

		foreach ($custom_post_types as $post_type) {
			$post_type_obj = get_post_type_object($post_type);
			if (!$post_type_obj) {
				continue;
			}

			// phpcs:disable WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- Required for user-defined post exclusions.
			$args = array(
				'post_type' => $post_type,
				'post_status' => 'publish',
				'posts_per_page' => 50,
				'orderby' => 'date',
				'order' => 'DESC',
				'post__not_in' => $excluded_ids,
			);
			// phpcs:enable WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in

			$query = new WP_Query($args);
			$items = array();

			if ($query->have_posts()) {
				while ($query->have_posts()) {
					$query->the_post();
					$post_id = get_the_ID();

					// Check SEO exclusion
					if ($this->seo_integration->should_exclude_post($post_id)) {
						continue;
					}

					$items[] = '- ' . esc_html(get_the_title()) . ': ' . esc_url(get_permalink());
				}
				wp_reset_postdata();

				if (!empty($items)) {
					$all_items[] = '# ' . esc_html($post_type_obj->label);
					$all_items[] = implode("\n", $items);
				}
			}
		}

		return implode("\n\n", $all_items);
	}

	/**
	 * Get sitemap link.
	 *
	 * @return string Sitemap link.
	 */
	private function get_sitemap()
	{
		$sitemap_urls = array();

		// Check for Yoast sitemap
		if (class_exists('WPSEO_Sitemaps')) {
			$sitemap_urls[] = '- XML Sitemap: ' . esc_url(home_url('/sitemap_index.xml'));
		}

		// Check for Rank Math sitemap
		if (class_exists('RankMath')) {
			$sitemap_urls[] = '- XML Sitemap: ' . esc_url(home_url('/sitemap.xml'));
		}

		// Check for AIOSEO sitemap
		if (class_exists('AIOSEO\Plugin\Plugin')) {
			$sitemap_urls[] = '- XML Sitemap: ' . esc_url(home_url('/sitemap.xml'));
		}

		// Default sitemap
		if (empty($sitemap_urls)) {
			$sitemap_urls[] = '- XML Sitemap: ' . esc_url(home_url('/sitemap.xml'));
		}

		return implode("\n", $sitemap_urls);
	}
}
