<?php
/**
 * Cache management class.
 *
 * @package WPG_LLMsTxt_Manager
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Cache class.
 */
class WPG_LLMsTxt_Manager_Cache {

	/**
	 * Cache key prefix.
	 *
	 * @var string
	 */
	private $cache_prefix = 'wpg_llmstxt_';

	/**
	 * Get cached content.
	 *
	 * @return string|false Cached content or false if not found.
	 */
	public function get() {
		$cache_key = $this->cache_prefix . 'content';
		return get_transient( $cache_key );
	}

	/**
	 * Set cached content.
	 *
	 * @param string $content Content to cache.
	 * @return bool True on success, false on failure.
	 */
	public function set( $content ) {
		$cache_key     = $this->cache_prefix . 'content';
		$cache_duration = get_option( 'wpg_llmstxt_cache_duration', 24 );
		$cache_duration = absint( $cache_duration ) * HOUR_IN_SECONDS;

		return set_transient( $cache_key, $content, $cache_duration );
	}

	/**
	 * Clear cache.
	 *
	 * @return bool True on success, false on failure.
	 */
	public function clear() {
		$cache_key = $this->cache_prefix . 'content';
		return delete_transient( $cache_key );
	}

	/**
	 * Check if cache exists.
	 *
	 * @return bool True if cache exists, false otherwise.
	 */
	public function exists() {
		$cache_key = $this->cache_prefix . 'content';
		return false !== get_transient( $cache_key );
	}

	/**
	 * Get cache statistics.
	 *
	 * @return array Cache statistics.
	 */
	public function get_stats() {
		$cache_key = $this->cache_prefix . 'content';
		$timeout   = get_option( '_transient_timeout_' . $cache_key );

		if ( false === $timeout ) {
			return array(
				'exists'  => false,
				'expires' => null,
			);
		}

		return array(
			'exists'  => true,
			'expires' => $timeout,
		);
	}
}

