<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package WPG_LLMsTxt_Manager
 */

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
	exit;
}

/**
 * Admin class.
 */
class WPG_LLMsTxt_Manager_Admin
{

	/**
	 * The ID of this plugin.
	 *
	 * @var string
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param string $plugin_name The name of this plugin.
	 * @param string $version      The version of this plugin.
	 */
	public function __construct($plugin_name, $version)
	{
		$this->plugin_name = $plugin_name;
		$this->version = $version;

		$this->init_settings();
	}

	/**
	 * Initialize settings.
	 */
	private function init_settings()
	{
		add_action('admin_menu', array($this, 'add_settings_page'));
		add_action('admin_init', array($this, 'register_settings'));
		add_action('admin_notices', array($this, 'display_notices'));
		add_filter('plugin_action_links_' . WPG_LLMSTXT_BASENAME, array($this, 'add_action_links'));
	}

	/**
	 * Register the stylesheets for the admin area.
	 */
	public function enqueue_styles()
	{
		$screen = get_current_screen();
		if ('settings_page_wpgeared-llms-txt-generator' === $screen->id) {
			wp_enqueue_style(
				$this->plugin_name,
				WPG_LLMSTXT_URL . 'admin/css/admin.css',
				array(),
				$this->version,
				'all'
			);
		}
	}

	/**
	 * Register the JavaScript for the admin area.
	 */
	public function enqueue_scripts()
	{
		$screen = get_current_screen();
		if ('settings_page_wpgeared-llms-txt-generator' === $screen->id) {
			wp_enqueue_script(
				$this->plugin_name,
				WPG_LLMSTXT_URL . 'admin/js/admin.js',
				array('jquery'),
				$this->version,
				false
			);
		}
	}

	/**
	 * Add settings page to admin menu.
	 */
	public function add_settings_page()
	{
		add_options_page(
			__('WPGeared LLMs.txt Generator', 'wpgeared-llms-txt-generator'),
			__('WPGeared LLMs.txt Generator', 'wpgeared-llms-txt-generator'),
			'manage_options',
			'wpgeared-llms-txt-generator',
			array($this, 'display_settings_page')
		);
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings()
	{
		// General Settings Section
		add_settings_section(
			'wpg_llmstxt_general',
			__('General Settings', 'wpgeared-llms-txt-generator'),
			array($this, 'general_section_callback'),
			'wpgeared-llms-txt-generator'
		);

		// Enable/Disable
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_enabled', array('sanitize_callback' => 'absint'));
		add_settings_field(
			'wpg_llmstxt_enabled',
			__('Enable LLMs.txt', 'wpgeared-llms-txt-generator'),
			array($this, 'enabled_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_general'
		);

		// Post Types
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_post_types', array($this, 'sanitize_post_types'));
		add_settings_field(
			'wpg_llmstxt_post_types',
			__('Post Types (Including Custom Post Types)', 'wpgeared-llms-txt-generator'),
			array($this, 'post_types_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_general'
		);

		// Include Categories
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_include_categories', array('sanitize_callback' => 'absint'));
		add_settings_field(
			'wpg_llmstxt_include_categories',
			__('Include Categories', 'wpgeared-llms-txt-generator'),
			array($this, 'include_categories_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_general'
		);

		// Include Tags
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_include_tags', array('sanitize_callback' => 'absint'));
		add_settings_field(
			'wpg_llmstxt_include_tags',
			__('Include Tags', 'wpgeared-llms-txt-generator'),
			array($this, 'include_tags_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_general'
		);

		// Site Description
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_site_description', array($this, 'sanitize_textarea'));
		add_settings_field(
			'wpg_llmstxt_site_description',
			__('Site Description', 'wpgeared-llms-txt-generator'),
			array($this, 'site_description_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_general'
		);

		// Excluded IDs
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_excluded_ids', array($this, 'sanitize_excluded_ids'));
		add_settings_field(
			'wpg_llmstxt_excluded_ids',
			__('Excluded Post/Page IDs', 'wpgeared-llms-txt-generator'),
			array($this, 'excluded_ids_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_general'
		);

		// Crawler Detection Section
		add_settings_section(
			'wpg_llmstxt_crawler',
			__('Crawler Detection', 'wpgeared-llms-txt-generator'),
			array($this, 'crawler_section_callback'),
			'wpgeared-llms-txt-generator'
		);

		// Enable Crawler Detection
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_crawler_detection', array('sanitize_callback' => 'absint'));
		add_settings_field(
			'wpg_llmstxt_crawler_detection',
			__('Enable Crawler Detection', 'wpgeared-llms-txt-generator'),
			array($this, 'crawler_detection_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_crawler'
		);

		// Cache Settings Section
		add_settings_section(
			'wpg_llmstxt_cache',
			__('Cache Settings', 'wpgeared-llms-txt-generator'),
			array($this, 'cache_section_callback'),
			'wpgeared-llms-txt-generator'
		);

		// Cache Duration
		register_setting('wpg_llmstxt_settings', 'wpg_llmstxt_cache_duration', array($this, 'sanitize_cache_duration'));
		add_settings_field(
			'wpg_llmstxt_cache_duration',
			__('Cache Duration (hours)', 'wpgeared-llms-txt-generator'),
			array($this, 'cache_duration_field_callback'),
			'wpgeared-llms-txt-generator',
			'wpg_llmstxt_cache'
		);
	}

	/**
	 * Sanitize post types.
	 *
	 * @param array $input Post types array.
	 * @return array Sanitized post types.
	 */
	public function sanitize_post_types($input)
	{
		if (!is_array($input)) {
			return array();
		}
		return array_map('sanitize_text_field', $input);
	}

	/**
	 * Sanitize textarea.
	 *
	 * @param string $input Textarea content.
	 * @return string Sanitized content.
	 */
	public function sanitize_textarea($input)
	{
		return sanitize_textarea_field($input);
	}

	/**
	 * Sanitize excluded IDs.
	 *
	 * @param string|array $input Comma-separated IDs or array.
	 * @return array Array of IDs.
	 */
	public function sanitize_excluded_ids($input)
	{
		if (empty($input)) {
			return array();
		}
		// If already an array, sanitize and return
		if (is_array($input)) {
			return array_map('absint', array_filter($input));
		}
		// If string, convert to array
		$ids = explode(',', $input);
		$ids = array_map('trim', $ids);
		$ids = array_map('absint', $ids);
		return array_filter($ids);
	}

	/**
	 * Sanitize cache duration.
	 *
	 * @param int $input Cache duration in hours.
	 * @return int Sanitized duration.
	 */
	public function sanitize_cache_duration($input)
	{
		return absint($input);
	}

	/**
	 * General section callback.
	 */
	public function general_section_callback()
	{
		echo '<p>' . esc_html__('Configure the general settings for LLMs.txt generation.', 'wpgeared-llms-txt-generator') . '</p>';
	}

	/**
	 * Enabled field callback.
	 */
	public function enabled_field_callback()
	{
		$value = get_option('wpg_llmstxt_enabled', 1);
		?>
		<label>
			<input type="checkbox" name="wpg_llmstxt_enabled" value="1" <?php checked($value, 1); ?>>
			<?php esc_html_e('Enable LLMs.txt file generation', 'wpgeared-llms-txt-generator'); ?>
		</label>
		<?php
	}

	/**
	 * Post types field callback.
	 */
	public function post_types_field_callback()
	{
		$selected = get_option('wpg_llmstxt_post_types', array('post', 'page'));
		$post_types = get_post_types(array('public' => true), 'objects');

		// Separate built-in and custom post types
		$built_in = array('post', 'page', 'attachment');
		$built_in_types = array();
		$custom_types = array();

		foreach ($post_types as $post_type) {
			if (in_array($post_type->name, $built_in, true)) {
				$built_in_types[$post_type->name] = $post_type;
			} else {
				$custom_types[$post_type->name] = $post_type;
			}
		}
		?>
		<fieldset>
			<?php if (!empty($built_in_types)): ?>
				<h4 style="margin: 15px 0 10px 0;"><?php esc_html_e('Built-in Post Types:', 'wpgeared-llms-txt-generator'); ?></h4>
				<?php foreach ($built_in_types as $post_type): ?>
					<label style="display: block; margin-bottom: 8px;">
						<input type="checkbox" name="wpg_llmstxt_post_types[]" value="<?php echo esc_attr($post_type->name); ?>"
							<?php checked(in_array($post_type->name, $selected, true)); ?>>
						<strong><?php echo esc_html($post_type->label); ?></strong>
						<code style="margin-left: 5px; color: #666;">(<?php echo esc_html($post_type->name); ?>)</code>
					</label>
				<?php endforeach; ?>
			<?php endif; ?>

			<?php if (!empty($custom_types)): ?>
				<h4 style="margin: 20px 0 10px 0;"><?php esc_html_e('Custom Post Types:', 'wpgeared-llms-txt-generator'); ?></h4>
				<?php foreach ($custom_types as $post_type): ?>
					<label style="display: block; margin-bottom: 8px;">
						<input type="checkbox" name="wpg_llmstxt_post_types[]" value="<?php echo esc_attr($post_type->name); ?>"
							<?php checked(in_array($post_type->name, $selected, true)); ?>>
						<strong><?php echo esc_html($post_type->label); ?></strong>
						<code style="margin-left: 5px; color: #666;">(<?php echo esc_html($post_type->name); ?>)</code>
					</label>
				<?php endforeach; ?>
			<?php else: ?>
				<p style="color: #666; font-style: italic; margin-top: 10px;">
					<?php esc_html_e('No custom post types found. Custom post types will appear here once they are registered.', 'wpgeared-llms-txt-generator'); ?>
				</p>
			<?php endif; ?>
		</fieldset>
		<p class="description">
			<?php esc_html_e('Select which post types (including custom post types) to include in the LLMs.txt file. Custom post types will be displayed in separate sections in the output.', 'wpgeared-llms-txt-generator'); ?>
		</p>
		<?php
	}

	/**
	 * Include categories field callback.
	 */
	public function include_categories_field_callback()
	{
		$value = get_option('wpg_llmstxt_include_categories', 1);
		?>
		<label>
			<input type="checkbox" name="wpg_llmstxt_include_categories" value="1" <?php checked($value, 1); ?>>
			<?php esc_html_e('Include categories in LLMs.txt', 'wpgeared-llms-txt-generator'); ?>
		</label>
		<?php
	}

	/**
	 * Include tags field callback.
	 */
	public function include_tags_field_callback()
	{
		$value = get_option('wpg_llmstxt_include_tags', 1);
		?>
		<label>
			<input type="checkbox" name="wpg_llmstxt_include_tags" value="1" <?php checked($value, 1); ?>>
			<?php esc_html_e('Include tags in LLMs.txt', 'wpgeared-llms-txt-generator'); ?>
		</label>
		<?php
	}

	/**
	 * Site description field callback.
	 */
	public function site_description_field_callback()
	{
		$value = get_option('wpg_llmstxt_site_description', '');
		?>
		<textarea name="wpg_llmstxt_site_description" rows="3"
			class="large-text"><?php echo esc_textarea($value); ?></textarea>
		<p class="description">
			<?php esc_html_e('Custom site description for LLMs.txt. Leave empty to use the site tagline.', 'wpgeared-llms-txt-generator'); ?>
		</p>
		<?php
	}

	/**
	 * Excluded IDs field callback.
	 */
	public function excluded_ids_field_callback()
	{
		$value = get_option('wpg_llmstxt_excluded_ids', array());
		if (is_array($value)) {
			$value = implode(', ', $value);
		}
		?>
		<input type="text" name="wpg_llmstxt_excluded_ids" value="<?php echo esc_attr($value); ?>" class="regular-text">
		<p class="description">
			<?php esc_html_e('Comma-separated list of post/page IDs to exclude from LLMs.txt.', 'wpgeared-llms-txt-generator'); ?>
		</p>
		<?php
	}

	/**
	 * Crawler section callback.
	 */
	public function crawler_section_callback()
	{
		echo '<p>' . esc_html__('Configure AI crawler detection and logging.', 'wpgeared-llms-txt-generator') . '</p>';
	}

	/**
	 * Crawler detection field callback.
	 */
	public function crawler_detection_field_callback()
	{
		$value = get_option('wpg_llmstxt_crawler_detection', 1);
		?>
		<label>
			<input type="checkbox" name="wpg_llmstxt_crawler_detection" value="1" <?php checked($value, 1); ?>>
			<?php esc_html_e('Enable AI crawler detection and logging', 'wpgeared-llms-txt-generator'); ?>
		</label>
		<?php
	}

	/**
	 * Cache section callback.
	 */
	public function cache_section_callback()
	{
		echo '<p>' . esc_html__('Configure caching settings for better performance.', 'wpgeared-llms-txt-generator') . '</p>';
	}

	/**
	 * Cache duration field callback.
	 */
	public function cache_duration_field_callback()
	{
		$value = get_option('wpg_llmstxt_cache_duration', 24);
		?>
		<input type="number" name="wpg_llmstxt_cache_duration" value="<?php echo esc_attr($value); ?>" min="1" max="168"
			class="small-text">
		<p class="description">
			<?php esc_html_e('Number of hours to cache the generated LLMs.txt file. (1-168 hours)', 'wpgeared-llms-txt-generator'); ?>
		</p>
		<?php
	}

	/**
	 * Display settings page.
	 */
	public function display_settings_page()
	{
		if (!current_user_can('manage_options')) {
			return;
		}

		// Handle form submission
		if (isset($_POST['wpg_llmstxt_action']) && isset($_POST['_wpnonce'])) {
			$action = sanitize_text_field(wp_unslash($_POST['wpg_llmstxt_action']));
			$nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';

			if ('clear_cache' === $action && wp_verify_nonce($nonce, 'wpg_llmstxt_clear_cache')) {
				$cache = new WPG_LLMsTxt_Manager_Cache();
				$cache->clear();
				add_settings_error('wpg_llmstxt_messages', 'wpg_llmstxt_message', __('Cache cleared successfully.', 'wpgeared-llms-txt-generator'), 'success');
			}

			if ('regenerate' === $action && wp_verify_nonce($nonce, 'wpg_llmstxt_regenerate')) {
				$cache = new WPG_LLMsTxt_Manager_Cache();
				$cache->clear();
				add_settings_error('wpg_llmstxt_messages', 'wpg_llmstxt_message', __('LLMs.txt file will be regenerated on next request.', 'wpgeared-llms-txt-generator'), 'success');
			}
		}

		// Handle settings save
		if (isset($_POST['submit']) && isset($_POST['_wpnonce'])) {
			$nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
			if (wp_verify_nonce($nonce, 'wpg_llmstxt-settings-options')) {
				// Settings are saved by Settings API
				add_settings_error('wpg_llmstxt_messages', 'wpg_llmstxt_message', __('Settings saved.', 'wpgeared-llms-txt-generator'), 'success');
			}
		}

		include WPG_LLMSTXT_PATH . 'admin/views/settings-page.php';
	}

	/**
	 * Add settings link to plugin action links.
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified links.
	 */
	public function add_action_links($links)
	{
		$settings_link = '<a href="options-general.php?page=wpgeared-llms-txt-generator">' . __('Settings', 'wpgeared-llms-txt-generator') . '</a>';
		array_unshift($links, $settings_link);
		return $links;
	}

	/**
	 * Display admin notices.
	 */
	public function display_notices()
	{
		settings_errors('wpg_llmstxt_messages');
	}
}

